/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.util;

import java.util.List;

import org.apache.portals.applications.webcontent2.proxy.impl.AbstractProxyCommand;
import org.apache.portals.applications.webcontent2.proxy.impl.ProxyProcessingChain;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Utility to invoke the life cycle methods easily on each {@link AbstractProxyCommand}
 * in a {@link ProxyProcessingChain}.
 */
public class ProxyCommandUtils
{

    private static Logger log = LoggerFactory.getLogger(ProxyCommandUtils.class);

    private ProxyCommandUtils()
    {
    }

    /**
     * Invokes {@link AbstractProxyCommand#initialize()} on each item in the <code>chain</code>.
     * @param chain
     */
    public static void initializeAllCommands(ProxyProcessingChain chain)
    {
        if (chain == null)
        {
            return;
        }

        List<AbstractProxyCommand> allProxyCommands = chain.getAllProxyCommands();

        for (AbstractProxyCommand proxyCommand : allProxyCommands)
        {
            proxyCommand.initialize();
        }
    }

    /**
     * Invokes {@link AbstractProxyCommand#destroy()} on each item in the <code>chain</code>.
     * @param chain
     */
    public static void destroyAllCommands(ProxyProcessingChain chain)
    {
        if (chain == null)
        {
            return;
        }

        List<AbstractProxyCommand> allProxyCommands = chain.getAllProxyCommands();

        for (AbstractProxyCommand proxyCommand : allProxyCommands)
        {
            try
            {
                proxyCommand.destroy();
            }
            catch (Exception e)
            {
                log.error("Failed to destroy proxy command, " + proxyCommand, e);
            }
        }
    }
}
