/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.services.portletregistry;

import org.apache.jetspeed.portletcontainer.om.applicationregistry.*;
import org.apache.jetspeed.portletcontainer.om.portletregistry.*;
import org.apache.turbine.services.TurbineServices;
import java.util.*;
import java.io.IOException;

import com.ibm.wps.util.ObjectID;

/**
 * This class is a static accessor for a <code>PortletRegistryService</code>
 * implementation.
 * 
 * @author <a href="shesmer@raleigh.ibm.com">Stephan Hesmer</a>
 */
public class PortletRegistryAccess
{
    /**
     * Method for getting a reference to the service singleton
     * 
     * @return the reference to this singleton
     */
    private static PortletRegistryService getService()
    {
        return (PortletRegistryService)TurbineServices
                .getInstance()
                .getService(PortletRegistryService.SERVICE_NAME);
    }
    
    /**
     * Returns all portlet applications definied in the registry
     * as an enumeration of <CODE>ApplicationEntry</CODE> elements.
     * 
     * @return an enumeration of <CODE>ApplicationEntry</CODE> elements
     * @throws IOException
     */
    public static Enumeration getApplications() throws IOException
    {
        return getService().getApplications();
    }

    /**
     * Checks whether the application with the given identifier is a concrete application or not.
     * 
     * @param aid the application identifier
     * @return true if the application identifier refers to a concrete application
     */
    public static boolean isConcreteApplication(ObjectID aid) throws IOException
    {
        return getService().isConcreteApplication(aid);
    }

    /**
     * Returns the portlet application with the given name.
     * 
     * @param aid   the identifier of the portlet application to be returned
     * @return the portlet application
     * @throws IOException
     */
    public static ApplicationEntry getApplication(ObjectID aid) throws IOException
    {
        return getService().getApplication(aid);
    }
   
    /**
     * Returns all portlet applications definied in the registry
     * as an enumeration of <CODE>ConcreteApplicationEntry</CODE> elements.
     * 
     * @return an enumeration of <CODE>ConcreteApplicationEntry</CODE> elements
     * @throws IOException
     */
    public static Enumeration getConcreteApplications() throws IOException
    {
        return getService().getConcreteApplications();
    }

    /**
     * Returns the concrete portlet application with the given name.
     * 
     * @param caid   the identifier of the concrete portlet application to be returned
     * @return the concrete portlet application
     * @throws IOException
     */
    public static ConcreteApplicationEntry getConcreteApplication(ObjectID caid) throws IOException
    {
        return getService().getConcreteApplication(caid);
    }

    /**
     * Returns all portlets definied in the registry
     * as an enumeration of <CODE>PortletEntry</CODE> objects.
     * 
     * @return an enumeration of <CODE>PortletEntry</CODE> objects
     * @throws IOException
     */
    public static Enumeration getPortlets() throws IOException
    {
        return getService().getPortlets();
    }

    /**
     * Returns all portlets of the given application definied in the registry
     * as an enumeration of <CODE>PortletEntry</CODE> elements.
     * 
     * @return an enumeration of <CODE>PortletEntry</CODE> elements
     * @throws IOException
     */
    public static Enumeration getPortlets(ApplicationEntry application) throws IOException
    {
        return getService().getPortlets(application);
    }

    /**
     * Checks whether the portlet with the given identifier is a concrete portlet or not.
     * 
     * @param pid the portlet identifier
     * @return true if the portlet identifier refers to a concrete portlet
     */
    public static boolean isConcretePortlet(ObjectID pid) throws IOException
    {
        return getService().isConcretePortlet(pid);
    }

    /**
     * Returns the portlet with the given portlet identifier.
     * 
     * @param pid   the portlet identifier
     * @return the portlet
     * @throws IOException
     */
    public static PortletEntry getPortlet(ObjectID pid) throws IOException
    {
        return getService().getPortlet(pid);
    }

    /**
     * Returns all portlets definied in the registry
     * as an enumeration of <CODE>ConcretePortletEntry</CODE> objects.
     * 
     * @return an enumeration of <CODE>ConcretePortletEntry</CODE> objects
     * @throws IOException
     */
    public static Enumeration getConcretePortlets() throws IOException
    {
        return getService().getConcretePortlets();
    }

    /**
     * Returns all portlets of the given concrete application definied in the registry
     * as an enumeration of <CODE>ConcretePortletEntry</CODE> elements.
     * 
     * @return an enumeration of <CODE>ConcretePortletEntry</CODE> elements
     * @throws IOException
     */
    public static Enumeration getConcretePortlets(ConcreteApplicationEntry application) throws IOException
    {
        return getService().getConcretePortlets(application);
    }

    /**
     * Returns the concrete portlet with the given portlet identifier.
     * 
     * @param cpid   the concrete portlet identifier
     * @return the concrete portlet
     * @throws IOException
     */
    public static ConcretePortletEntry getConcretePortlet(ObjectID cpid) throws IOException
    {
        return getService().getConcretePortlet(cpid);
    }

    /**
     * Checks if application settings are changed and sends the corresponding
     * events to all portlets of the given application except to the given
     * portlet.
     * 
     * @param concretePortlet
     *               the concrete portlet that changed the application settings
     * @param orgSettings
     *               Map containing the settings of the original application ([String]/[String])
     */
    public static void applicationSettingsChanged(ConcretePortletEntry concretePortlet,
                                                  Map orgSettings)
    {
        getService().applicationSettingsChanged(concretePortlet,
                                                orgSettings);
    }

}