/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.storage.internals.log;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import org.apache.kafka.common.InvalidRecordException;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.compress.Compression;
import org.apache.kafka.common.errors.ApiException;
import org.apache.kafka.common.errors.CorruptRecordException;
import org.apache.kafka.common.errors.InvalidTimestampException;
import org.apache.kafka.common.errors.UnsupportedCompressionTypeException;
import org.apache.kafka.common.errors.UnsupportedForMessageFormatException;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.record.AbstractRecords;
import org.apache.kafka.common.record.CompressionType;
import org.apache.kafka.common.record.MemoryRecords;
import org.apache.kafka.common.record.MemoryRecordsBuilder;
import org.apache.kafka.common.record.MutableRecordBatch;
import org.apache.kafka.common.record.Record;
import org.apache.kafka.common.record.RecordBatch;
import org.apache.kafka.common.record.RecordValidationStats;
import org.apache.kafka.common.record.TimestampType;
import org.apache.kafka.common.requests.ProduceResponse;
import org.apache.kafka.common.utils.BufferSupplier;
import org.apache.kafka.common.utils.CloseableIterator;
import org.apache.kafka.common.utils.PrimitiveRef;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.server.common.MetadataVersion;
import org.apache.kafka.storage.internals.log.AppendOrigin;
import org.apache.kafka.storage.internals.log.RecordValidationException;

public class LogValidator {
    private final MemoryRecords records;
    private final TopicPartition topicPartition;
    private final Time time;
    private final CompressionType sourceCompressionType;
    private final Compression targetCompression;
    private final boolean compactedTopic;
    private final byte toMagic;
    private final TimestampType timestampType;
    private final long timestampBeforeMaxMs;
    private final long timestampAfterMaxMs;
    private final int partitionLeaderEpoch;
    private final AppendOrigin origin;
    private final MetadataVersion interBrokerProtocolVersion;

    public LogValidator(MemoryRecords records, TopicPartition topicPartition, Time time, CompressionType sourceCompressionType, Compression targetCompression, boolean compactedTopic, byte toMagic, TimestampType timestampType, long timestampBeforeMaxMs, long timestampAfterMaxMs, int partitionLeaderEpoch, AppendOrigin origin, MetadataVersion interBrokerProtocolVersion) {
        this.records = records;
        this.topicPartition = topicPartition;
        this.time = time;
        this.sourceCompressionType = sourceCompressionType;
        this.targetCompression = targetCompression;
        this.compactedTopic = compactedTopic;
        this.toMagic = toMagic;
        this.timestampType = timestampType;
        this.timestampBeforeMaxMs = timestampBeforeMaxMs;
        this.timestampAfterMaxMs = timestampAfterMaxMs;
        this.partitionLeaderEpoch = partitionLeaderEpoch;
        this.origin = origin;
        this.interBrokerProtocolVersion = interBrokerProtocolVersion;
    }

    public ValidationResult validateMessagesAndAssignOffsets(PrimitiveRef.LongRef offsetCounter, MetricsRecorder metricsRecorder, BufferSupplier bufferSupplier) {
        if (this.sourceCompressionType == CompressionType.NONE && this.targetCompression.type() == CompressionType.NONE) {
            if (!this.records.hasMatchingMagic(this.toMagic)) {
                return this.convertAndAssignOffsetsNonCompressed(offsetCounter, metricsRecorder);
            }
            return this.assignOffsetsNonCompressed(offsetCounter, metricsRecorder);
        }
        return this.validateMessagesAndAssignOffsetsCompressed(offsetCounter, metricsRecorder, bufferSupplier);
    }

    private static MutableRecordBatch getFirstBatchAndMaybeValidateNoMoreBatches(MemoryRecords records, CompressionType sourceCompression) {
        Iterator batchIterator = records.batches().iterator();
        if (!batchIterator.hasNext()) {
            throw new InvalidRecordException("Record batch has no batches at all");
        }
        MutableRecordBatch batch = (MutableRecordBatch)batchIterator.next();
        if ((batch.magic() >= 2 || sourceCompression != CompressionType.NONE) && batchIterator.hasNext()) {
            throw new InvalidRecordException("Compressed outer record has more than one batch");
        }
        return batch;
    }

    private ValidationResult convertAndAssignOffsetsNonCompressed(PrimitiveRef.LongRef offsetCounter, MetricsRecorder metricsRecorder) {
        long now = this.time.milliseconds();
        long startNanos = this.time.nanoseconds();
        int sizeInBytesAfterConversion = AbstractRecords.estimateSizeInBytes((byte)this.toMagic, (long)offsetCounter.value, (CompressionType)CompressionType.NONE, (Iterable)this.records.records());
        MutableRecordBatch firstBatch = LogValidator.getFirstBatchAndMaybeValidateNoMoreBatches(this.records, CompressionType.NONE);
        long producerId = firstBatch.producerId();
        short producerEpoch = firstBatch.producerEpoch();
        int sequence = firstBatch.baseSequence();
        boolean isTransactional = firstBatch.isTransactional();
        ByteBuffer newBuffer = ByteBuffer.allocate(sizeInBytesAfterConversion);
        MemoryRecordsBuilder builder = MemoryRecords.builder((ByteBuffer)newBuffer, (byte)this.toMagic, (Compression)Compression.NONE, (TimestampType)this.timestampType, (long)offsetCounter.value, (long)now, (long)producerId, (short)producerEpoch, (int)sequence, (boolean)isTransactional, (int)this.partitionLeaderEpoch);
        for (RecordBatch batch : this.records.batches()) {
            LogValidator.validateBatch(this.topicPartition, (RecordBatch)firstBatch, batch, this.origin, this.toMagic, metricsRecorder);
            ArrayList<ApiRecordError> recordErrors = new ArrayList<ApiRecordError>(0);
            int recordIndex = 0;
            for (Record record : batch) {
                Optional<ApiRecordError> recordError = LogValidator.validateRecord(batch, this.topicPartition, record, recordIndex, now, this.timestampType, this.timestampBeforeMaxMs, this.timestampAfterMaxMs, this.compactedTopic, metricsRecorder);
                recordError.ifPresent(e -> recordErrors.add((ApiRecordError)e));
                if (recordErrors.isEmpty()) {
                    builder.appendWithOffset(offsetCounter.value++, record);
                }
                ++recordIndex;
            }
            LogValidator.processRecordErrors(recordErrors);
        }
        MemoryRecords convertedRecords = builder.build();
        MemoryRecordsBuilder.RecordsInfo info = builder.info();
        RecordValidationStats recordValidationStats = new RecordValidationStats((long)builder.uncompressedBytesWritten(), builder.numRecords(), this.time.nanoseconds() - startNanos);
        return new ValidationResult(now, convertedRecords, info.maxTimestamp, info.shallowOffsetOfMaxTimestamp, true, recordValidationStats);
    }

    public ValidationResult assignOffsetsNonCompressed(PrimitiveRef.LongRef offsetCounter, MetricsRecorder metricsRecorder) {
        long now = this.time.milliseconds();
        long maxTimestamp = -1L;
        long shallowOffsetOfMaxTimestamp = -1L;
        long initialOffset = offsetCounter.value;
        MutableRecordBatch firstBatch = LogValidator.getFirstBatchAndMaybeValidateNoMoreBatches(this.records, CompressionType.NONE);
        for (MutableRecordBatch batch : this.records.batches()) {
            LogValidator.validateBatch(this.topicPartition, (RecordBatch)firstBatch, (RecordBatch)batch, this.origin, this.toMagic, metricsRecorder);
            long maxBatchTimestamp = -1L;
            ArrayList<ApiRecordError> recordErrors = new ArrayList<ApiRecordError>(0);
            int recordIndex = 0;
            for (Record record : batch) {
                Optional<ApiRecordError> recordError = LogValidator.validateRecord((RecordBatch)batch, this.topicPartition, record, recordIndex, now, this.timestampType, this.timestampBeforeMaxMs, this.timestampAfterMaxMs, this.compactedTopic, metricsRecorder);
                recordError.ifPresent(recordErrors::add);
                ++offsetCounter.value;
                if (batch.magic() > 0 && record.timestamp() > maxBatchTimestamp) {
                    maxBatchTimestamp = record.timestamp();
                }
                ++recordIndex;
            }
            LogValidator.processRecordErrors(recordErrors);
            if (batch.magic() > 0 && maxBatchTimestamp > maxTimestamp) {
                maxTimestamp = maxBatchTimestamp;
                shallowOffsetOfMaxTimestamp = offsetCounter.value - 1L;
            }
            batch.setLastOffset(offsetCounter.value - 1L);
            if (batch.magic() >= 2) {
                batch.setPartitionLeaderEpoch(this.partitionLeaderEpoch);
            }
            if (batch.magic() <= 0) continue;
            if (this.timestampType == TimestampType.LOG_APPEND_TIME) {
                batch.setMaxTimestamp(TimestampType.LOG_APPEND_TIME, now);
                continue;
            }
            batch.setMaxTimestamp(this.timestampType, maxBatchTimestamp);
        }
        if (this.timestampType == TimestampType.LOG_APPEND_TIME) {
            maxTimestamp = now;
            switch (this.toMagic) {
                case 0: {
                    maxTimestamp = -1L;
                    shallowOffsetOfMaxTimestamp = -1L;
                    break;
                }
                case 1: {
                    shallowOffsetOfMaxTimestamp = initialOffset;
                    break;
                }
                default: {
                    shallowOffsetOfMaxTimestamp = offsetCounter.value - 1L;
                }
            }
        }
        return new ValidationResult(now, this.records, maxTimestamp, shallowOffsetOfMaxTimestamp, false, RecordValidationStats.EMPTY);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ValidationResult validateMessagesAndAssignOffsetsCompressed(PrimitiveRef.LongRef offsetCounter, MetricsRecorder metricsRecorder, BufferSupplier bufferSupplier) {
        if (this.targetCompression.type() == CompressionType.ZSTD && this.interBrokerProtocolVersion.isLessThan(MetadataVersion.IBP_2_1_IV0)) {
            throw new UnsupportedCompressionTypeException("Produce requests to inter.broker.protocol.version < 2.1 broker are not allowed to use ZStandard compression");
        }
        boolean inPlaceAssignment = this.sourceCompressionType == this.targetCompression.type();
        long now = this.time.milliseconds();
        long maxTimestamp = -1L;
        PrimitiveRef.LongRef expectedInnerOffset = PrimitiveRef.ofLong((long)0L);
        ArrayList<Record> validatedRecords = new ArrayList<Record>();
        int uncompressedSizeInBytes = 0;
        MutableRecordBatch firstBatch = LogValidator.getFirstBatchAndMaybeValidateNoMoreBatches(this.records, this.sourceCompressionType);
        if (firstBatch.magic() != this.toMagic || this.toMagic == 0) {
            inPlaceAssignment = false;
        }
        if (this.sourceCompressionType == CompressionType.NONE && firstBatch.isControlBatch()) {
            inPlaceAssignment = true;
        }
        for (MutableRecordBatch batch : this.records.batches()) {
            LogValidator.validateBatch(this.topicPartition, (RecordBatch)firstBatch, (RecordBatch)batch, this.origin, this.toMagic, metricsRecorder);
            uncompressedSizeInBytes += AbstractRecords.recordBatchHeaderSizeInBytes((byte)this.toMagic, (CompressionType)batch.compressionType());
            try (CloseableIterator recordsIterator = inPlaceAssignment && firstBatch.magic() >= 2 ? batch.skipKeyValueIterator(bufferSupplier) : batch.streamingIterator(bufferSupplier);){
                ArrayList<ApiRecordError> recordErrors = new ArrayList<ApiRecordError>(0);
                int recordIndex = 0;
                while (recordsIterator.hasNext()) {
                    Record record = (Record)recordsIterator.next();
                    ++expectedInnerOffset.value;
                    Optional<ApiRecordError> recordError = LogValidator.validateRecordCompression(this.sourceCompressionType, recordIndex, record);
                    if (!recordError.isPresent()) {
                        recordError = LogValidator.validateRecord((RecordBatch)batch, this.topicPartition, record, recordIndex, now, this.timestampType, this.timestampBeforeMaxMs, this.timestampAfterMaxMs, this.compactedTopic, metricsRecorder);
                    }
                    if (!recordError.isPresent() && batch.magic() > 0 && this.toMagic > 0) {
                        long expectedOffset;
                        if (record.timestamp() > maxTimestamp) {
                            maxTimestamp = record.timestamp();
                        }
                        if (record.offset() != expectedOffset) {
                            inPlaceAssignment = false;
                        }
                    }
                    if (recordError.isPresent()) {
                        recordErrors.add(recordError.get());
                    } else {
                        uncompressedSizeInBytes += record.sizeInBytes();
                        validatedRecords.add(record);
                    }
                    ++recordIndex;
                }
                LogValidator.processRecordErrors(recordErrors);
            }
        }
        if (!inPlaceAssignment) {
            return this.buildRecordsAndAssignOffsets(offsetCounter, now, (RecordBatch)firstBatch, validatedRecords, uncompressedSizeInBytes);
        }
        offsetCounter.value += (long)validatedRecords.size();
        long lastOffset = offsetCounter.value - 1L;
        firstBatch.setLastOffset(lastOffset);
        if (this.timestampType == TimestampType.LOG_APPEND_TIME) {
            maxTimestamp = now;
        }
        if (this.toMagic >= 1) {
            firstBatch.setMaxTimestamp(this.timestampType, maxTimestamp);
        }
        if (this.toMagic >= 2) {
            firstBatch.setPartitionLeaderEpoch(this.partitionLeaderEpoch);
        }
        RecordValidationStats recordValidationStats = new RecordValidationStats((long)uncompressedSizeInBytes, 0, 0L);
        return new ValidationResult(now, this.records, maxTimestamp, lastOffset, false, recordValidationStats);
    }

    private ValidationResult buildRecordsAndAssignOffsets(PrimitiveRef.LongRef offsetCounter, long logAppendTime, RecordBatch firstBatch, List<Record> validatedRecords, int uncompressedSizeInBytes) {
        long startNanos = this.time.nanoseconds();
        int estimatedSize = AbstractRecords.estimateSizeInBytes((byte)this.toMagic, (long)offsetCounter.value, (CompressionType)this.targetCompression.type(), validatedRecords);
        ByteBuffer buffer = ByteBuffer.allocate(estimatedSize);
        MemoryRecordsBuilder builder = MemoryRecords.builder((ByteBuffer)buffer, (byte)this.toMagic, (Compression)this.targetCompression, (TimestampType)this.timestampType, (long)offsetCounter.value, (long)logAppendTime, (long)firstBatch.producerId(), (short)firstBatch.producerEpoch(), (int)firstBatch.baseSequence(), (boolean)firstBatch.isTransactional(), (int)this.partitionLeaderEpoch);
        for (Record record : validatedRecords) {
            builder.appendWithOffset(offsetCounter.value++, record);
        }
        MemoryRecords records = builder.build();
        MemoryRecordsBuilder.RecordsInfo info = builder.info();
        int conversionCount = builder.numRecords();
        RecordValidationStats recordValidationStats = new RecordValidationStats((long)(uncompressedSizeInBytes + builder.uncompressedBytesWritten()), conversionCount, this.time.nanoseconds() - startNanos);
        return new ValidationResult(logAppendTime, records, info.maxTimestamp, info.shallowOffsetOfMaxTimestamp, true, recordValidationStats);
    }

    private static void validateBatch(TopicPartition topicPartition, RecordBatch firstBatch, RecordBatch batch, AppendOrigin origin, byte toMagic, MetricsRecorder metricsRecorder) {
        if (firstBatch.magic() != batch.magic()) {
            metricsRecorder.recordInvalidMagic();
            throw new InvalidRecordException("Batch magic " + batch.magic() + " is not the same as the first batch's magic byte " + firstBatch.magic() + " in topic partition " + topicPartition);
        }
        if (origin == AppendOrigin.CLIENT) {
            if (batch.magic() >= 2) {
                long countFromOffsets = batch.lastOffset() - batch.baseOffset() + 1L;
                if (countFromOffsets <= 0L) {
                    metricsRecorder.recordInvalidOffset();
                    throw new InvalidRecordException("Batch has an invalid offset range: [" + batch.baseOffset() + ", " + batch.lastOffset() + "] in topic partition " + topicPartition);
                }
                long count = batch.countOrNull().intValue();
                if (count <= 0L) {
                    metricsRecorder.recordInvalidOffset();
                    throw new InvalidRecordException("Invalid reported count for record batch: " + count + " in topic partition " + topicPartition);
                }
                if (countFromOffsets != count) {
                    metricsRecorder.recordInvalidOffset();
                    throw new InvalidRecordException("Inconsistent batch offset range [" + batch.baseOffset() + ", " + batch.lastOffset() + "] and count of records " + count + " in topic partition " + topicPartition);
                }
            }
            if (batch.isControlBatch()) {
                metricsRecorder.recordInvalidOffset();
                throw new InvalidRecordException("Clients are not allowed to write control records in topic partition " + topicPartition);
            }
            if (batch.hasProducerId() && batch.baseSequence() < 0) {
                metricsRecorder.recordInvalidSequence();
                throw new InvalidRecordException("Invalid sequence number " + batch.baseSequence() + " in record batch with producerId " + batch.producerId() + " in topic partition " + topicPartition);
            }
        }
        if (batch.isTransactional() && toMagic < 2) {
            throw new UnsupportedForMessageFormatException("Transactional records cannot be used with magic version " + toMagic);
        }
        if (batch.hasProducerId() && toMagic < 2) {
            throw new UnsupportedForMessageFormatException("Idempotent records cannot be used with magic version " + toMagic);
        }
    }

    private static Optional<ApiRecordError> validateRecord(RecordBatch batch, TopicPartition topicPartition, Record record, int recordIndex, long now, TimestampType timestampType, long timestampBeforeMaxMs, long timestampAfterMaxMs, boolean compactedTopic, MetricsRecorder metricsRecorder) {
        Optional<ApiRecordError> keyError;
        if (!record.hasMagic(batch.magic())) {
            metricsRecorder.recordInvalidMagic();
            return Optional.of(new ApiRecordError(Errors.INVALID_RECORD, new ProduceResponse.RecordError(recordIndex, "Record " + record + "'s magic does not match outer magic " + batch.magic() + " in topic partition " + topicPartition)));
        }
        if (batch.magic() <= 1 && batch.isCompressed()) {
            try {
                record.ensureValid();
            }
            catch (InvalidRecordException e) {
                metricsRecorder.recordInvalidChecksums();
                throw new CorruptRecordException(e.getMessage() + " in topic partition " + topicPartition);
            }
        }
        if ((keyError = LogValidator.validateKey(record, recordIndex, topicPartition, compactedTopic, metricsRecorder)).isPresent()) {
            return keyError;
        }
        return LogValidator.validateTimestamp(batch, record, recordIndex, now, timestampType, timestampBeforeMaxMs, timestampAfterMaxMs);
    }

    private static Optional<ApiRecordError> validateKey(Record record, int recordIndex, TopicPartition topicPartition, boolean compactedTopic, MetricsRecorder metricsRecorder) {
        if (compactedTopic && !record.hasKey()) {
            metricsRecorder.recordNoKeyCompactedTopic();
            return Optional.of(new ApiRecordError(Errors.INVALID_RECORD, new ProduceResponse.RecordError(recordIndex, "Compacted topic cannot accept message without key in topic partition " + topicPartition)));
        }
        return Optional.empty();
    }

    private static Optional<ApiRecordError> validateTimestamp(RecordBatch batch, Record record, int recordIndex, long now, TimestampType timestampType, long timestampBeforeMaxMs, long timestampAfterMaxMs) {
        if (timestampType == TimestampType.CREATE_TIME && record.timestamp() != -1L) {
            if (LogValidator.recordHasInvalidTimestamp(record, now, timestampBeforeMaxMs, timestampAfterMaxMs)) {
                return Optional.of(new ApiRecordError(Errors.INVALID_TIMESTAMP, new ProduceResponse.RecordError(recordIndex, "Timestamp " + record.timestamp() + " of message with offset " + record.offset() + " is out of range. The timestamp should be within [" + (now - timestampBeforeMaxMs) + ", " + (now + timestampAfterMaxMs) + "]")));
            }
        } else if (batch.timestampType() == TimestampType.LOG_APPEND_TIME) {
            return Optional.of(new ApiRecordError(Errors.INVALID_TIMESTAMP, new ProduceResponse.RecordError(recordIndex, "Invalid timestamp type in message " + record + ". Producer should not set timestamp type to LogAppendTime.")));
        }
        return Optional.empty();
    }

    private static boolean recordHasInvalidTimestamp(Record record, long now, long timestampBeforeMaxMs, long timestampAfterMaxMs) {
        long timestampDiff = now - record.timestamp();
        return timestampDiff > timestampBeforeMaxMs || -1L * timestampDiff > timestampAfterMaxMs;
    }

    private static Optional<ApiRecordError> validateRecordCompression(CompressionType sourceCompression, int recordIndex, Record record) {
        if (sourceCompression != CompressionType.NONE && record.isCompressed()) {
            return Optional.of(new ApiRecordError(Errors.INVALID_RECORD, new ProduceResponse.RecordError(recordIndex, "Compressed outer record should not have an inner record with a compression attribute set: " + record)));
        }
        return Optional.empty();
    }

    private static void processRecordErrors(List<ApiRecordError> recordErrors) {
        if (!recordErrors.isEmpty()) {
            List<ProduceResponse.RecordError> errors = recordErrors.stream().map(e -> e.recordError).collect(Collectors.toList());
            if (recordErrors.stream().anyMatch(e -> e.apiError == Errors.INVALID_TIMESTAMP)) {
                throw new RecordValidationException((ApiException)new InvalidTimestampException("One or more records have been rejected due to invalid timestamp"), errors);
            }
            throw new RecordValidationException((ApiException)new InvalidRecordException("One or more records have been rejected due to " + errors.size() + " record errors in total, and only showing the first three errors at most: " + errors.subList(0, Math.min(errors.size(), 3))), errors);
        }
    }

    private static class ApiRecordError {
        final Errors apiError;
        final ProduceResponse.RecordError recordError;

        private ApiRecordError(Errors apiError, ProduceResponse.RecordError recordError) {
            this.apiError = apiError;
            this.recordError = recordError;
        }
    }

    public static class ValidationResult {
        public final long logAppendTimeMs;
        public final MemoryRecords validatedRecords;
        public final long maxTimestampMs;
        public final long shallowOffsetOfMaxTimestamp;
        public final boolean messageSizeMaybeChanged;
        public final RecordValidationStats recordValidationStats;

        public ValidationResult(long logAppendTimeMs, MemoryRecords validatedRecords, long maxTimestampMs, long shallowOffsetOfMaxTimestamp, boolean messageSizeMaybeChanged, RecordValidationStats recordValidationStats) {
            this.logAppendTimeMs = logAppendTimeMs;
            this.validatedRecords = validatedRecords;
            this.maxTimestampMs = maxTimestampMs;
            this.shallowOffsetOfMaxTimestamp = shallowOffsetOfMaxTimestamp;
            this.messageSizeMaybeChanged = messageSizeMaybeChanged;
            this.recordValidationStats = recordValidationStats;
        }
    }

    public static interface MetricsRecorder {
        public void recordInvalidMagic();

        public void recordInvalidOffset();

        public void recordInvalidSequence();

        public void recordInvalidChecksums();

        public void recordNoKeyCompactedTopic();
    }
}

