/*
 * Decompiled with CFR 0.152.
 */
package io.churchkey.ssh;

import io.churchkey.Key;
import io.churchkey.dsa.Dsa;
import io.churchkey.ec.Curve;
import io.churchkey.ec.EcPoints;
import io.churchkey.ec.Ecdsa;
import io.churchkey.rsa.Rsa;
import io.churchkey.ssh.KeyInput;
import io.churchkey.ssh.KeyOutput;
import io.churchkey.ssh.OpenSSHPublicKey;
import io.churchkey.util.Pem;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.math.BigInteger;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.interfaces.DSAPrivateKey;
import java.security.interfaces.DSAPublicKey;
import java.security.interfaces.ECPrivateKey;
import java.security.interfaces.ECPublicKey;
import java.security.interfaces.RSAPrivateCrtKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.ECPoint;
import java.security.spec.InvalidKeySpecException;
import java.util.HashMap;

public class OpenSSHPrivateKey {
    private OpenSSHPrivateKey() {
    }

    public static Key decode(byte[] bytes) {
        Pem pem = Pem.parse(bytes);
        try {
            KeyInput keyInput = new KeyInput(pem.getData());
            OpenSSHPrivateKey.assertString("Auth Magic", "openssh-key-v1", keyInput.readAuthMagic());
            OpenSSHPrivateKey.assertString("ciphername", "none", keyInput.readString());
            OpenSSHPrivateKey.assertString("kdfname", "none", keyInput.readString());
            OpenSSHPrivateKey.assertString("kdf", "", keyInput.readString());
            OpenSSHPrivateKey.assertInt("number of keys", 1, keyInput.readInt());
            byte[] sshpublic = keyInput.readBytes();
            keyInput.readInt();
            keyInput.readInt();
            keyInput.readInt();
            String keyType = keyInput.readString();
            if ("ssh-rsa".equals(keyType)) {
                return OpenSSHPrivateKey.readRsaPrivateKey(keyInput);
            }
            if ("ssh-dss".equals(keyType)) {
                return OpenSSHPrivateKey.readPrivateDssKey(keyInput);
            }
            if ("ecdsa-sha2-nistp256".equals(keyType)) {
                return OpenSSHPrivateKey.readEcdsaPrivateKey(Curve.nistp256, keyInput);
            }
            if ("ecdsa-sha2-nistp384".equals(keyType)) {
                return OpenSSHPrivateKey.readEcdsaPrivateKey(Curve.nistp384, keyInput);
            }
            if ("ecdsa-sha2-nistp521".equals(keyType)) {
                return OpenSSHPrivateKey.readEcdsaPrivateKey(Curve.nistp521, keyInput);
            }
            throw new UnsupportedOperationException("Unsupported key type: " + keyType);
        }
        catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException(e);
        }
        catch (IOException | InvalidKeySpecException e) {
            throw new RuntimeException(e);
        }
    }

    public static byte[] encode(Key key) {
        byte[] result;
        try {
            KeyOutput out = new KeyOutput();
            out.writeAuthMagic("openssh-key-v1");
            out.writeString("none");
            out.writeString("none");
            out.writeString("");
            out.writeInt(1);
            out.writeBytes(OpenSSHPrivateKey.encodePublicKey(key));
            out.writeBytes(OpenSSHPrivateKey.pad(OpenSSHPrivateKey.encodePrivateKey(key)));
            result = out.toByteArray();
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        byte[] bytes = result;
        return Pem.builder().type("OPENSSH PRIVATE KEY").wrap(70).data(bytes).format().getBytes();
    }

    private static byte[] pad(byte[] bytes) {
        int i = bytes.length % 8;
        if (i == 0) {
            return bytes;
        }
        int needed = 8 - i;
        byte[] padding = new byte[]{1, 2, 3, 4, 5, 6, 7};
        byte[] padded = new byte[bytes.length + needed];
        System.arraycopy(bytes, 0, padded, 0, bytes.length);
        System.arraycopy(padding, 0, padded, bytes.length, needed);
        return padded;
    }

    private static byte[] encodePublicKey(Key key) throws IOException {
        if (key.getPublicKey() == null) {
            return new byte[0];
        }
        java.security.Key publicKey = key.getPublicKey().getKey();
        if (publicKey instanceof RSAPublicKey) {
            return OpenSSHPublicKey.RsaPublic.write((RSAPublicKey)publicKey);
        }
        if (publicKey instanceof DSAPublicKey) {
            return OpenSSHPublicKey.DsaPublic.write((DSAPublicKey)publicKey);
        }
        if (publicKey instanceof ECPublicKey) {
            ECPublicKey ecPublicKey = (ECPublicKey)publicKey;
            String curveName = OpenSSHPublicKey.EcPublic.curveName(ecPublicKey.getParams());
            return OpenSSHPublicKey.EcPublic.write(ecPublicKey, curveName);
        }
        throw new UnsupportedOperationException("Unsupported key type: " + publicKey.getClass().getName());
    }

    private static byte[] encodePrivateKey(Key key) throws IOException {
        KeyOutput out = new KeyOutput();
        int i = new SecureRandom().nextInt();
        out.writeInt(i);
        out.writeInt(i);
        if (key.getAlgorithm() == Key.Algorithm.RSA) {
            out.writeString("ssh-rsa");
            return OpenSSHPrivateKey.writeRsaPrivateKey(key, out);
        }
        if (key.getAlgorithm() == Key.Algorithm.DSA) {
            out.writeString("ssh-dss");
            return OpenSSHPrivateKey.writePrivateDssKey(key, out);
        }
        if (key.getAlgorithm() == Key.Algorithm.EC) {
            ECPrivateKey privateKey = (ECPrivateKey)key.getKey();
            String curve = OpenSSHPublicKey.EcPublic.curveName(privateKey.getParams());
            out.writeString("ecdsa-sha2-" + curve);
            return OpenSSHPrivateKey.writeEcdsaPrivateKey(key, curve, out);
        }
        throw new UnsupportedOperationException("Unsupported key type: " + (Object)((Object)key.getAlgorithm()));
    }

    private static byte[] writePrivateDssKey(Key key, KeyOutput out) throws IOException {
        DSAPublicKey publicKey = (DSAPublicKey)key.getPublicKey().getKey();
        DSAPrivateKey privateKey = (DSAPrivateKey)key.getKey();
        out.writeBigInteger(privateKey.getParams().getP());
        out.writeBigInteger(privateKey.getParams().getQ());
        out.writeBigInteger(privateKey.getParams().getG());
        out.writeBigInteger(publicKey.getY());
        out.writeBigInteger(privateKey.getX());
        out.writeString(OpenSSHPrivateKey.getComment(key));
        return out.toByteArray();
    }

    private static Key readPrivateDssKey(KeyInput keyInput) throws IOException, NoSuchAlgorithmException, InvalidKeySpecException {
        BigInteger p = keyInput.readBigInteger();
        BigInteger q = keyInput.readBigInteger();
        BigInteger g = keyInput.readBigInteger();
        BigInteger y = keyInput.readBigInteger();
        BigInteger x = keyInput.readBigInteger();
        Dsa.Private build = Dsa.Private.builder().p(p).q(q).g(g).x(x).build();
        DSAPrivateKey privateKey = build.toKey();
        DSAPublicKey publicKey = build.toPublic().toKey();
        HashMap<String, String> attributes = new HashMap<String, String>();
        attributes.put("Comment", keyInput.readString());
        return new Key(privateKey, publicKey, Key.Type.PRIVATE, Key.Algorithm.DSA, Key.Format.OPENSSH, attributes);
    }

    private static byte[] writeRsaPrivateKey(Key key, KeyOutput out) throws IOException {
        RSAPrivateCrtKey privateKey = (RSAPrivateCrtKey)key.getKey();
        out.writeBigInteger(privateKey.getModulus());
        out.writeBigInteger(privateKey.getPublicExponent());
        out.writeBigInteger(privateKey.getPrivateExponent());
        out.writeBigInteger(privateKey.getCrtCoefficient());
        out.writeBigInteger(privateKey.getPrimeP());
        out.writeBigInteger(privateKey.getPrimeQ());
        out.writeString(OpenSSHPrivateKey.getComment(key));
        return out.toByteArray();
    }

    private static String getComment(Key key) {
        return key.getAttribute("Comment") == null ? "none" : key.getAttribute("Comment");
    }

    private static Key readRsaPrivateKey(KeyInput keyInput) throws IOException, NoSuchAlgorithmException, InvalidKeySpecException {
        BigInteger modulus = keyInput.readBigInteger();
        BigInteger publicExp = keyInput.readBigInteger();
        BigInteger privateExp = keyInput.readBigInteger();
        BigInteger crtCoef = keyInput.readBigInteger();
        BigInteger primeP = keyInput.readBigInteger();
        BigInteger primeQ = keyInput.readBigInteger();
        String comment = keyInput.readString();
        BigInteger one = BigInteger.valueOf(1L);
        BigInteger primeExpP = privateExp.mod(primeP.subtract(one));
        BigInteger primeExpQ = privateExp.mod(primeQ.subtract(one));
        Rsa.Private.Builder rsa = Rsa.Private.builder().modulus(modulus).publicExponent(publicExp).privateExponent(privateExp).crtCoefficient(crtCoef).primeP(primeP).primeQ(primeQ).primeExponentP(primeExpP).primeExponentQ(primeExpQ);
        Rsa.Private build = rsa.build();
        RSAPrivateCrtKey privateKey = build.toKey();
        RSAPublicKey publicKey = build.toPublic().toKey();
        HashMap<String, String> attributes = new HashMap<String, String>();
        attributes.put("Comment", comment);
        return new Key(privateKey, publicKey, Key.Type.PRIVATE, Key.Algorithm.RSA, Key.Format.OPENSSH, attributes);
    }

    private static byte[] writeEcdsaPrivateKey(Key key, String curve, KeyOutput out) throws IOException {
        if (key.getPublicKey() == null) {
            throw new IllegalStateException("ECPublicKey is missing.  This is required to write an ECPrivateKey to OPENSSH private key format");
        }
        ECPrivateKey privateKey = (ECPrivateKey)key.getKey();
        ECPublicKey publicKey = (ECPublicKey)key.getPublicKey().getKey();
        out.writeString(curve);
        out.writeBytes(EcPoints.toBytes(publicKey.getW()));
        out.writeBigInteger(privateKey.getS());
        out.writeString(OpenSSHPrivateKey.getComment(key));
        return out.toByteArray();
    }

    private static Key readEcdsaPrivateKey(Curve curve, KeyInput keyInput) throws IOException {
        String curveName = keyInput.readString();
        if (!curve.name().equals(curveName)) {
            throw new IllegalStateException(String.format("Mismatched curve %s does not match key type of ecdsa-sha2-%s", curveName, curve.name()));
        }
        byte[] q = keyInput.readBytes();
        ECPoint ecPoint = EcPoints.fromBytes(q);
        BigInteger d = new BigInteger(1, keyInput.readBytes());
        Ecdsa.Private build = Ecdsa.Private.builder().curveName(curveName).d(d).x(ecPoint.getAffineX()).y(ecPoint.getAffineY()).build();
        ECPrivateKey ecPrivateKey = build.toKey();
        ECPublicKey publicKey = build.toPublic().toKey();
        HashMap<String, String> attributes = new HashMap<String, String>();
        String comment = keyInput.readString();
        if (comment != null) {
            attributes.put("Comment", comment);
        } else {
            attributes.put("Comment", "");
        }
        return new Key(ecPrivateKey, publicKey, Key.Type.PRIVATE, Key.Algorithm.EC, Key.Format.OPENSSH, attributes);
    }

    public static void assertString(String name, String expected, String actual) {
        if (!expected.equals(actual)) {
            throw new IllegalArgumentException(String.format("Expected %s of '%s'. Found '%s'", name, expected, actual));
        }
    }

    public static void assertInt(String name, int expected, int actual) {
        if (expected != actual) {
            throw new IllegalArgumentException(String.format("Expected %s of '%s'. Found '%s'", name, expected, actual));
        }
    }
}

