/**
 * Copyright 2013-2023 Software Radio Systems Limited
 *
 * This file is part of srsRAN.
 *
 * srsRAN is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of
 * the License, or (at your option) any later version.
 *
 * srsRAN is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * A copy of the GNU Affero General Public License can be found in
 * the LICENSE file in the top-level directory of this distribution
 * and at http://www.gnu.org/licenses/.
 *
 */

#ifndef SRSRAN_PBCH_MSG_NR_H
#define SRSRAN_PBCH_MSG_NR_H

#include "srsran/config.h"
#include "srsran/phy/common/phy_common_nr.h"
#include <stdbool.h>
#include <stdint.h>

/**
 * @brief NR PBCH payload size generated by higher layers, deduced from TS 38.331 MIB description
 */
#define SRSRAN_PBCH_MSG_NR_SZ 24

/**
 * @brief Define the payload buffer for SRSRAN_PBCH_MSG_NR_SZ to be 32 for alignment purposes
 */
#define SRSRAN_PBCH_MSG_NR_MAX_SZ 32

/**
 * @brief Describes the NR PBCH message
 */
typedef struct SRSRAN_API {
  uint8_t payload[SRSRAN_PBCH_MSG_NR_MAX_SZ]; ///< Actual PBCH payload provided by higher layers
  uint8_t sfn_4lsb;                       ///< SFN 4 LSB
  uint8_t ssb_idx;                        ///< SS/PBCH blocks index described in TS 38.213 4.1
  uint8_t k_ssb_msb;                      ///< Subcarrier offset MSB described in TS 38.211 7.4.3.1
  bool    hrf;                            ///< Half Radio Frame bit
  bool    crc;                            ///< Decoder only, it is true only if the received CRC matches
} srsran_pbch_msg_nr_t;

typedef struct SRSRAN_API {
  uint32_t                    sfn;                    ///< System frame number
  uint8_t                     ssb_idx;                ///< SS/PBCH blocks index described in TS 38.213 4.1
  bool                        hrf;                    ///< Half Radio Frame bit
  srsran_subcarrier_spacing_t scs_common;             ///< Subcarrier spacing common
  uint32_t                    ssb_offset;             ///< SSB subcarrier offset
  srsran_dmrs_sch_typeA_pos_t dmrs_typeA_pos;         ///< DMRS typeA position
  uint32_t                    coreset0_idx;           ///< CORESET Zero configuration index (0-15)
  uint32_t                    ss0_idx;                ///< SearchSpace Zero configuration index (0-15)
  bool                        cell_barred;            ///< Set to true if the cell is barred
  bool                        intra_freq_reselection; ///< Set to true if allowed
  uint32_t                    spare;                  ///< Unused bits
} srsran_mib_nr_t;

SRSRAN_API bool srsran_pbch_msg_nr_is_mib(const srsran_pbch_msg_nr_t* msg);

SRSRAN_API int srsran_pbch_msg_nr_mib_pack(const srsran_mib_nr_t* mib, srsran_pbch_msg_nr_t* msg);

SRSRAN_API int srsran_pbch_msg_nr_mib_unpack(const srsran_pbch_msg_nr_t* pbch_msg, srsran_mib_nr_t* mib);

SRSRAN_API uint32_t srsran_pbch_msg_info(const srsran_pbch_msg_nr_t* msg, char* str, uint32_t str_len);

SRSRAN_API uint32_t srsran_pbch_msg_nr_mib_info(const srsran_mib_nr_t* mib, char* str, uint32_t str_len);

#endif // SRSRAN_PBCH_MSG_NR_H
