﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/personalize/Personalize_EXPORTS.h>
#include <aws/personalize/model/DatasetUpdateSummary.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Personalize {
namespace Model {

/**
 * <p>Provides metadata for a dataset.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/personalize-2018-05-22/Dataset">AWS
 * API Reference</a></p>
 */
class Dataset {
 public:
  AWS_PERSONALIZE_API Dataset() = default;
  AWS_PERSONALIZE_API Dataset(Aws::Utils::Json::JsonView jsonValue);
  AWS_PERSONALIZE_API Dataset& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_PERSONALIZE_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The name of the dataset.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  Dataset& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the dataset that you want metadata for.</p>
   */
  inline const Aws::String& GetDatasetArn() const { return m_datasetArn; }
  inline bool DatasetArnHasBeenSet() const { return m_datasetArnHasBeenSet; }
  template <typename DatasetArnT = Aws::String>
  void SetDatasetArn(DatasetArnT&& value) {
    m_datasetArnHasBeenSet = true;
    m_datasetArn = std::forward<DatasetArnT>(value);
  }
  template <typename DatasetArnT = Aws::String>
  Dataset& WithDatasetArn(DatasetArnT&& value) {
    SetDatasetArn(std::forward<DatasetArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the dataset group.</p>
   */
  inline const Aws::String& GetDatasetGroupArn() const { return m_datasetGroupArn; }
  inline bool DatasetGroupArnHasBeenSet() const { return m_datasetGroupArnHasBeenSet; }
  template <typename DatasetGroupArnT = Aws::String>
  void SetDatasetGroupArn(DatasetGroupArnT&& value) {
    m_datasetGroupArnHasBeenSet = true;
    m_datasetGroupArn = std::forward<DatasetGroupArnT>(value);
  }
  template <typename DatasetGroupArnT = Aws::String>
  Dataset& WithDatasetGroupArn(DatasetGroupArnT&& value) {
    SetDatasetGroupArn(std::forward<DatasetGroupArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>One of the following values:</p> <ul> <li> <p>Interactions</p> </li> <li>
   * <p>Items</p> </li> <li> <p>Users</p> </li> <li> <p>Actions</p> </li> <li>
   * <p>Action_Interactions</p> </li> </ul>
   */
  inline const Aws::String& GetDatasetType() const { return m_datasetType; }
  inline bool DatasetTypeHasBeenSet() const { return m_datasetTypeHasBeenSet; }
  template <typename DatasetTypeT = Aws::String>
  void SetDatasetType(DatasetTypeT&& value) {
    m_datasetTypeHasBeenSet = true;
    m_datasetType = std::forward<DatasetTypeT>(value);
  }
  template <typename DatasetTypeT = Aws::String>
  Dataset& WithDatasetType(DatasetTypeT&& value) {
    SetDatasetType(std::forward<DatasetTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the associated schema.</p>
   */
  inline const Aws::String& GetSchemaArn() const { return m_schemaArn; }
  inline bool SchemaArnHasBeenSet() const { return m_schemaArnHasBeenSet; }
  template <typename SchemaArnT = Aws::String>
  void SetSchemaArn(SchemaArnT&& value) {
    m_schemaArnHasBeenSet = true;
    m_schemaArn = std::forward<SchemaArnT>(value);
  }
  template <typename SchemaArnT = Aws::String>
  Dataset& WithSchemaArn(SchemaArnT&& value) {
    SetSchemaArn(std::forward<SchemaArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the dataset.</p> <p>A dataset can be in one of the following
   * states:</p> <ul> <li> <p>CREATE PENDING &gt; CREATE IN_PROGRESS &gt; ACTIVE -or-
   * CREATE FAILED</p> </li> <li> <p>DELETE PENDING &gt; DELETE IN_PROGRESS</p> </li>
   * </ul>
   */
  inline const Aws::String& GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  template <typename StatusT = Aws::String>
  void SetStatus(StatusT&& value) {
    m_statusHasBeenSet = true;
    m_status = std::forward<StatusT>(value);
  }
  template <typename StatusT = Aws::String>
  Dataset& WithStatus(StatusT&& value) {
    SetStatus(std::forward<StatusT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The creation date and time (in Unix time) of the dataset.</p>
   */
  inline const Aws::Utils::DateTime& GetCreationDateTime() const { return m_creationDateTime; }
  inline bool CreationDateTimeHasBeenSet() const { return m_creationDateTimeHasBeenSet; }
  template <typename CreationDateTimeT = Aws::Utils::DateTime>
  void SetCreationDateTime(CreationDateTimeT&& value) {
    m_creationDateTimeHasBeenSet = true;
    m_creationDateTime = std::forward<CreationDateTimeT>(value);
  }
  template <typename CreationDateTimeT = Aws::Utils::DateTime>
  Dataset& WithCreationDateTime(CreationDateTimeT&& value) {
    SetCreationDateTime(std::forward<CreationDateTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A time stamp that shows when the dataset was updated.</p>
   */
  inline const Aws::Utils::DateTime& GetLastUpdatedDateTime() const { return m_lastUpdatedDateTime; }
  inline bool LastUpdatedDateTimeHasBeenSet() const { return m_lastUpdatedDateTimeHasBeenSet; }
  template <typename LastUpdatedDateTimeT = Aws::Utils::DateTime>
  void SetLastUpdatedDateTime(LastUpdatedDateTimeT&& value) {
    m_lastUpdatedDateTimeHasBeenSet = true;
    m_lastUpdatedDateTime = std::forward<LastUpdatedDateTimeT>(value);
  }
  template <typename LastUpdatedDateTimeT = Aws::Utils::DateTime>
  Dataset& WithLastUpdatedDateTime(LastUpdatedDateTimeT&& value) {
    SetLastUpdatedDateTime(std::forward<LastUpdatedDateTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Describes the latest update to the dataset.</p>
   */
  inline const DatasetUpdateSummary& GetLatestDatasetUpdate() const { return m_latestDatasetUpdate; }
  inline bool LatestDatasetUpdateHasBeenSet() const { return m_latestDatasetUpdateHasBeenSet; }
  template <typename LatestDatasetUpdateT = DatasetUpdateSummary>
  void SetLatestDatasetUpdate(LatestDatasetUpdateT&& value) {
    m_latestDatasetUpdateHasBeenSet = true;
    m_latestDatasetUpdate = std::forward<LatestDatasetUpdateT>(value);
  }
  template <typename LatestDatasetUpdateT = DatasetUpdateSummary>
  Dataset& WithLatestDatasetUpdate(LatestDatasetUpdateT&& value) {
    SetLatestDatasetUpdate(std::forward<LatestDatasetUpdateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the event tracker for an Action interactions dataset. You specify
   * the tracker's ID in the <code>PutActionInteractions</code> API operation. Amazon
   * Personalize uses it to direct new data to the Action interactions dataset in
   * your dataset group.</p>
   */
  inline const Aws::String& GetTrackingId() const { return m_trackingId; }
  inline bool TrackingIdHasBeenSet() const { return m_trackingIdHasBeenSet; }
  template <typename TrackingIdT = Aws::String>
  void SetTrackingId(TrackingIdT&& value) {
    m_trackingIdHasBeenSet = true;
    m_trackingId = std::forward<TrackingIdT>(value);
  }
  template <typename TrackingIdT = Aws::String>
  Dataset& WithTrackingId(TrackingIdT&& value) {
    SetTrackingId(std::forward<TrackingIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;

  Aws::String m_datasetArn;

  Aws::String m_datasetGroupArn;

  Aws::String m_datasetType;

  Aws::String m_schemaArn;

  Aws::String m_status;

  Aws::Utils::DateTime m_creationDateTime{};

  Aws::Utils::DateTime m_lastUpdatedDateTime{};

  DatasetUpdateSummary m_latestDatasetUpdate;

  Aws::String m_trackingId;
  bool m_nameHasBeenSet = false;
  bool m_datasetArnHasBeenSet = false;
  bool m_datasetGroupArnHasBeenSet = false;
  bool m_datasetTypeHasBeenSet = false;
  bool m_schemaArnHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_creationDateTimeHasBeenSet = false;
  bool m_lastUpdatedDateTimeHasBeenSet = false;
  bool m_latestDatasetUpdateHasBeenSet = false;
  bool m_trackingIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace Personalize
}  // namespace Aws
