package MailBot::Config;

=head1 NAME

MailBot::Config - interface to config/mailbot.config file

=head1 SYNOPSIS

 use MailBot::Config;
 MailBot::Config -> setRoot("/usr/local/my_bot/");
 my $config = MailBot::Config -> current;

 $config -> getIniVal("mysection","myparam","mydefault");

=head1 DESCRIPTION

MailBot::Config provides an interface for reading the configuration
file.

=head1 METHODS

=cut

use MailBot::IniConf;

use Carp;
use strict;

my $gRoot = "/tmp/mailbot";
my $gCurrentConfig;

=head2 setRoot($root)

Set the root to $root, so that the config file is at
$root/config/mailbot.config, and the pub directory is at $root/pub/,
etc.

Call this method BEFORE your first call to L<current>, otherwise your
default root will be /tmp/mailbot/.

=cut

sub setRoot
{
    my $self = shift;
    my $rootDir = shift;

    $gRoot = $rootDir;

    $gCurrentConfig = new MailBot::Config($gRoot);
}

=head2 current()

get the current MailBot::Config instance.  Do not call this method
before your call to L<setRoot>.

=cut

sub current
{
    $gCurrentConfig or croak "have not yet set root";
    $gCurrentConfig;
}

sub getRoot
{
    $gRoot;
}

sub getIni
{
    my $ini = shift -> {INI};
    croak "unloaded inifile" if ! defined $ini;
    $ini;
}

sub getMaxSize
{
    shift -> getIniVal("load","maxsize",500000);
}

sub getEnvelopeVal
{
    my $self = shift;
    my $condition = shift;
    my $param = shift;

    my $lower = lc $condition;
    $lower =~ s/ /_/g;
    my $section = "envelope.$lower";

    # check if we should use default
    my @aSection = $self -> getIni -> Sections;
    grep($_ eq $section,@aSection) or
	$section = "envelope.";

    $self -> getIniVal($section,$param);
}

=head2 getIniVal($section,$param,[$default])

Looks up section and parameter in ini-file.  If value is not listed in
inifile, returns $default (which may be the undefined value).

=cut

sub getIniVal
{
    my $ini = shift -> getIni;
    my ($section,$param,$default) = @_;
    my $val = scalar($ini -> val($section,$param));
    return $val if defined $val;
    $default;
}

sub getSplitSize
{
    shift -> getIniVal('smtp','split_size');
}

sub getSplitMultiplier
{
    shift -> getIniVal('smtp','split_multiplier');
}

sub neverSend
{
    shift -> getIniVal('smtp','never_send');
}

sub new
{
    my $type = shift;
    my $root = shift;

    my $self = {
	INTERACTIVE => 0
    };

    my $mailBotConfigFile = "$root/config/mailbot.config";
    my $iniConf = new MailBot::IniConf -file => $mailBotConfigFile;
    $$self{INI} = $iniConf;

    bless($self,$type);

    $self;
}

sub isInteractive
{
    my $self = shift;
    $self -> {INTERACTIVE};
}

sub getLocalSpool
{
    my $spoolDir = shift -> getDir('spool');

    my $progName = $0;
    $progName =~ s/.+\///;
    $progName =~ s/\.pl$//;
    return "$spoolDir/rfc822.in.$progName";
}

sub getMailSpool
{
    shift -> {SPOOL};
}

=head2 setMailSpool($spool_file), setInteractive(), setCGI()

set the interface method: mail spool, interactive command-line, or
CGI.  Default is individual mail messages.

=cut

sub setMailSpool
{
    my $self = shift;
    $self -> {SPOOL} = shift;
}

sub setInteractive
{
    my $self = shift;
    $self -> {INTERACTIVE} = 1;
}

sub isCGI
{
    my $self = shift;
    $self -> {CGI};
}

sub setCGI
{
    my $self = shift;
    $self -> {CGI} = 1;
}

sub getBody
{
    shift -> {BODY};
}

sub getBounceAddr
{
    my $ini = shift -> {INI};
    return undef if ! defined $ini;
    $ini -> val('address','bounce');
}

sub setBody
{
    my $self = shift;
    $self -> {BODY} = shift;
}

sub getSubject
{
    shift -> {SUBJECT};
}

sub setSubject
{
    my $self = shift;
    $self -> {SUBJECT} = shift;
}

sub getDir
{
    my $self = shift;
    my $dirName = shift;
    
    my $path = "$gRoot/$dirName";
    if (! -d $path)
    {
	mkdir($path,0777) or die "could not mkdir $path: $!";
    }
    $path;
}

sub getSaveDir
{
    shift -> getDir('save');
}

sub getQuotaDir
{
    shift -> getDir('quota');
}

=head2 getPubDir()

Returns the public directory, usually $root/pub/.

=cut

sub getPubDir
{
    "$gRoot/pub";
}

sub getLogDir
{
    shift -> getDir('log');
}

=head2 log($message_string)

log message to $root/log/log file, with date.

=cut

sub log
{
    my $self = shift;
    my $string = join('',@_);
    chomp($string);
    $string =~ s/\n/\\n/g;

    my $logDir = $self -> getLogDir();
    my $logFile = "$logDir/log";

    my $logString = scalar(localtime()) . ": " . $string . "\n";

    if (open(LOG,">>$logFile"))
    {
	print LOG $logString;
	close(LOG);
    }
    else
    {
	print STDERR $logString;
    }
}

1;

=head1 SEE ALSO

L<GetWeb::ProcMsg>, L<getweb>, L<MailBot::MailBot>,
L<http://www.healthnet.org/dist/getweb/>

=head1 COPYRIGHT

Copyright (c) 1997 SatelLife.  All rights reserved.

This library is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 AUTHOR

Rolf Nelson <rolf@usa.healthnet.org>

=cut
