package MailBot::Envelope;

=head1 NAME

MailBot::Envelope - encapsulates MailBot-specific state information
relating to an outgoing message

=head1 SYNOPSIS

 use MailBot::Entity;
 use MailBot::Envelope;

 my $envelope = new MailBot::Envelope();
 $envelope -> setSubject("my subject");
 $envelope -> setMIME(0);  #send binary files uuencoded, not base64

 MailBot::Entity::setEnvelope(new MailBot::Envelope());

=head1 DESCRIPTION

MailBot::Entity extends extends L<MIME::Entity> to support quotas and
L<MailBot::Envelope> objects.

=head1 METHODS

=cut

use MailBot::Recip;
use MailBot::UI;
use MailBot::Config;

use strict;

sub d
{
    &MailBot::Util::debug(@_);
}

=head2 new()

Create a new envelope, with the default recipient being the sender of
the original, incoming message.

=cut

sub new
{
    my $type = shift;

    my $ui = MailBot::UI::current;

    my $replyTo = $ui -> getReturnAddress;
    my $defaultRecip = new MailBot::Recip($replyTo);

    my $self = {DEFAULT_RECIP => $defaultRecip,
		MESSAGE_TYPE => "normal",
		QUOTA_MULTIPLIER => 1,
		MIME => 1};

    bless($self,$type);
}

=head2 setSplitSize($size)

Set size which message cannot exceed without being split into multiple
MIME message/partial fragments before sending

=cut

sub setSplitSize
{
    my $self = shift;
    my $size = shift;

    $size =~ /^\d+$/ or
	die "illegal split size: $size\n";

    $$self{SPLIT_SIZE} = $size;
}

sub getSplitSize
{
    my $self = shift;
    my $size = $$self{SPLIT_SIZE};
    return $size if defined $size;

    my $config = MailBot::Config::current;
    $config -> getSplitSize;
}

=head2 setMIME($bool)

If $bool is 1, binary messages are as MIME base64.  If $bool is 0,
binary messages are sent uuencoded.

=cut

sub setMIME
{
    my $self = shift;
    $$self{MIME} = shift;
}

sub getRecip
{
    my $self = shift;
    my $recip = $$self{RECIP};
    return $recip if defined $recip;

    # &d("using default");
    $self -> {DEFAULT_RECIP};
}

sub getQuotaMultiplier
{
    shift -> {QUOTA_MULTIPLIER};
}

=head2 setRecipientList($address_string [...])

Set the recipient list to the given address(es).  The first address
given will be 'To', subsequent ones will be Cc'd.

=cut

sub setRecipientList
{
    my $self = shift;
    # &d("setting list to ",@_);
    $self -> {RECIP} = new MailBot::Recip(@_);
}

sub setDefaultAddress
{
    my $self = shift;

    $self -> {DEFAULT_RECIP} = new MailBot::Recip(@_);
}

=head2 setQuotaMultiplier($new_quota_multiplier)

Default quota multiplier is 1.  If the user should not be charged for
this message, set to 0.

=cut

sub setQuotaMultiplier
{
    my $self = shift;

    $$self{QUOTA_MULTIPLIER} = shift;
}

sub getFrom
{
    shift -> {FROM};
}

sub setFrom
{
    my $self = shift;
    my $sender = shift;

    $$self{FROM} = $sender;
}

sub getSubject
{
    shift -> {SUBJECT};
}

=head2 setSubject($subject)

sets the Subject: line of the outgoing message.

=cut

sub setSubject
{
    my $self = shift;
    my $subject = shift;

    $$self{SUBJECT} = $subject;
}

sub getLastByte
{
    my $config = MailBot::Config::current;
    $config -> getMaxSize;
}

1;

=head1 SEE ALSO

L<MailBot::Quota>, L<MailBot::Entity>

=head1 COPYRIGHT

Copyright (c) 1997 SatelLife.  All rights reserved.

This library is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 AUTHOR

Rolf Nelson <rolf@usa.healthnet.org>

=cut
