/*  $Header: /home/mwicks/Repository/dvipdfm/pdfdoc.c,v 1.32 1998/12/30 19:36:10 mwicks Exp $
 
    This is dvipdf, a DVI to PDF translator.
    Copyright (C) 1998  by Mark A. Wicks

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
    
    The author may be contacted via the e-mail address

	mwicks@kettering.edu
*/


#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include "c-auto.h"
#include "system.h"
#include "pdflimits.h"
#include "pdfobj.h"
#include "error.h"
#include "mem.h"
#include "pdfdoc.h"
#include "pdfspecial.h"
#include "pdfdev.h"
#include "numbers.h"
#include "mfileio.h"

static pdf_obj *catalog = NULL;
static pdf_obj *docinfo = NULL;
static pdf_obj *page_tree = NULL, *page_tree_ref = NULL;

int outline_depth=0;
static struct 
{
  int kid_count;
  pdf_obj *entry;
} outline[MAX_OUTLINE_DEPTH];


static pdf_obj *current_page_resources = NULL;
static pdf_obj *this_page_contents = NULL;
static pdf_obj *glob_page_bop, *glob_page_eop;
static pdf_obj *this_page_bop = NULL, *this_page_eop = NULL;
static pdf_obj *this_page_beads = NULL;
static pdf_obj *this_page_annots = NULL;
static pdf_obj *this_page_xobjects = NULL, *this_page_fonts = NULL;
static pdf_obj *tmp1;

static unsigned long page_count = 0;
static struct pages {
  pdf_obj *page_dict;
  pdf_obj *page_ref;
} *pages = NULL;
static unsigned long max_pages = 0;

static void start_page_tree (void);
static void create_catalog (void);
static void start_current_page_resources (void);
static void finish_page_tree(void);
static void start_name_tree(void);
static void finish_dests_tree(void);
static void finish_pending_xobjects(void);
static void start_articles(void);

static unsigned char verbose = 0, debug=0;

void pdf_doc_set_verbose(void)
{
  if (verbose < 255) 
    verbose += 1;
}

void pdf_doc_set_debug(void)
{
  debug = 1;
}

static void resize_pages (unsigned long newsize)
{
  unsigned long i;
  if (newsize > max_pages) {
    pages = RENEW (pages, newsize, struct pages);
    for (i=max_pages; i<newsize; i++) {
      pages[i].page_dict = NULL;
      pages[i].page_ref = NULL;
    }
    max_pages = newsize;
  }
}

static pdf_obj *type_name, *page_name, *pages_name, *contents_name, *annots_name,
  *resources_name, *bead_name, *count_name, *kids_name, *parent_name,
  *mediabox_name, *limits_name;

static void make_short_cuts(void) 
{
  /* Define some shorthand for names that will conserve memory (and time)
     (similar to the latex \@ne trick */
  type_name = pdf_new_name("Type");
  page_name = pdf_new_name("Page");
  pages_name = pdf_new_name("Pages");
  count_name = pdf_new_name("Count");
  kids_name = pdf_new_name("Kids");
  parent_name = pdf_new_name("Parent");
  contents_name = pdf_new_name("Contents");
  annots_name = pdf_new_name("Annots");
  resources_name = pdf_new_name ("Resources");
  bead_name = pdf_new_name ("B");
  mediabox_name = pdf_new_name ("MediaBox");
  limits_name = pdf_new_name ("Limits");
}
static void release_short_cuts(void)
{
  /* Release those shorthand name we created */
  pdf_release_obj (type_name);
  pdf_release_obj (page_name);
  pdf_release_obj (pages_name);
  pdf_release_obj (count_name);
  pdf_release_obj (kids_name);
  pdf_release_obj (parent_name);
  pdf_release_obj (contents_name);
  pdf_release_obj (annots_name);
  pdf_release_obj (resources_name);
  pdf_release_obj (bead_name);
  pdf_release_obj (mediabox_name);
  pdf_release_obj (limits_name);
}

static void start_page_tree (void)
{
  if (debug) {
    fprintf (stderr, "(start_page_tree)");
  }
  /* Create empty page tree */
  page_tree = pdf_new_dict();
  page_tree_ref = pdf_ref_obj (page_tree);
  /* Both page_tree and page_tree_ref are kept open until the
     document is closed.  This allows the user to write to page_tree
     if he so choses. */
  /* Link /Pages into the catalog
     and poing it to indirect reference since we
     haven't built the tree yet */
  glob_page_bop = pdf_new_stream(0);
  glob_page_eop = pdf_new_stream(0);
  return;
}

void pdf_doc_bop (char *string, unsigned length)
{
  if (length > 0)
    pdf_add_stream (glob_page_bop, string, length);
}

void pdf_doc_this_bop (char *string, unsigned length)
{
  if (length > 0)
    pdf_add_stream (this_page_bop, string, length);
}

void pdf_doc_this_eop (char *string, unsigned length)
{
  if (length > 0)
    pdf_add_stream (this_page_eop, string, length);
}


void pdf_doc_eop (char *string, unsigned length)
{
  if (length > 0)
    pdf_add_stream (glob_page_eop, string, length);
}

static void start_outline_tree (void)
{
  if (debug) {
    fprintf (stderr, "(start_outline_tree)");
  }
  /* Create empty outline tree */
  outline[outline_depth].entry = pdf_new_dict();
  outline[outline_depth].kid_count = 0;
  return;
}

static pdf_obj *names_dict;

static void start_name_tree (void)
{
  if (debug) {
    fprintf (stderr, "(start_name_tree)");
  }
  names_dict = pdf_new_dict ();
}

static char *asn_date (void)
{
#ifndef HAVE_TIMEZONE
  #ifdef TM_GM_TOFF
     #define timezone (bdtime->gm_toff)
  #else
     #define timezone 0l
#endif /* TM_GM_TOFF */
#endif /* HAVE_TIMEZONE */
  static char date_string[22];
  time_t current_time;
  struct tm *bd_time;
  if (debug) {
    fprintf (stderr, "(asn_date)");
  }
  time(&current_time);
  bd_time = localtime(&current_time);
  sprintf (date_string, "D:%04d%02d%02d%02d%02d%02d%0+3ld'%02ld'",
	   bd_time -> tm_year+1900, bd_time -> tm_mon+1, bd_time -> tm_mday,
	   bd_time -> tm_hour, bd_time -> tm_min, bd_time -> tm_sec,
	   -timezone/3600, timezone%3600);
  return date_string;
}

#define BANNER "dvipdfm %s, Copyright \251 1998, by Mark A. Wicks"
static void create_docinfo (void)
{
  /* Create an empty Info entry and make it
     be the root object */
  if (debug) {
    fprintf (stderr, "(create_docinfo)");
  }
  docinfo = pdf_new_dict ();
  pdf_set_info (docinfo);
  return;
}

static void finish_docinfo(void)
{
  char *time_string, *banner;
  banner = NEW (strlen(BANNER)+20,char);
  sprintf(banner, BANNER, VERSION);
  pdf_add_dict (docinfo, 
		pdf_new_name ("Producer"),
		pdf_new_string (banner, strlen (banner)));
  RELEASE (banner);
  time_string = asn_date();
  pdf_add_dict (docinfo, 
		pdf_new_name ("CreationDate"),
		pdf_new_string (time_string, strlen (time_string)));
  pdf_release_obj (docinfo);
  return;
}

void pdf_doc_merge_with_docinfo (pdf_obj *dictionary)
{
  pdf_merge_dict (docinfo, dictionary);
}

void pdf_doc_merge_with_catalog (pdf_obj *dictionary)
{
  pdf_merge_dict (catalog, dictionary);
}

static void create_catalog (void)
{
  if (debug) {
    fprintf (stderr, "(create_catalog)");
  }
  catalog = pdf_new_dict ();
  pdf_set_root (catalog);
  /* Create /Type attribute */
  pdf_add_dict (catalog,
		pdf_link_obj (type_name),
		pdf_new_name("Catalog"));
 /* Create only those parts of the page tree required for the catalog.
    That way, the rest of the page tree can be finished at any time */
  start_page_tree(); 
  /* Likewise for outline tree */
  start_outline_tree ();
  start_name_tree();
  start_articles();
  return;
}

static void start_current_page_resources (void)
{
  /* work on resources to put in Pages */
  if (debug) {
    fprintf (stderr, "(start_current_page_resources)");
  }
  current_page_resources = pdf_new_dict ();
  tmp1 = pdf_new_array ();
  pdf_add_array (tmp1, pdf_new_name ("PDF"));
  pdf_add_array (tmp1, pdf_new_name ("Text"));
  pdf_add_array (tmp1, pdf_new_name ("ImageC"));
  pdf_add_dict (current_page_resources,
		pdf_new_name ("ProcSet"),
		tmp1);
  this_page_fonts = pdf_new_dict ();
  pdf_add_dict (current_page_resources, 
		pdf_new_name ("Font"),
		pdf_ref_obj (this_page_fonts));
  this_page_xobjects = pdf_new_dict ();
  pdf_add_dict (current_page_resources,
		pdf_new_name ("XObject"),
		pdf_ref_obj (this_page_xobjects));
  return;
}

void pdf_doc_add_to_page_fonts (const char *name, pdf_obj
				   *resource)
{
#ifdef MEM_DEBUG
MEM_START
#endif
  if (debug) {
    fprintf (stderr, "(pdf_doc_add_to_page_fonts)");
  }
  pdf_add_dict (this_page_fonts,
		pdf_new_name (name), resource);
#ifdef MEM_DEBUG
MEM_END
#endif
}

void pdf_doc_add_to_page_xobjects (const char *name, pdf_obj
				   *resource)
{
  if (debug) {
    fprintf (stderr, "(pdf_doc_add_to_page_xojects)");
  }
  pdf_add_dict (this_page_xobjects,
		pdf_new_name (name), 
		resource);
}


void pdf_doc_add_to_page_resources (const char *name, pdf_obj *resource)
{
  if (debug) {
    fprintf (stderr, "(pdf_doc_add_to_page_resources)");
  }
  pdf_add_dict (current_page_resources,
		pdf_new_name (name), 
		resource);
}

void pdf_doc_add_to_page_annots (pdf_obj *annot)
{
  if (debug) {
    fprintf (stderr, "(pdf_doc_add_to_page_annots)");
  }
  pdf_add_array (this_page_annots,
		 annot);
}

static pdf_obj *page_subtree (struct pages *pages, unsigned long npages,
		       pdf_obj *parent_ref)
{
#define PAGE_CLUSTER 4
  pdf_obj *self, *self_ref, *kid_array;
  self = pdf_new_dict();
  /* This is a slight kludge which allow the subtree
     dictionary generated by this routine to be merged with the
     real page_tree dictionary, while keeping the indirect 
     object references right */
  if (parent_ref == NULL)
    self_ref = pdf_ref_obj (page_tree);
  else
    self_ref = pdf_ref_obj (self);
  pdf_add_dict (self, pdf_link_obj (type_name),
		pdf_link_obj (pages_name));
  pdf_add_dict (self, pdf_link_obj (count_name),
		pdf_new_number((double) npages));
  kid_array = pdf_new_array();
  pdf_add_dict (self, pdf_link_obj (kids_name), 
		kid_array);
  if (parent_ref != NULL) {
    pdf_add_dict (self, pdf_link_obj(parent_name),
		  parent_ref);
  }
  if (npages > 0 && npages <= PAGE_CLUSTER) {
    int i;
    for (i=0; i<npages; i++) {
      pdf_add_array (kid_array, pdf_link_obj(pages[i].page_ref));
      pdf_add_dict (pages[i].page_dict, pdf_link_obj (parent_name),
		    pdf_link_obj(self_ref));
      pdf_release_obj (pages[i].page_dict);
      pdf_release_obj (pages[i].page_ref);
      pages[i].page_dict = NULL;
      pages[i].page_ref = NULL;
    }
  } else if (npages > 0) {
    int i;
    for (i=0; i<PAGE_CLUSTER; i++) {
      pdf_obj *subtree;
      unsigned long start, end;
      start = (i*npages)/PAGE_CLUSTER;
      end = ((i+1)*npages)/PAGE_CLUSTER;
      subtree = page_subtree (pages+start, end-start, pdf_link_obj(self_ref));
      pdf_add_array (kid_array, pdf_ref_obj (subtree));
      pdf_release_obj (subtree);
    }
  }
  pdf_release_obj (self_ref);
  return self;
}

static void finish_page_tree(void)
{
  pdf_obj *subtree;
  if (debug) {
    fprintf (stderr, "(finish_page_tree)");
  }
  
  subtree = page_subtree (pages, page_count, NULL);
  pdf_merge_dict (page_tree, subtree);
  pdf_release_obj (subtree);
  pdf_release_obj (page_tree);
  pdf_add_dict (catalog,
		pdf_link_obj (pages_name),
		pdf_link_obj (page_tree_ref));
  pdf_release_obj (page_tree_ref);
  RELEASE (pages);
  return;
}

void pdf_doc_change_outline_depth(int new_depth)
{
  int i;
  if (debug) {
    fprintf (stderr, "(change_outline_depth)");
  }
  if (outline_depth >= MAX_OUTLINE_DEPTH -1)
    ERROR ("Outline is too deep.");
  if (new_depth == outline_depth)
    /* Nothing to do */
    return;
  if (new_depth > outline_depth+1)
    ERROR ("Can't increase outline depth by more than one at a time\n");
  if (outline[outline_depth].entry == NULL)
    ERROR ("change_outline_depth: Fix me, I'm broke. This shouldn't happen!");
  /* Terminate all entries above this depth */
  for (i=outline_depth-1; i>=new_depth; i--) {
    pdf_add_dict (outline[i].entry,
		  pdf_new_name ("Last"),
		  pdf_ref_obj (outline[i+1].entry));
    if (i > 0) 
      tmp1 = pdf_new_number (-outline[i].kid_count);
    else
      tmp1 = pdf_new_number (outline[i].kid_count);

    pdf_add_dict (outline[i].entry,
		  pdf_link_obj (count_name),
		  tmp1);
  }
  /* Flush out all entries above this depth */
  for (i=new_depth+1; i<=outline_depth; i++) {
    pdf_release_obj (outline[i].entry);
    outline[i].entry = NULL;
    outline[i].kid_count = 0;
  }
  outline_depth = new_depth;
}

static void finish_outline(void)
{
  if (debug)
    fprintf (stderr, "(finish_outline)");
  /* Link it into the catalog */
  /* Point /Outline attribute to indirect reference */
  pdf_doc_change_outline_depth (0);
  pdf_add_dict (catalog,
		pdf_new_name ("Outlines"),
		pdf_ref_obj(outline[outline_depth].entry));
  pdf_release_obj (outline[0].entry);
  outline[0].entry = NULL;
}


void pdf_doc_add_outline (pdf_obj *dict)
{
  pdf_obj *new_entry;
  if (outline_depth < 1)
    ERROR ("Can't add to outline at depth < 1");
  new_entry = pdf_new_dict ();
  pdf_merge_dict (new_entry, dict);
  /* Caller doesn't know we don't actually use the dictionary,
     so he *gave* dict to us.  We have to free it */
  pdf_release_obj (dict);
  /* Tell it where its parent is */
  pdf_add_dict (new_entry,
		pdf_link_obj (parent_name),
		pdf_ref_obj (outline[outline_depth-1].entry));
  /* Give mom and dad the good news */
  outline[outline_depth-1].kid_count += 1;

  /* Is this the first entry at this depth? */
  if (outline[outline_depth].entry == NULL) {
    /* Is so, tell the parent we are first born */
    pdf_add_dict (outline[outline_depth-1].entry,
		  pdf_new_name ("First"),
		  pdf_ref_obj (new_entry));
  }
  else {
    /* Point us back to sister */
    pdf_add_dict (new_entry,
		  pdf_new_name ("Prev"),
		  pdf_ref_obj (outline[outline_depth].entry));
    /* Point our elder sister toward us */
    pdf_add_dict (outline[outline_depth].entry,
		  pdf_new_name ("Next"),
		  pdf_ref_obj (new_entry));
    /* Bye-Bye sis */
    pdf_release_obj (outline[outline_depth].entry);
  }
  outline[outline_depth].entry = new_entry;
  /* Just born, so don't have any kids */
  outline[outline_depth].kid_count = 0;
}

struct dests 
{
  char *name;
  unsigned length;
  pdf_obj *array;
};
typedef struct dests dest_entry;
static dest_entry *dests = NULL;
unsigned long max_dests = 0;


#define MIN(a,b) ((a)<(b)? (a): (b))

static int CDECL cmp_dest (const void *d1, const void *d2)
{
  unsigned length;
  int tmp;
  length = MIN (((dest_entry *) d1) -> length, ((dest_entry *) d2) ->
		length);
  if ((tmp = memcmp (((dest_entry *) d1) -> name, ((dest_entry *) d2)
		      -> name, length)) != 0)
    return tmp;
  if (((dest_entry *) d1) -> length == ((dest_entry *) d2) -> length)
    return 0;
  return (((dest_entry *) d1) -> length < ((dest_entry *) d2) -> length ? -1 : 1 );
}

static pdf_obj *name_subtree (dest_entry *dests, unsigned long ndests)
{
#define CLUSTER 4
  pdf_obj *result, *name_array, *limit_array, *kid_array;
  result = pdf_new_dict();
  limit_array = pdf_new_array();
  pdf_add_dict (result, pdf_link_obj(limits_name), limit_array);
  pdf_add_array (limit_array, pdf_new_string(dests[0].name,
					     dests[0].length)); 
  pdf_add_array (limit_array, pdf_new_string(dests[ndests-1].name,
					     dests[ndests-1].length));
  if (ndests > 0 && ndests <= CLUSTER) {
    int i;
    name_array = pdf_new_array();
    pdf_add_dict (result, pdf_new_name ("Names"),
		  name_array);
    for (i=0; i<ndests; i++) {
      pdf_add_array (name_array, pdf_new_string (dests[i].name,
						 dests[i].length));
      RELEASE (dests[i].name);
      pdf_add_array (name_array, dests[i].array);
    }
  } else if (ndests > 0) {
    int i;
    kid_array = pdf_new_array();
    pdf_add_dict (result, pdf_link_obj (kids_name), kid_array);
    for (i=0; i<CLUSTER; i++) {
      pdf_obj *subtree;
      unsigned long start, end;
      start = (i*ndests)/CLUSTER;
      end = ((i+1)*ndests)/CLUSTER;
      subtree = name_subtree (dests+start, end-start);
      pdf_add_array (kid_array, pdf_ref_obj (subtree));
      pdf_release_obj (subtree);
    }
  }
  return result;
}

static number_dests = 0;

static void finish_dests_tree (void)
{
  pdf_obj *kid;
  if (number_dests > 0) {
    /* Sort before writing any /Dests entries */
    qsort(dests, number_dests, sizeof(dests[0]), cmp_dest);
    kid = name_subtree (dests, number_dests);
    /* Each entry in dests has been assigned to another object, so
       we can free the entire array without freeing the entries. */
    RELEASE (dests);
    pdf_add_dict (names_dict,
		  pdf_new_name ("Dests"),
		  pdf_ref_obj (kid));
    pdf_release_obj (kid);
  }
}

void pdf_doc_add_dest (char *name, unsigned length, pdf_obj *array_ref)
{
#ifdef MEM_DEBUG
MEM_START
#endif
  if (number_dests >= max_dests) {
    max_dests += DESTS_ALLOC_SIZE;
    dests = RENEW (dests, max_dests, dest_entry);
  }
  dests[number_dests].name = NEW (length, char);
  memcpy (dests[number_dests].name, name, length);
  dests[number_dests].length = length;
  dests[number_dests].array = array_ref;
  number_dests++;
#ifdef MEM_DEBUG
MEM_END
#endif
  return;
}

struct articles
{
  char *name;
  pdf_obj *info;
  pdf_obj *first;
  pdf_obj *last;
  pdf_obj *this;
};

typedef struct articles article_entry;
static article_entry articles[MAX_ARTICLES];
static number_articles = 0;

static pdf_obj *articles_array;
static void start_articles (void)
{
  articles_array = pdf_new_array();
}

void pdf_doc_start_article (char *name, pdf_obj *info)
{
  if (number_articles >= MAX_ARTICLES) {
    ERROR ("pdf_doc_add_article:  Too many articles\n");
  }
  if (name == NULL || strlen (name) == 0)
    ERROR ("pdf_doc_start_article called null name");
  articles[number_articles].name = NEW (strlen(name)+1, char);
  strcpy (articles[number_articles].name, name);
  articles[number_articles].info = info;
  articles[number_articles].first = NULL;
  articles[number_articles].last = NULL;
  /* Start dictionary for this article even though we can't finish it
     until we get the first bead */
  articles[number_articles].this = pdf_new_dict();
  number_articles++;
  return;
}

void pdf_doc_add_bead (char *article_name, pdf_obj *partial_dict)
{
  /* partial_dict should have P (Page) and R (Rect) already filled in */
  /* See if the specified article exists */
  int i;
  for (i=0; i<number_articles; i++) {
    if (!strcmp (articles[i].name, article_name))
      break;
  }
  if (i == number_articles) {
    fprintf (stderr, "Bead specified thread that doesn't exist\n");
    return;
  }
  /* Is this the first bead? */
  if (articles[i].last == NULL) {
    articles[i].first = pdf_link_obj (partial_dict);
    /* Add pointer to its first object */ 
    pdf_add_dict (articles[i].this,
		  pdf_new_name ("F"),
		  pdf_ref_obj (articles[i].first));
    /* Next add pointer to its Info dictionary */
    pdf_add_dict (articles[i].this,
		  pdf_new_name ("I"),
		  pdf_ref_obj (articles[i].info));
    /* Point first bead to parent article */
    pdf_add_dict (partial_dict,
		  pdf_new_name ("T"),
		  pdf_ref_obj (articles[i].this));
    /* Ship it out and forget it */
    pdf_add_array (articles_array, pdf_ref_obj (articles[i].this));
    pdf_release_obj (articles[i].this);
    articles[i].this = NULL;
  } else {
    /* Link it in... */
    /* Point last object to this one */
    pdf_add_dict (articles[i].last,
		  pdf_new_name ("N"),
		  pdf_ref_obj (partial_dict));
    /* Point this one to last */
    pdf_add_dict (partial_dict,
		  pdf_new_name ("V"),
		  pdf_ref_obj (articles[i].last));
    pdf_release_obj (articles[i].last);
  }
  articles[i].last = partial_dict;
  pdf_add_array (this_page_beads,
		 pdf_ref_obj (partial_dict));
}

void finish_articles(void)
{
  int i;
  pdf_add_dict (catalog,
		pdf_new_name ("Threads"),
		pdf_ref_obj (articles_array));
  pdf_release_obj (articles_array);
  for (i=0; i<number_articles; i++) {
    if (articles[i].last == NULL) {
      fprintf (stderr, "Article started, but no beads\n");
      break;
    }
    /* Close the loop */
    pdf_add_dict (articles[i].last,
		  pdf_new_name ("N"),
		  pdf_ref_obj (articles[i].first));
    pdf_add_dict (articles[i].first,
		  pdf_new_name ("V"),
		  pdf_ref_obj (articles[i].last));
    pdf_release_obj (articles[i].first);
    pdf_release_obj (articles[i].last);
    pdf_release_obj (articles[i].info);
    RELEASE (articles[i].name);
  }
}


static void finish_last_page ()
{
#ifdef MEM_DEBUG
MEM_START
#endif  
  if (debug) {
    fprintf (stderr, "(finish_last_page)");
  }
  finish_pending_xobjects();
  /* Flush this page */
  /* Page_count is the index of the current page, starting at 1 */
  if (page_count > 0) {
    /* Write out MediaBox as the very last thing we do on the
       previous page */
    tmp1 = pdf_new_array ();
    pdf_add_array (tmp1, pdf_new_number (0));
    pdf_add_array (tmp1, pdf_new_number (0));
    pdf_add_array (tmp1, pdf_new_number
		   (ROUND(dev_page_width(),1.0)));
    pdf_add_array (tmp1, pdf_new_number (ROUND(dev_page_height(),1.0)));
    pdf_add_dict (pages[page_count-1].page_dict,
		  pdf_link_obj (mediabox_name), tmp1);
    /* We keep .page_dict open because we don't know the parent yet */
  }
  if (debug) {
    fprintf (stderr, "(finish_last_page)");
  }
  if (this_page_bop != NULL) {
    pdf_release_obj (this_page_bop);
    this_page_bop = NULL;
  }
  if (this_page_eop != NULL) {
    pdf_release_obj (this_page_eop);
    this_page_eop = NULL;
  }
  if (this_page_contents != NULL) {
    pdf_release_obj (this_page_contents);
    this_page_contents = NULL;
  }
  if (current_page_resources != NULL) {
    pdf_release_obj (current_page_resources);
    current_page_resources = NULL;
  }
  if (this_page_annots != NULL) {
    pdf_release_obj (this_page_annots);
    this_page_annots = NULL;
  }
  if (this_page_beads != NULL) {
    pdf_release_obj (this_page_beads);
    this_page_beads = NULL;
  }
  if (this_page_fonts != NULL) {
    pdf_release_obj (this_page_fonts);
    this_page_fonts = NULL;
  }
  if (this_page_xobjects != NULL) {
    pdf_release_obj (this_page_xobjects);
    this_page_xobjects = NULL;
  }
#ifdef MEM_DEBUG
MEM_END
#endif  
}

pdf_obj *pdf_doc_current_page_resources (void)
{
  return current_page_resources;
}


static int highest_page_ref = 0;
pdf_obj *pdf_doc_ref_page (unsigned long page_no)
{
  if (debug)
    fprintf (stderr, "(doc_ref_page:page_no=%ld)", page_no);
  if (page_no >= max_pages) {
    resize_pages (page_no+PAGES_ALLOC_SIZE);
  }
  /* Has this page been referenced yet? */ 
  if (pages[page_no-1].page_dict == NULL) {
    /* If not, create it */
    pages[page_no-1].page_dict = pdf_new_dict ();
    /* and reference it */
    pages[page_no-1].page_ref = pdf_ref_obj (pages[page_no-1].page_dict);
  }
  if (page_no > highest_page_ref)
    highest_page_ref = page_no;
  return pdf_link_obj (pages[page_no-1].page_ref);
}

pdf_obj *pdf_doc_names (void)
{
  return names_dict;
}

pdf_obj *pdf_doc_page_tree (void)
{
  return page_tree;
}

pdf_obj *pdf_doc_catalog (void)
{
  return catalog;
}

pdf_obj *pdf_doc_this_page (void)
{
  if (page_count <= 0) {
    ERROR ("Reference to current page, but no pages have been started yet");
  }
  return pages[page_count-1].page_dict;
}

pdf_obj *pdf_doc_this_page_ref (void)
{
  if (page_count <= 0) {
    ERROR ("Reference to current page, but no pages have been started yet");
  }
  return pdf_doc_ref_page(page_count);
}

pdf_obj *pdf_doc_prev_page_ref (void)
{
  if (page_count <= 0) {
    ERROR ("Reference to previous page, but no pages have been started yet");
  }
  return pdf_doc_ref_page(page_count>1?page_count-1:1);
}

pdf_obj *pdf_doc_next_page_ref (void)
{
  if (page_count <= 0) {
    ERROR ("Reference to previous page, but no pages have been started yet");
  }
  return pdf_doc_ref_page(page_count+1);
}

void pdf_doc_new_page (void)
{
#ifdef MEM_DEBUG
MEM_START
#endif
  if (debug) {
    fprintf (stderr, "(pdf_doc_new_page)");
    fprintf (stderr, "page_count=%ld, max_pages=%ld\n", page_count,
	     max_pages);
  }
  /* See if we need more pages allocated yet */
  if (page_count >= max_pages) {
    resize_pages(max_pages+PAGES_ALLOC_SIZE);
  }
  if (this_page_contents != NULL) {
    finish_last_page();
  }
  this_page_bop = pdf_new_stream(STREAM_COMPRESS);
  this_page_eop = pdf_new_stream(STREAM_COMPRESS);
  /* Was this page already instantiated by a forward reference to it? */
  if (pages[page_count].page_ref == NULL) {
    /* If not, create it. */
    pages[page_count].page_dict = pdf_new_dict ();
    /* and reference it */
    pages[page_count].page_ref = pdf_ref_obj(pages[page_count].page_dict);
  }
  pdf_add_dict (pages[page_count].page_dict,
		pdf_link_obj(type_name), pdf_link_obj(page_name));
  tmp1 = pdf_new_array ();
  pdf_add_array (tmp1, pdf_ref_obj (glob_page_bop));
  pdf_add_array (tmp1, pdf_ref_obj (this_page_bop));
  /* start the contents stream for the new page */
  this_page_contents = pdf_new_stream(STREAM_COMPRESS);
  pdf_add_array (tmp1, pdf_ref_obj (this_page_contents));
  pdf_add_array (tmp1, pdf_ref_obj (this_page_eop));
  pdf_add_array (tmp1, pdf_ref_obj (glob_page_eop));
  pdf_add_dict (pages[page_count].page_dict,
		pdf_link_obj(contents_name), tmp1);
  this_page_annots = pdf_new_array ();
  pdf_add_dict (pages[page_count].page_dict,
		pdf_link_obj(annots_name),
		pdf_ref_obj (this_page_annots));
  start_current_page_resources();
  pdf_add_dict (pages[page_count].page_dict,
		pdf_link_obj (resources_name),
		pdf_ref_obj (current_page_resources));
  this_page_beads = pdf_new_array();
  pdf_add_dict (pages[page_count].page_dict,
		pdf_link_obj (bead_name),
		pdf_ref_obj (this_page_beads));
  /* Contents are still available as this_page_contents until next
     page is started */
  /* Even though the page is gone, a Reference to this page is kept
     until program ends */
  page_count += 1;
#ifdef MEM_DEBUG
MEM_END
#endif
}

void pdf_doc_add_to_page (char *buffer, unsigned length)
{
  pdf_add_stream (this_page_contents, buffer, length);
}

void pdf_doc_init (char *filename) 
{
  if (debug) fprintf (stderr, "pdf_doc_init:\n");
  pdf_out_init (filename);
  make_short_cuts();
  create_docinfo ();
  create_catalog ();
}

void pdf_doc_creator (char *s)
{
  pdf_add_dict (docinfo, pdf_new_name ("Creator"),
		pdf_new_string (s, strlen(s)));
}

void pdf_doc_close ()
{
  if (debug) fprintf (stderr, "pdf_doc_finish:\n");
  if (this_page_contents != NULL) {
    finish_last_page();
  }
  /* Following things were kept around so user can add dictionary
     items */
  finish_docinfo();
  finish_page_tree();
  pdf_release_obj (glob_page_bop);
  pdf_release_obj (glob_page_eop);
  /* Add names dict to catalog */
  finish_outline();
  finish_dests_tree();
  finish_articles();
  pdf_add_dict (catalog,
		pdf_new_name ("Names"),
		pdf_ref_obj (names_dict));
  pdf_release_obj (names_dict);
  pdf_release_obj (catalog);
  /* Do consistency check on forward references to pages */
  if (highest_page_ref > page_count) {
    fprintf (stderr, "\nWarning:  Nonexistent page(s) referenced\n");
    fprintf (stderr, "          (PDF file may not work right)\n");
  }
  pdf_finish_specials();
  release_short_cuts();
  pdf_out_flush ();
}

static pdf_obj *build_scale_array (int a, int b, int c, int d, int e, int f)
{
  pdf_obj *result;
  result = pdf_new_array();
  pdf_add_array (result, pdf_new_number (a));
  pdf_add_array (result, pdf_new_number (b));
  pdf_add_array (result, pdf_new_number (c));
  pdf_add_array (result, pdf_new_number (d));
  pdf_add_array (result, pdf_new_number (e));
  pdf_add_array (result, pdf_new_number (f));
  return result;
}

/* All this routine does is give the form a name
   and add a unity scaling matrix. It fills
   in required fields.  The caller must initialize
   the stream */
int num_xobjects = 0;
void doc_make_form_xobj (pdf_obj *this_form_contents, pdf_obj *bbox,
			    pdf_obj *resources)
{
  pdf_obj *xobj_dict, *tmp1;
  xobj_dict = pdf_stream_dict (this_form_contents);
  num_xobjects += 1;
  sprintf (work_buffer, "Fm%d", num_xobjects);
  pdf_add_dict (xobj_dict, pdf_new_name ("Name"),
		pdf_new_name (work_buffer));
  pdf_add_dict (xobj_dict, pdf_link_obj (type_name),
		pdf_new_name ("XObject"));
  pdf_add_dict (xobj_dict, pdf_new_name ("Subtype"),
		pdf_new_name ("Form"));
  pdf_add_dict (xobj_dict, pdf_new_name ("BBox"), bbox);
  pdf_add_dict (xobj_dict, pdf_new_name ("FormType"), 
		pdf_new_number(1.0));
  tmp1 = build_scale_array (1, 0, 0, 1, 0, 0);
  pdf_add_dict (xobj_dict, pdf_new_name ("Matrix"), tmp1);
  pdf_add_dict (xobj_dict, pdf_link_obj (resources_name), resources);
  return;
}

static pdf_obj *save_page_contents, *save_page_fonts;
static pdf_obj *save_page_xobjects, *save_page_resources;
static xobject_pending = 0;

pdf_obj *begin_form_xobj (double bbllx, double bblly, double bburx,
			  double bbury)
{
  pdf_obj *bbox;
  if (xobject_pending) {
    fprintf (stderr, "\nCannot next form XObjects\n");
    return NULL;
  }
  dev_close_all_xforms();
  xobject_pending = 1;
  /* This is a hack.  We basically treat an xobj as a separate mini
     page unto itself.  Save all the page structures and reinitialize them. */
  save_page_resources = current_page_resources;
  save_page_xobjects = this_page_xobjects;
  save_page_fonts = this_page_fonts;
  save_page_contents = this_page_contents;
  start_current_page_resources(); /* Starts current_page_resources,
				     this_page_xobjects, and this_page_fonts */
  this_page_contents = pdf_new_stream (STREAM_COMPRESS);
  /* Make a bounding box for this Xobject */
  bbox = pdf_new_array ();
  pdf_add_array (bbox, pdf_new_number (0.0));
  pdf_add_array (bbox, pdf_new_number (0.0));
  pdf_add_array (bbox, pdf_new_number (ROUND(bburx-bbllx,0.1)));
  pdf_add_array (bbox, pdf_new_number (ROUND(bbury-bblly,0.1)));
  /* Resource is already made, so call doc_make_form_xobj() */
  sprintf (work_buffer, "1 0 0 1 %g %g cm",
	   ROUND(-bbllx,0.1), ROUND(-bblly,0.1));
  pdf_doc_add_to_page (work_buffer, strlen(work_buffer));
  doc_make_form_xobj (this_page_contents, bbox,
		      pdf_ref_obj(current_page_resources));
  /* Make sure the object is self-contained by adding the
     current font to the object stream */
  dev_reselect_font();
  /* Likewise for color */
  dev_do_color();
  return pdf_link_obj (this_page_contents);
}

void end_form_xobj (void)
{
  if (xobject_pending) {
    xobject_pending = 0;
    dev_close_all_xforms();
    pdf_release_obj (current_page_resources);
    pdf_release_obj (this_page_xobjects);
    pdf_release_obj (this_page_fonts);
    pdf_release_obj (this_page_contents);
    current_page_resources = save_page_resources;
    this_page_xobjects = save_page_xobjects;
    this_page_fonts = save_page_fonts;
    this_page_contents = save_page_contents;
    /* Must reselect the font again in case there was a font change in
       the object */
    dev_reselect_font();
    /* Must reselect color too */
    dev_do_color();
  } else{
    fprintf (stderr, "\nSpecial: exobj: Tried to close a nonexistent xobject\n");
  }
  return;
}

void finish_pending_xobjects (void)
{
  if (xobject_pending) {
    fprintf (stderr, "\nFinishing a pending form XObject at end of page\n");
    end_form_xobj();
  }
}
