/**
 ********************************************************************************
 * Copyright (c) 2020 Robert Bosch GmbH and others.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 ********************************************************************************
 */
package org.eclipse.app4mc.amalthea.model.io;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;

import javax.xml.XMLConstants;
import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.events.StartElement;
import javax.xml.stream.events.XMLEvent;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class AmaltheaFileHelper {
	
	private static final Logger LOGGER = LoggerFactory.getLogger(AmaltheaFileHelper.class);

	public static final String INVALID = "invalid";
	
	// Suppress default constructor
	private AmaltheaFileHelper() {
		throw new IllegalStateException("Utility class");
	}

	/**
	 * Check if the given file is a zip archive.
	 * 
	 * @param file The {@link File} to check.
	 * @return <code>true</code> if the given file is a zip archive,
	 *         <code>false</code> if not.
	 */
	public static boolean isZipFile(File file) {
		boolean result = false;

		if (file != null) {
			try (ZipFile f = new ZipFile(file)) {
				// zipped file detected
				result = true;
			} catch (IOException e) {
				// IOException includes ZipException -> not a zip file
			}
		}

		return result;
	}

	/**
	 * Extract the model namespace of the given input file.
	 * 
	 * @param inputFile The file for which the model namespace is
	 *                       requested.
	 * @return The model namespace of the given input file or <i>invalid</i> if
	 *         an error occurs on parsing or the model namespace is invalid.
	 */
	public static String getNamespaceFromModel(File inputFile) {
		String result = INVALID;
		
		if (inputFile != null && inputFile.exists() && inputFile.getName().endsWith(".amxmi")) {
			Path inputPath = Paths.get(inputFile.toURI());
			if (isZipFile(inputFile)) {
				return getNamespaceFromZip(inputPath);
			} else {
				return getNamespaceFromFile(inputPath);
			}
		}
		
		return result;
	}
	
	/**
	 * Extract the model namespace of the given input file.
	 * 
	 * @param inputPath The {@link Path} of the model file for which the model
	 *                  namespace is requested.
	 * @return The model namespace of the given input file or <i>invalid</i> if an
	 *         error occurs on parsing or the model namespace is invalid.
	 */
	public static String getNamespaceFromZip(Path inputPath) {
		try (ZipInputStream input = new ZipInputStream(Files.newInputStream(inputPath))) {
			ZipEntry zipEntry = input.getNextEntry();
			if (zipEntry != null) {
				return getNamespace(input);
			}
		} catch (IOException | XMLStreamException e) {
			LOGGER.error("Error on parsing input model file for namespace", e);
		}
		
		return INVALID;
	}

	/**
	 * Extract the model namespace of the given input file.
	 * 
	 * @param inputPath The {@link Path} of the model file for which the model
	 *                  namespace is requested.
	 * @return The model namespace of the given input file or <i>invalid</i> if an
	 *         error occurs on parsing or the model namespace is invalid.
	 */
	public static String getNamespaceFromFile(Path inputPath) {
		try (InputStream input = Files.newInputStream(inputPath)) {
			return getNamespace(input);
		} catch (IOException | XMLStreamException e) {
			LOGGER.error("Error on parsing input model file for namespace", e);
		}
		
		return INVALID;
	}

	
	private static String getNamespace(InputStream input) throws XMLStreamException {
		XMLInputFactory xmlInputFactory = XMLInputFactory.newInstance();
		xmlInputFactory.setProperty(XMLConstants.ACCESS_EXTERNAL_DTD, "");
		xmlInputFactory.setProperty(XMLConstants.ACCESS_EXTERNAL_SCHEMA, "");
		
		XMLEventReader reader = xmlInputFactory.createXMLEventReader(input);
		while (reader.hasNext()) {
			XMLEvent nextEvent = reader.nextEvent();
			if (nextEvent.isStartElement()) {
				StartElement startElement = nextEvent.asStartElement();
				if ("am".equals(startElement.getName().getPrefix())) {
					return startElement.getNamespaceURI("am");
				}
			}
		}
		
		return INVALID;
	}
}
