/**
 * Core definitions of the Eclipse Layout Kernel.
 */
package org.eclipse.elk.core.options;

import org.eclipse.elk.core.data.ILayoutMetaDataProvider;
import org.eclipse.elk.core.math.ElkPadding;
import org.eclipse.elk.core.options.CoreOptions;
import org.eclipse.elk.core.util.RandomLayoutProvider;
import org.eclipse.elk.graph.properties.IProperty;
import org.eclipse.elk.graph.properties.Property;

@SuppressWarnings("all")
public class RandomLayouterOptions implements ILayoutMetaDataProvider {
  /**
   * The id of the Randomizer algorithm.
   */
  public final static String ALGORITHM_ID = "org.eclipse.elk.random";
  
  /**
   * Default value for {@link #PADDING} with algorithm "Randomizer".
   */
  private final static ElkPadding PADDING_DEFAULT = new ElkPadding(15);
  
  /**
   * The padding to be left to a parent element's border when placing child elements. This can
   * also serve as an output option of a layout algorithm if node size calculation is setup
   * appropriately.
   */
  public final static IProperty<ElkPadding> PADDING = new Property<ElkPadding>(
                                CoreOptions.PADDING,
                                PADDING_DEFAULT);
  
  /**
   * Default value for {@link #SPACING_NODE_NODE} with algorithm "Randomizer".
   */
  private final static double SPACING_NODE_NODE_DEFAULT = 15;
  
  /**
   * The minimal distance to be preserved between each two nodes.
   */
  public final static IProperty<Double> SPACING_NODE_NODE = new Property<Double>(
                                CoreOptions.SPACING_NODE_NODE,
                                SPACING_NODE_NODE_DEFAULT);
  
  /**
   * Default value for {@link #RANDOM_SEED} with algorithm "Randomizer".
   */
  private final static int RANDOM_SEED_DEFAULT = 0;
  
  /**
   * Seed used for pseudo-random number generators to control the layout algorithm. If the
   * value is 0, the seed shall be determined pseudo-randomly (e.g. from the system time).
   */
  public final static IProperty<Integer> RANDOM_SEED = new Property<Integer>(
                                CoreOptions.RANDOM_SEED,
                                RANDOM_SEED_DEFAULT);
  
  /**
   * Default value for {@link #ASPECT_RATIO} with algorithm "Randomizer".
   */
  private final static double ASPECT_RATIO_DEFAULT = 1.6f;
  
  /**
   * The desired aspect ratio of the drawing, that is the quotient of width by height.
   */
  public final static IProperty<Double> ASPECT_RATIO = new Property<Double>(
                                CoreOptions.ASPECT_RATIO,
                                ASPECT_RATIO_DEFAULT);
  
  public void apply(final org.eclipse.elk.core.data.ILayoutMetaDataProvider.Registry registry) {
    registry.register(new org.eclipse.elk.core.data.LayoutAlgorithmData(
        "org.eclipse.elk.random",
        "Randomizer",
        "Distributes the nodes randomly on the plane, leading to very obfuscating layouts. Can be useful to demonstrate the power of \"real\" layout algorithms.",
        new org.eclipse.elk.core.util.AlgorithmFactory(RandomLayoutProvider.class, ""),
        null,
        "ELK",
        "images/random.png",
        null
    ));
    registry.addOptionSupport(
        "org.eclipse.elk.random",
        "org.eclipse.elk.padding",
        PADDING_DEFAULT
    );
    registry.addOptionSupport(
        "org.eclipse.elk.random",
        "org.eclipse.elk.spacing.nodeNode",
        SPACING_NODE_NODE_DEFAULT
    );
    registry.addOptionSupport(
        "org.eclipse.elk.random",
        "org.eclipse.elk.randomSeed",
        RANDOM_SEED_DEFAULT
    );
    registry.addOptionSupport(
        "org.eclipse.elk.random",
        "org.eclipse.elk.aspectRatio",
        ASPECT_RATIO_DEFAULT
    );
  }
}
