//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2026 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.relations.options;

import static org.eclipse.escet.common.java.Strings.makeInitialUppercase;

import java.util.List;
import java.util.Locale;

import org.eclipse.escet.common.app.framework.Paths;
import org.eclipse.escet.common.app.framework.options.Option;
import org.eclipse.escet.common.app.framework.options.Options;
import org.eclipse.escet.common.app.framework.options.StringOption;

/** Base class for options to write a DMM CSV file to the file system. */
public class BaseDmmOutputFileOption extends StringOption {
    /**
     * Constructor of the {@link BaseDmmOutputFileOption} class.
     *
     * @param dmmKind The kind of DMM to write.
     * @param defaultValue Default value of the option. {@code null} means using the name of the CIF input file,
     *     {@code "no"} means not writing the file, and everything else is the default path.
     */
    public BaseDmmOutputFileOption(String dmmKind, String defaultValue) {
        super(makeName(dmmKind), makeExplanation(dmmKind) + " " + makeDefaultText(defaultValue),
                null, makeLongCommand(dmmKind), "PATH", defaultValue, true, true, makeExplanation(dmmKind),
                "DMM file path:");
    }

    /**
     * Construct the name of the option.
     *
     * @param dmmKind The kind of the DMM file.
     * @return The name of the option.
     */
    private static String makeName(String dmmKind) {
        return makeInitialUppercase(dmmKind) + " DMM output file path";
    }

    /**
     * Construct the explanation of the option.
     *
     * @param dmmKind The kind of the DMM file.
     * @return The explanation of the option.
     */
    private static String makeExplanation(String dmmKind) {
        return "The path for writing the " + dmmKind + " DMM file. "
                + "If the option value is left empty, a path is constructed from the path of the input CIF file, "
                + "by first removing either a \".cif\" or a \".cifx\" suffix if it exists, and appending "
                + "\"." + dmmKind + ".csv\". "
                + "Use the text \"no\" to skip writing the file.";
    }

    /**
     * Construct the explanation of the default behavior of the option.
     *
     * @param defaultValue The default value of the option.
     * @return The explanation of the default behavior of the option.
     */
    private static String makeDefaultText(String defaultValue) {
        if (defaultValue == null) {
            return "The default behavior is to write the DMM file to a path derived from the CIF input file.";
        } else if (skipWrite(defaultValue)) {
            return "The default behavior is to skip writing the DMM file.";
        } else {
            return "The default behavior is to write the DMM file to the \"" + defaultValue + "\" path.";
        }
    }

    /**
     * Construct the long option name of the option.
     *
     * @param dmmKind The kind of the DMM file.
     * @return The long command name of the option.
     */
    private static String makeLongCommand(String dmmKind) {
        return "write-" + dmmKind;
    }

    /**
     * Get the absolute or relative local file system path to write the DMM file.
     *
     * @param optionClass Class of the option.
     * @param userInputPath Input path of the CIF file as given by the user.
     * @param dmmKind Kind of DMM to write.
     * @return If not {@code null}, the path to write the DMM file.
     */
    protected static String getFilePath(Class<? extends Option<String>> optionClass, String userInputPath,
            String dmmKind)
    {
        String optionValue = Options.get(optionClass);
        if (optionValue == null) {
            return Paths.pathChangeSuffix(userInputPath, List.of(".cif", ".cifx"), "." + dmmKind + ".csv");
        }
        if (skipWrite(optionValue)) {
            return null;
        }
        return optionValue;
    }

    /**
     * Decide whether the value of the option expresses to skip writing the DMM file.
     *
     * @param optValue Value of the option.
     * @return Whether the given value means not to write the file.
     */
    private static boolean skipWrite(String optValue) {
        return optValue.toLowerCase(Locale.US).equals("no");
    }
}
