/**
 * Copyright (c) 2016, 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package org.eclipse.gemoc.opsemanticsview.gen.k3;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import opsemanticsview.OperationalSemanticsView;
import opsemanticsview.OpsemanticsviewFactory;
import opsemanticsview.Rule;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EOperation;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EcoreFactory;
import org.eclipse.gemoc.commons.eclipse.jdt.CallHierarchyHelper;
import org.eclipse.jdt.core.IAnnotation;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IMember;
import org.eclipse.jdt.core.IMemberValuePair;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.ITypeHierarchy;
import org.eclipse.jdt.internal.corext.callhierarchy.CallLocation;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function0;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.Functions.Function2;
import org.eclipse.xtext.xbase.lib.InputOutput;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;

@SuppressWarnings("all")
public class K3StepExtractor {
  private final Set<IType> allClasses;
  
  private final EPackage extendedMetamodel;
  
  private final OperationalSemanticsView ecoreExtension;
  
  private final Map<IType, EClass> stepAspectsClassToAspectedClasses = new HashMap<IType, EClass>();
  
  private final Set<IMethod> allMethods = new HashSet<IMethod>();
  
  private final Set<IMethod> allk3Methods = new HashSet<IMethod>();
  
  private final Set<IMethod> allSuperMethods = new HashSet<IMethod>();
  
  private final Set<IMethod> stepFunctions = new HashSet<IMethod>();
  
  private final Set<IMethod> eventFunctions = new HashSet<IMethod>();
  
  private final Map<IMethod, Rule> functionToRule = new HashMap<IMethod, Rule>();
  
  private final Set<IType> inspectedClasses = new HashSet<IType>();
  
  private final Map<IMethod, IMethod> methodToK3Method = new HashMap<IMethod, IMethod>();
  
  private final Map<IMethod, IMethod> k3MethodToMethod = new HashMap<IMethod, IMethod>();
  
  private final Map<IMethod, IMethod> superMethodTok3Method = new HashMap<IMethod, IMethod>();
  
  private final Map<IMethod, Set<IMethod>> k3MethodToCalledMethods = new HashMap<IMethod, Set<IMethod>>();
  
  private final Map<IMethod, Set<IMethod>> methodToOverridingMethods = new HashMap<IMethod, Set<IMethod>>();
  
  private final Map<IMethod, Set<IMethod>> callGraph = new HashMap<IMethod, Set<IMethod>>();
  
  private final Map<IType, Set<IType>> classToSubClasses = new HashMap<IType, Set<IType>>();
  
  private final Map<IType, Set<IType>> classToSuperClasses = new HashMap<IType, Set<IType>>();
  
  public K3StepExtractor(final Set<IType> aspects, final String languageName, final EPackage extendedMetamodel, final OperationalSemanticsView inConstructionOperationalSemanticsView) {
    this.allClasses = aspects;
    this.extendedMetamodel = extendedMetamodel;
    this.ecoreExtension = inConstructionOperationalSemanticsView;
  }
  
  public void generate() {
    this.generateStepFromXtend(this.allClasses);
  }
  
  private Rule getRuleOfFunction(final IMethod function) {
    boolean _containsKey = this.functionToRule.containsKey(function);
    if (_containsKey) {
      return this.functionToRule.get(function);
    } else {
      final Rule rule = OpsemanticsviewFactory.eINSTANCE.createRule();
      EList<Rule> _rules = this.ecoreExtension.getRules();
      _rules.add(rule);
      final IType containingClass = function.getDeclaringType();
      EClass _get = this.stepAspectsClassToAspectedClasses.get(containingClass);
      rule.setContainingClass(_get);
      EOperation candidate = null;
      EClass _containingClass = rule.getContainingClass();
      boolean _notEquals = (!Objects.equal(_containingClass, null));
      if (_notEquals) {
        EClass _containingClass_1 = rule.getContainingClass();
        EList<EOperation> _eAllOperations = _containingClass_1.getEAllOperations();
        final Function1<EOperation, Boolean> _function = new Function1<EOperation, Boolean>() {
          @Override
          public Boolean apply(final EOperation o) {
            String _name = o.getName();
            String _elementName = function.getElementName();
            return Boolean.valueOf(_name.equals(_elementName));
          }
        };
        EOperation _findFirst = IterableExtensions.<EOperation>findFirst(_eAllOperations, _function);
        candidate = _findFirst;
      }
      boolean _notEquals_1 = (!Objects.equal(candidate, null));
      if (_notEquals_1) {
        rule.setOperation(candidate);
      } else {
        EOperation _xtendFunctionToEOperation = this.xtendFunctionToEOperation(function);
        rule.setOperation(_xtendFunctionToEOperation);
      }
      boolean _contains = this.stepFunctions.contains(function);
      rule.setStepRule(_contains);
      boolean _isMain = this.isMain(function);
      rule.setMain(_isMain);
      this.functionToRule.put(function, rule);
      return rule;
    }
  }
  
  private void inspectForBigStep(final IMethod function) {
    try {
      final Rule rule = this.getRuleOfFunction(function);
      final Set<IMethod> calledFunctions = this.callGraph.get(function);
      boolean _notEquals = (!Objects.equal(calledFunctions, null));
      if (_notEquals) {
        for (final IMethod calledFunction : calledFunctions) {
          if ((calledFunction != null)) {
            final Rule calledRule = this.getRuleOfFunction(calledFunction);
            EList<Rule> _calledRules = rule.getCalledRules();
            _calledRules.add(calledRule);
          }
        }
      }
      IType _declaringType = function.getDeclaringType();
      final Set<IType> subtypes = this.classToSubClasses.get(_declaringType);
      boolean _notEquals_1 = (!Objects.equal(subtypes, null));
      if (_notEquals_1) {
        for (final IType t : subtypes) {
          IMethod[] _methods = t.getMethods();
          for (final IMethod f : _methods) {
            String _elementName = f.getElementName();
            String _elementName_1 = function.getElementName();
            boolean _equals = _elementName.equals(_elementName_1);
            if (_equals) {
              final Rule overridingRule = this.getRuleOfFunction(f);
              EList<Rule> _overridenBy = rule.getOverridenBy();
              _overridenBy.add(overridingRule);
            }
          }
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private EOperation xtendFunctionToEOperation(final IMethod function) {
    final EOperation result = EcoreFactory.eINSTANCE.createEOperation();
    String _elementName = function.getElementName();
    result.setName(_elementName);
    return result;
  }
  
  private void inspectClass(final IType type) {
    try {
      boolean _contains = this.inspectedClasses.contains(type);
      boolean _not = (!_contains);
      if (_not) {
        IMethod[] _methods = type.getMethods();
        final Function1<IMethod, Boolean> _function = new Function1<IMethod, Boolean>() {
          @Override
          public Boolean apply(final IMethod it) {
            String _elementName = it.getElementName();
            return Boolean.valueOf(_elementName.startsWith("_privk3_"));
          }
        };
        final Iterable<IMethod> typeK3Methods = IterableExtensions.<IMethod>filter(((Iterable<IMethod>)Conversions.doWrapArray(_methods)), _function);
        Iterables.<IMethod>addAll(this.allk3Methods, typeK3Methods);
        IMethod[] _methods_1 = type.getMethods();
        final Function1<IMethod, Boolean> _function_1 = new Function1<IMethod, Boolean>() {
          @Override
          public Boolean apply(final IMethod m) {
            final Function1<IMethod, Boolean> _function = new Function1<IMethod, Boolean>() {
              @Override
              public Boolean apply(final IMethod c) {
                String _elementName = c.getElementName();
                String _substring = _elementName.substring(8);
                String _elementName_1 = m.getElementName();
                return Boolean.valueOf(_substring.equals(_elementName_1));
              }
            };
            return Boolean.valueOf(IterableExtensions.<IMethod>exists(typeK3Methods, _function));
          }
        };
        final Iterable<IMethod> typeMethods = IterableExtensions.<IMethod>filter(((Iterable<IMethod>)Conversions.doWrapArray(_methods_1)), _function_1);
        Iterables.<IMethod>addAll(this.allMethods, typeMethods);
        final Consumer<IMethod> _function_2 = new Consumer<IMethod>() {
          @Override
          public void accept(final IMethod m) {
            final Function1<IMethod, Boolean> _function = new Function1<IMethod, Boolean>() {
              @Override
              public Boolean apply(final IMethod c) {
                String _elementName = c.getElementName();
                String _substring = _elementName.substring(8);
                String _elementName_1 = m.getElementName();
                return Boolean.valueOf(_substring.equals(_elementName_1));
              }
            };
            final IMethod k3m = IterableExtensions.<IMethod>findFirst(typeK3Methods, _function);
            K3StepExtractor.this.k3MethodToMethod.put(k3m, m);
            K3StepExtractor.this.methodToK3Method.put(m, k3m);
          }
        };
        typeMethods.forEach(_function_2);
        final Set<IMethod> candidateSupers = new HashSet<IMethod>();
        IMethod[] _methods_2 = type.getMethods();
        final Function1<IMethod, Boolean> _function_3 = new Function1<IMethod, Boolean>() {
          @Override
          public Boolean apply(final IMethod it) {
            String _elementName = it.getElementName();
            return Boolean.valueOf(_elementName.startsWith("super_"));
          }
        };
        Iterable<IMethod> _filter = IterableExtensions.<IMethod>filter(((Iterable<IMethod>)Conversions.doWrapArray(_methods_2)), _function_3);
        Iterables.<IMethod>addAll(candidateSupers, _filter);
        final Function1<IMethod, Boolean> _function_4 = new Function1<IMethod, Boolean>() {
          @Override
          public Boolean apply(final IMethod c) {
            try {
              IMethod[] _methods = type.getMethods();
              final Function1<IMethod, Boolean> _function = new Function1<IMethod, Boolean>() {
                @Override
                public Boolean apply(final IMethod m) {
                  String _elementName = c.getElementName();
                  String _substring = _elementName.substring(6);
                  String _elementName_1 = m.getElementName();
                  return Boolean.valueOf(_substring.equals(_elementName_1));
                }
              };
              return Boolean.valueOf(IterableExtensions.<IMethod>exists(((Iterable<IMethod>)Conversions.doWrapArray(_methods)), _function));
            } catch (Throwable _e) {
              throw Exceptions.sneakyThrow(_e);
            }
          }
        };
        Iterable<IMethod> _filter_1 = IterableExtensions.<IMethod>filter(candidateSupers, _function_4);
        Iterables.<IMethod>addAll(this.allSuperMethods, _filter_1);
        List<IAnnotation> _aspectAnnotations = this.getAspectAnnotations(type);
        for (final IAnnotation a : _aspectAnnotations) {
          {
            final EClass aspectedEClass = this.getAspectized(a);
            this.stepAspectsClassToAspectedClasses.put(type, aspectedEClass);
            IMethod[] _methods_3 = type.getMethods();
            final Function1<IMethod, Boolean> _function_5 = new Function1<IMethod, Boolean>() {
              @Override
              public Boolean apply(final IMethod it) {
                return Boolean.valueOf(K3StepExtractor.this.isStep(it));
              }
            };
            Iterable<IMethod> _filter_2 = IterableExtensions.<IMethod>filter(((Iterable<IMethod>)Conversions.doWrapArray(_methods_3)), _function_5);
            Iterables.<IMethod>addAll(this.stepFunctions, _filter_2);
            IMethod[] _methods_4 = type.getMethods();
            final Function1<IMethod, Boolean> _function_6 = new Function1<IMethod, Boolean>() {
              @Override
              public Boolean apply(final IMethod it) {
                return Boolean.valueOf(K3StepExtractor.this.isEvent(it));
              }
            };
            Iterable<IMethod> _filter_3 = IterableExtensions.<IMethod>filter(((Iterable<IMethod>)Conversions.doWrapArray(_methods_4)), _function_6);
            Iterables.<IMethod>addAll(this.eventFunctions, _filter_3);
          }
        }
        this.inspectedClasses.add(type);
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private void gatherCallsFromK3(final IMethod function) {
    final HashSet<CallLocation> callingSites = CallHierarchyHelper.getCallLocationsOf(function);
    final Consumer<CallLocation> _function = new Consumer<CallLocation>() {
      @Override
      public void accept(final CallLocation cl) {
        final IMethod f = function;
        final IMember member = cl.getMember();
        final IMethod method = K3StepExtractor.this.getContainingAspectMethod(((IMethod) member));
        final Function1<IMethod, Boolean> _function = new Function1<IMethod, Boolean>() {
          @Override
          public Boolean apply(final IMethod m) {
            return Boolean.valueOf(Objects.equal(m, method));
          }
        };
        Iterable<IMethod> _filter = IterableExtensions.<IMethod>filter(K3StepExtractor.this.allk3Methods, _function);
        final Consumer<IMethod> _function_1 = new Consumer<IMethod>() {
          @Override
          public void accept(final IMethod m) {
            Set<IMethod> calledMethods = K3StepExtractor.this.k3MethodToCalledMethods.get(m);
            boolean _equals = Objects.equal(calledMethods, null);
            if (_equals) {
              HashSet<IMethod> _hashSet = new HashSet<IMethod>();
              calledMethods = _hashSet;
              K3StepExtractor.this.k3MethodToCalledMethods.put(m, calledMethods);
            }
            calledMethods.add(f);
          }
        };
        _filter.forEach(_function_1);
      }
    };
    callingSites.forEach(_function);
  }
  
  private void gatherCallsFromSuper(final IMethod function) {
    final HashSet<CallLocation> callingSites = CallHierarchyHelper.getCallLocationsOf(function);
    final Consumer<CallLocation> _function = new Consumer<CallLocation>() {
      @Override
      public void accept(final CallLocation cl) {
        final Function1<IMethod, Boolean> _function = new Function1<IMethod, Boolean>() {
          @Override
          public Boolean apply(final IMethod m) {
            IMember _member = cl.getMember();
            return Boolean.valueOf(Objects.equal(m, _member));
          }
        };
        Iterable<IMethod> _filter = IterableExtensions.<IMethod>filter(K3StepExtractor.this.allSuperMethods, _function);
        final Consumer<IMethod> _function_1 = new Consumer<IMethod>() {
          @Override
          public void accept(final IMethod m) {
            K3StepExtractor.this.superMethodTok3Method.put(m, function);
          }
        };
        _filter.forEach(_function_1);
      }
    };
    callingSites.forEach(_function);
  }
  
  private void gatherOverridenMethods(final IMethod method) {
    boolean _isOverride = this.isOverride(method);
    if (_isOverride) {
      final String methodName = method.getElementName();
      final IType declaringType = method.getDeclaringType();
      final Set<IType> superClasses = this.classToSuperClasses.get(declaringType);
      boolean _notEquals = (!Objects.equal(superClasses, null));
      if (_notEquals) {
        final Consumer<IType> _function = new Consumer<IType>() {
          @Override
          public void accept(final IType c) {
            try {
              IMethod[] _methods = c.getMethods();
              final Function1<IMethod, Boolean> _function = new Function1<IMethod, Boolean>() {
                @Override
                public Boolean apply(final IMethod m) {
                  String _elementName = m.getElementName();
                  return Boolean.valueOf(_elementName.equals(methodName));
                }
              };
              final IMethod overridenMethod = IterableExtensions.<IMethod>findFirst(((Iterable<IMethod>)Conversions.doWrapArray(_methods)), _function);
              boolean _notEquals = (!Objects.equal(overridenMethod, null));
              if (_notEquals) {
                Set<IMethod> overridingMethods = K3StepExtractor.this.methodToOverridingMethods.get(overridenMethod);
                boolean _equals = Objects.equal(overridingMethods, null);
                if (_equals) {
                  HashSet<IMethod> _hashSet = new HashSet<IMethod>();
                  overridingMethods = _hashSet;
                  K3StepExtractor.this.methodToOverridingMethods.put(overridenMethod, overridingMethods);
                }
                overridingMethods.add(method);
              }
            } catch (Throwable _e) {
              throw Exceptions.sneakyThrow(_e);
            }
          }
        };
        superClasses.forEach(_function);
      }
    }
  }
  
  private void generateStepFromXtend(final Set<IType> files) {
    final Consumer<IType> _function = new Consumer<IType>() {
      @Override
      public void accept(final IType c) {
        Set<IType> _allSuperClasses = K3StepExtractor.this.getAllSuperClasses(c);
        final Function1<IType, Boolean> _function = new Function1<IType, Boolean>() {
          @Override
          public Boolean apply(final IType t) {
            return Boolean.valueOf(K3StepExtractor.this.allClasses.contains(t));
          }
        };
        Iterable<IType> _filter = IterableExtensions.<IType>filter(_allSuperClasses, _function);
        final Set<IType> allSuperClasses = IterableExtensions.<IType>toSet(_filter);
        K3StepExtractor.this.classToSuperClasses.put(c, allSuperClasses);
        Set<IType> _allSubClasses = K3StepExtractor.this.getAllSubClasses(c);
        final Function1<IType, Boolean> _function_1 = new Function1<IType, Boolean>() {
          @Override
          public Boolean apply(final IType t) {
            return Boolean.valueOf(K3StepExtractor.this.allClasses.contains(t));
          }
        };
        Iterable<IType> _filter_1 = IterableExtensions.<IType>filter(_allSubClasses, _function_1);
        final Set<IType> allSubClasses = IterableExtensions.<IType>toSet(_filter_1);
        K3StepExtractor.this.classToSubClasses.put(c, allSubClasses);
      }
    };
    this.allClasses.forEach(_function);
    for (final IType c : this.allClasses) {
      this.inspectClass(c);
    }
    final Consumer<IMethod> _function_1 = new Consumer<IMethod>() {
      @Override
      public void accept(final IMethod it) {
        K3StepExtractor.this.gatherCallsFromSuper(it);
      }
    };
    this.allk3Methods.forEach(_function_1);
    final Consumer<IMethod> _function_2 = new Consumer<IMethod>() {
      @Override
      public void accept(final IMethod it) {
        K3StepExtractor.this.gatherOverridenMethods(it);
      }
    };
    this.allMethods.forEach(_function_2);
    final Consumer<IMethod> _function_3 = new Consumer<IMethod>() {
      @Override
      public void accept(final IMethod it) {
        K3StepExtractor.this.gatherCallsFromK3(it);
      }
    };
    this.allMethods.forEach(_function_3);
    final Consumer<IMethod> _function_4 = new Consumer<IMethod>() {
      @Override
      public void accept(final IMethod it) {
        K3StepExtractor.this.gatherCallsFromK3(it);
      }
    };
    this.allSuperMethods.forEach(_function_4);
    final Consumer<IMethod> _function_5 = new Consumer<IMethod>() {
      @Override
      public void accept(final IMethod m) {
        final IMethod k3m = K3StepExtractor.this.methodToK3Method.get(m);
        boolean _notEquals = (!Objects.equal(k3m, null));
        if (_notEquals) {
          final Set<IMethod> calledMethods = K3StepExtractor.this.k3MethodToCalledMethods.get(k3m);
          boolean _notEquals_1 = (!Objects.equal(calledMethods, null));
          if (_notEquals_1) {
            final Consumer<IMethod> _function = new Consumer<IMethod>() {
              @Override
              public void accept(final IMethod c) {
                boolean _contains = K3StepExtractor.this.allMethods.contains(c);
                if (_contains) {
                  Set<IMethod> tmp = K3StepExtractor.this.callGraph.get(m);
                  boolean _equals = Objects.equal(tmp, null);
                  if (_equals) {
                    HashSet<IMethod> _hashSet = new HashSet<IMethod>();
                    tmp = _hashSet;
                    K3StepExtractor.this.callGraph.put(m, tmp);
                  }
                  tmp.add(c);
                }
              }
            };
            calledMethods.forEach(_function);
          }
        }
      }
    };
    this.allMethods.forEach(_function_5);
    final Function0<Integer> _function_6 = new Function0<Integer>() {
      @Override
      public Integer apply() {
        Collection<Set<IMethod>> _values = K3StepExtractor.this.callGraph.values();
        final Function1<Set<IMethod>, Integer> _function = new Function1<Set<IMethod>, Integer>() {
          @Override
          public Integer apply(final Set<IMethod> s) {
            return Integer.valueOf(s.size());
          }
        };
        Iterable<Integer> _map = IterableExtensions.<Set<IMethod>, Integer>map(_values, _function);
        final Function2<Integer, Integer, Integer> _function_1 = new Function2<Integer, Integer, Integer>() {
          @Override
          public Integer apply(final Integer i1, final Integer i2) {
            return Integer.valueOf(((i1).intValue() + (i2).intValue()));
          }
        };
        return IterableExtensions.<Integer>reduce(_map, _function_1);
      }
    };
    final Function0<Integer> callGraphTotalLengthComputer = _function_6;
    Integer totalLength = callGraphTotalLengthComputer.apply();
    int previousTotalLength = (-1);
    while (((totalLength).intValue() > previousTotalLength)) {
      {
        final Consumer<IMethod> _function_7 = new Consumer<IMethod>() {
          @Override
          public void accept(final IMethod m) {
            Set<IMethod> _xifexpression = null;
            Set<IMethod> _get = K3StepExtractor.this.callGraph.get(m);
            boolean _equals = Objects.equal(_get, null);
            if (_equals) {
              HashSet<IMethod> _xblockexpression = null;
              {
                final HashSet<IMethod> tmp = new HashSet<IMethod>();
                K3StepExtractor.this.callGraph.put(m, tmp);
                _xblockexpression = tmp;
              }
              _xifexpression = _xblockexpression;
            } else {
              _xifexpression = K3StepExtractor.this.callGraph.get(m);
            }
            final Set<IMethod> calledMethods = _xifexpression;
            final Set<IMethod> overridingMethods = K3StepExtractor.this.methodToOverridingMethods.get(m);
            boolean _notEquals = (!Objects.equal(overridingMethods, null));
            if (_notEquals) {
              final Consumer<IMethod> _function = new Consumer<IMethod>() {
                @Override
                public void accept(final IMethod n) {
                  final Set<IMethod> calledByOverride = K3StepExtractor.this.callGraph.get(n);
                  boolean _notEquals = (!Objects.equal(calledByOverride, null));
                  if (_notEquals) {
                    calledMethods.addAll(calledByOverride);
                  }
                }
              };
              overridingMethods.forEach(_function);
            }
          }
        };
        this.allMethods.forEach(_function_7);
        previousTotalLength = (totalLength).intValue();
        Integer _apply = callGraphTotalLengthComputer.apply();
        totalLength = _apply;
      }
    }
    Integer _apply = callGraphTotalLengthComputer.apply();
    totalLength = _apply;
    previousTotalLength = (-1);
    while (((totalLength).intValue() > previousTotalLength)) {
      {
        final Consumer<IMethod> _function_7 = new Consumer<IMethod>() {
          @Override
          public void accept(final IMethod m) {
            final Set<IMethod> calledMethods = K3StepExtractor.this.callGraph.get(m);
            boolean _notEquals = (!Objects.equal(calledMethods, null));
            if (_notEquals) {
              final HashSet<IMethod> tmp = new HashSet<IMethod>();
              final Consumer<IMethod> _function = new Consumer<IMethod>() {
                @Override
                public void accept(final IMethod n) {
                  final Set<IMethod> overridingMethods = K3StepExtractor.this.methodToOverridingMethods.get(n);
                  boolean _notEquals = (!Objects.equal(overridingMethods, null));
                  if (_notEquals) {
                    tmp.addAll(overridingMethods);
                  }
                }
              };
              calledMethods.forEach(_function);
              calledMethods.addAll(tmp);
            }
          }
        };
        this.allMethods.forEach(_function_7);
        previousTotalLength = (totalLength).intValue();
        Integer _apply_1 = callGraphTotalLengthComputer.apply();
        totalLength = _apply_1;
      }
    }
    final Consumer<IMethod> _function_7 = new Consumer<IMethod>() {
      @Override
      public void accept(final IMethod m) {
        Set<IMethod> calledMethods = K3StepExtractor.this.callGraph.get(m);
        boolean _equals = Objects.equal(calledMethods, null);
        if (_equals) {
          HashSet<IMethod> _hashSet = new HashSet<IMethod>();
          calledMethods = _hashSet;
          K3StepExtractor.this.callGraph.put(m, calledMethods);
        }
        calledMethods.addAll(K3StepExtractor.this.eventFunctions);
      }
    };
    this.allMethods.forEach(_function_7);
    final Consumer<IMethod> _function_8 = new Consumer<IMethod>() {
      @Override
      public void accept(final IMethod m) {
        final IMethod k3m = K3StepExtractor.this.methodToK3Method.get(m);
        boolean _notEquals = (!Objects.equal(k3m, null));
        if (_notEquals) {
          final Set<IMethod> calledMethods = K3StepExtractor.this.k3MethodToCalledMethods.get(k3m);
          boolean _notEquals_1 = (!Objects.equal(calledMethods, null));
          if (_notEquals_1) {
            final Consumer<IMethod> _function = new Consumer<IMethod>() {
              @Override
              public void accept(final IMethod c) {
                boolean _contains = K3StepExtractor.this.allSuperMethods.contains(c);
                if (_contains) {
                  final IMethod actualk3Method = K3StepExtractor.this.superMethodTok3Method.get(c);
                  boolean _notEquals = (!Objects.equal(actualk3Method, null));
                  if (_notEquals) {
                    final IMethod actualMethod = K3StepExtractor.this.k3MethodToMethod.get(actualk3Method);
                    if (((!Objects.equal(actualMethod, null)) && K3StepExtractor.this.allMethods.contains(actualMethod))) {
                      Set<IMethod> tmp = K3StepExtractor.this.callGraph.get(m);
                      boolean _equals = Objects.equal(tmp, null);
                      if (_equals) {
                        HashSet<IMethod> _hashSet = new HashSet<IMethod>();
                        tmp = _hashSet;
                        K3StepExtractor.this.callGraph.put(m, tmp);
                      }
                      tmp.add(actualMethod);
                    }
                  }
                }
              }
            };
            calledMethods.forEach(_function);
          }
        }
      }
    };
    this.allMethods.forEach(_function_8);
    InputOutput.<String>println("Callgraph : \n\n");
    final BiConsumer<IMethod, Set<IMethod>> _function_9 = new BiConsumer<IMethod, Set<IMethod>>() {
      @Override
      public void accept(final IMethod m, final Set<IMethod> s) {
        IType _declaringType = m.getDeclaringType();
        String _elementName = _declaringType.getElementName();
        String _plus = (_elementName + ".");
        String _elementName_1 = m.getElementName();
        String _plus_1 = (_plus + _elementName_1);
        String _plus_2 = (_plus_1 + " : \n");
        final Function1<IMethod, String> _function = new Function1<IMethod, String>() {
          @Override
          public String apply(final IMethod n) {
            IType _declaringType = n.getDeclaringType();
            String _elementName = _declaringType.getElementName();
            String _plus = (_elementName + ".");
            String _elementName_1 = n.getElementName();
            return (_plus + _elementName_1);
          }
        };
        Iterable<String> _map = IterableExtensions.<IMethod, String>map(s, _function);
        final Function2<String, String, String> _function_1 = new Function2<String, String, String>() {
          @Override
          public String apply(final String s1, final String s2) {
            return ((s1 + ", ") + s2);
          }
        };
        String _reduce = IterableExtensions.<String>reduce(_map, _function_1);
        String _plus_3 = (_plus_2 + _reduce);
        String _plus_4 = (_plus_3 + "\n");
        InputOutput.<String>println(_plus_4);
      }
    };
    this.callGraph.forEach(_function_9);
    for (final IMethod function : this.allMethods) {
      this.inspectForBigStep(function);
    }
  }
  
  /**
   * Find annotations "@Aspect"
   */
  private List<IAnnotation> getAspectAnnotations(final IType type) {
    try {
      boolean _isClass = type.isClass();
      if (_isClass) {
        IAnnotation[] _annotations = type.getAnnotations();
        final Function1<IAnnotation, Boolean> _function = new Function1<IAnnotation, Boolean>() {
          @Override
          public Boolean apply(final IAnnotation annot) {
            boolean _xblockexpression = false;
            {
              final String name = annot.getElementName();
              final int lastDotIndex = name.lastIndexOf(".");
              String simpleName = name;
              if ((lastDotIndex != (-1))) {
                String _substring = name.substring((lastDotIndex + 1));
                simpleName = _substring;
              }
              _xblockexpression = simpleName.equals("Aspect");
            }
            return Boolean.valueOf(_xblockexpression);
          }
        };
        Iterable<IAnnotation> _filter = IterableExtensions.<IAnnotation>filter(((Iterable<IAnnotation>)Conversions.doWrapArray(_annotations)), _function);
        return IterableExtensions.<IAnnotation>toList(_filter);
      }
      return new ArrayList<IAnnotation>();
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private boolean testAnnotation(final IMethod method, final String annotationSimpleName) {
    try {
      IAnnotation[] _annotations = method.getAnnotations();
      final Function1<IAnnotation, Boolean> _function = new Function1<IAnnotation, Boolean>() {
        @Override
        public Boolean apply(final IAnnotation annot) {
          final String name = annot.getElementName();
          final int lastDotIndex = name.lastIndexOf(".");
          String simpleName = name;
          if ((lastDotIndex != (-1))) {
            String _substring = name.substring((lastDotIndex + 1));
            simpleName = _substring;
          }
          return Boolean.valueOf(simpleName.equals(annotationSimpleName));
        }
      };
      return IterableExtensions.<IAnnotation>exists(((Iterable<IAnnotation>)Conversions.doWrapArray(_annotations)), _function);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * Return true if 'method' is tagged with "@Step"
   */
  private boolean isStep(final IMethod method) {
    return this.testAnnotation(method, "Step");
  }
  
  /**
   * Return true if 'method' is tagged with "@EventProcessor"
   */
  private boolean isEvent(final IMethod method) {
    try {
      boolean _xblockexpression = false;
      {
        IAnnotation[] _annotations = method.getAnnotations();
        final Function1<IAnnotation, Boolean> _function = new Function1<IAnnotation, Boolean>() {
          @Override
          public Boolean apply(final IAnnotation a) {
            final String name = a.getElementName();
            final int lastDotIndex = name.lastIndexOf(".");
            String simpleName = name;
            if ((lastDotIndex != (-1))) {
              String _substring = name.substring((lastDotIndex + 1));
              simpleName = _substring;
            }
            return Boolean.valueOf(Objects.equal(simpleName, "Step"));
          }
        };
        final IAnnotation annotation = IterableExtensions.<IAnnotation>findFirst(((Iterable<IAnnotation>)Conversions.doWrapArray(_annotations)), _function);
        _xblockexpression = ((!Objects.equal(annotation, null)) && IterableExtensions.<IMemberValuePair>exists(((Iterable<IMemberValuePair>)Conversions.doWrapArray(annotation.getMemberValuePairs())), new Function1<IMemberValuePair, Boolean>() {
          @Override
          public Boolean apply(final IMemberValuePair p) {
            return Boolean.valueOf(((Objects.equal(p.getMemberName(), "eventHandler") && (p.getValue() instanceof Boolean)) && (((Boolean) p.getValue())).booleanValue()));
          }
        }));
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * Return true if 'method' is tagged with "@OverrideAspectMethod"
   */
  private boolean isOverride(final IMethod method) {
    return this.testAnnotation(method, "OverrideAspectMethod");
  }
  
  /**
   * Return true if 'method' is tagged with "@Main"
   */
  private boolean isMain(final IMethod method) {
    return this.testAnnotation(method, "Main");
  }
  
  /**
   * Return all sub types
   */
  private Set<IType> getAllSubClasses(final IType type) {
    try {
      NullProgressMonitor _nullProgressMonitor = new NullProgressMonitor();
      final ITypeHierarchy hierarchy = type.newTypeHierarchy(_nullProgressMonitor);
      IType[] _allSubtypes = hierarchy.getAllSubtypes(type);
      return IterableExtensions.<IType>toSet(((Iterable<IType>)Conversions.doWrapArray(_allSubtypes)));
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * Return all super types
   */
  private Set<IType> getAllSuperClasses(final IType type) {
    try {
      NullProgressMonitor _nullProgressMonitor = new NullProgressMonitor();
      final ITypeHierarchy hierarchy = type.newTypeHierarchy(_nullProgressMonitor);
      IType[] _allSuperclasses = hierarchy.getAllSuperclasses(type);
      return IterableExtensions.<IType>toSet(((Iterable<IType>)Conversions.doWrapArray(_allSuperclasses)));
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private EClass getAspectized(final IAnnotation annot) {
    try {
      IMemberValuePair[] _memberValuePairs = annot.getMemberValuePairs();
      final Function1<IMemberValuePair, Boolean> _function = new Function1<IMemberValuePair, Boolean>() {
        @Override
        public Boolean apply(final IMemberValuePair p) {
          String _memberName = p.getMemberName();
          return Boolean.valueOf(Objects.equal(_memberName, "className"));
        }
      };
      IMemberValuePair _findFirst = IterableExtensions.<IMemberValuePair>findFirst(((Iterable<IMemberValuePair>)Conversions.doWrapArray(_memberValuePairs)), _function);
      Object _value = _findFirst.getValue();
      final String aspectedClassName = ((String) _value);
      TreeIterator<EObject> _eAllContents = this.extendedMetamodel.eAllContents();
      Iterator<EClass> _filter = Iterators.<EClass>filter(_eAllContents, EClass.class);
      final Function1<EClass, Boolean> _function_1 = new Function1<EClass, Boolean>() {
        @Override
        public Boolean apply(final EClass c1) {
          String _name = c1.getName();
          return Boolean.valueOf(aspectedClassName.equals(_name));
        }
      };
      return IteratorExtensions.<EClass>findFirst(_filter, _function_1);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * Return the top level method in a type tagged @aspect
   * that contains 'function'<br>
   * <br>
   * Return 'function' if it is already a top level method.<br>
   * <br>
   * Return null if not inside a type with @aspect
   */
  private IMethod getContainingAspectMethod(final IMethod function) {
    final IType container = function.getDeclaringType();
    boolean _contains = this.allClasses.contains(container);
    if (_contains) {
      return function;
    }
    IJavaElement parent = function.getParent();
    while ((parent != null)) {
      {
        if ((parent instanceof IMethod)) {
          return this.getContainingAspectMethod(((IMethod)parent));
        }
        IJavaElement _parent = parent.getParent();
        parent = _parent;
      }
    }
    return null;
  }
}
