/*
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.lsat.common.scheduler.graph.util;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.EMap;
import org.eclipse.lsat.common.scheduler.graph.TimeStepData;

public class ResampleTimeStepData {
    public static Map<String, List<List<BigDecimal>>> resample(EMap<String, EList<TimeStepData>> executionData,
            double newSampleFrequency)
    {
        var result = new LinkedHashMap<String, List<List<BigDecimal>>>();
        for (var entry: executionData.entrySet()) {
            result.put(entry.getKey(), resample(entry.getValue(), newSampleFrequency));
        }
        return result;
    }

    public static List<List<BigDecimal>> resample(EList<TimeStepData> data, double newSampleFrequency) {
        var numInputSamples = data.size();
        var tStep = BigDecimal.valueOf(1.0 / newSampleFrequency);
        var tZero = data.get(0).timeValue();
        var tStart = tZero;
        var tEnd = tStart.add(tStep);
        var index = 0;
        var sampleList = new ArrayList<List<BigDecimal>>();
        var result = new ArrayList<List<BigDecimal>>();

        while (index < numInputSamples) {
            var t = data.get(index).timeValue();
            if (t.compareTo(tEnd) < 0) {
                var d = data.get(index).getValues();
                sampleList.add(d);
                index++;
            }
            if (t.compareTo(tEnd) >= 0 || index == numInputSamples) {
                var numSamples = sampleList.size();
                if (numSamples > 0) {
                    var resampledValues = new ArrayList<BigDecimal>();
                    resampledValues.add(tStart);
                    var valueSize = sampleList.get(0).size();
                    for (var n = 1; n < valueSize; n++) {
                        BigDecimal dataValue = null;
                        if (tStart.equals(tZero)) {
                            // preserve first value in dataSeries
                            dataValue = sampleList.get(0).get(n);
                        } else if (index == numInputSamples) {
                            // preserve last value in dataSeries
                            dataValue = sampleList.get(numSamples - 1).get(n);
                        } else if ((numSamples % 2) == 0) {
                            // median filter with an even number of samples
                            var sortedSampleList = new ArrayList<>(sampleList);
                            sortedSampleList.sort(Comparator.comparing(l -> l.get(0)));
                            var v1 = sortedSampleList.get(numSamples / 2 - 1).get(n);
                            var v2 = sortedSampleList.get(numSamples / 2).get(n);
                            dataValue = v1.add(v2).divide(BigDecimal.TWO);
                        } else {
                            // median filter with an odd number of samples
                            var sortedSampleList = new ArrayList<>(sampleList);
                            sortedSampleList.sort(Comparator.comparing(l -> l.get(0)));
                            dataValue = sortedSampleList.get(numSamples / 2).get(n);
                        }
                        resampledValues.add(dataValue);
                    }
                    result.add(resampledValues);
                    sampleList.clear();
                }
                tStart = tEnd;
                tEnd = tStart.add(tStep);
            }
        }
        return result;
    }
}
