/*******************************************************************************
 * Copyright (c) 2009 Ecliptical Software Inc. and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Ecliptical Software Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.mint.internal.genmodel;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.ISafeRunnable;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.SafeRunner;
import org.eclipse.core.runtime.dynamichelpers.ExtensionTracker;
import org.eclipse.core.runtime.dynamichelpers.IExtensionChangeHandler;
import org.eclipse.core.runtime.dynamichelpers.IExtensionTracker;
import org.eclipse.core.runtime.dynamichelpers.IFilter;
import org.eclipse.emf.codegen.ecore.genmodel.GenModelPackage;
import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.common.notify.impl.NotificationImpl;
import org.eclipse.emf.edit.provider.ChangeNotifier;
import org.eclipse.emf.edit.provider.ComposeableAdapterFactory;
import org.eclipse.emf.edit.provider.IChangeNotifier;
import org.eclipse.emf.edit.provider.IDisposable;
import org.eclipse.emf.edit.provider.INotifyChangedListener;
import org.eclipse.emf.mint.IItemJavaElementDescriptor;
import org.eclipse.emf.mint.IItemJavaElementSource;
import org.eclipse.emf.mint.IItemJavaElementSourceContributor;
import org.eclipse.emf.mint.IJavaElementNotification;
import org.eclipse.emf.mint.MintCore;

public class GenModelItemProviderAdapter extends AdapterImpl implements
		IChangeNotifier, IDisposable, IItemJavaElementSource,
		IExtensionChangeHandler {

	protected final AdapterFactory adapterFactory;

	private final ExtensionTracker extensionTracker;

	private final HashSet<IItemJavaElementSourceContributor> contributors = new HashSet<IItemJavaElementSourceContributor>();

	private final HashSet<IExtension> newExtensions = new HashSet<IExtension>();

	protected IChangeNotifier changeNotifier;

	protected List<Notifier> targets;

	public GenModelItemProviderAdapter(AdapterFactory adapterFactory) {
		this.adapterFactory = adapterFactory;
		IExtensionRegistry registry = Platform.getExtensionRegistry();
		extensionTracker = new ExtensionTracker(registry);
		IExtensionPoint xpt = registry.getExtensionPoint(MintCore.PLUGIN_ID,
				"itemJavaElementSourceContributors");
		extensionTracker.registerHandler(this, getExtensionPointFilter(xpt));
		addExtensions(xpt);
	}

	private synchronized void addExtensions(IExtensionPoint xpt) {
		for (IExtension ext : xpt.getExtensions())
			newExtensions.add(ext);
	}

	private IFilter getExtensionPointFilter(IExtensionPoint xpt) {
		return ExtensionTracker.createExtensionPointFilter(xpt);
	}

	@Override
	public boolean isAdapterForType(Object type) {
		return type == adapterFactory;
	}

	public AdapterFactory getAdapterFactory() {
		return adapterFactory;
	}

	protected AdapterFactory getRootAdapterFactory() {
		if (adapterFactory instanceof ComposeableAdapterFactory)
			return ((ComposeableAdapterFactory) adapterFactory)
					.getRootAdapterFactory();

		return adapterFactory;
	}

	public void addListener(INotifyChangedListener listener) {
		if (changeNotifier == null)
			changeNotifier = new ChangeNotifier();

		changeNotifier.addListener(listener);
	}

	public void removeListener(INotifyChangedListener listener) {
		if (changeNotifier != null)
			changeNotifier.removeListener(listener);
	}

	public void fireNotifyChanged(Notification notification) {
		if (changeNotifier != null)
			changeNotifier.fireNotifyChanged(notification);

		if (adapterFactory instanceof IChangeNotifier) {
			IChangeNotifier changeNotifier = (IChangeNotifier) adapterFactory;
			changeNotifier.fireNotifyChanged(notification);
		}
	}

	public List<IItemJavaElementDescriptor> getJavaElementDescriptors(
			Object object) {
		LinkedList<IItemJavaElementDescriptor> descriptors = new LinkedList<IItemJavaElementDescriptor>();
		Collection<IItemJavaElementSource> sources = getSources(object);
		for (IItemJavaElementSource source : sources)
			descriptors.addAll(source.getJavaElementDescriptors(object));

		return descriptors;
	}

	protected Collection<IItemJavaElementSource> getSources(final Object object) {
		final LinkedList<IItemJavaElementSource> sources = new LinkedList<IItemJavaElementSource>();
		Collection<IItemJavaElementSourceContributor> contributors = getContributors();
		for (IItemJavaElementSourceContributor contributor : contributors) {
			final IItemJavaElementSourceContributor c = contributor;
			SafeRunner.run(new ISafeRunnable() {
				
				public void run() throws Exception {
					IItemJavaElementSource source = c
							.getItemJavaElementSource(object);
					if (source != null)
						sources.add(source);
				}
				
				public void handleException(Throwable exception) {
					MintCore.getInstance().logError("Unexpected error while invoking contributor. Disabling contributor for the session.", exception);
					invalidateContributor(c);
				}
			});
		}

		return sources;
	}

	protected synchronized Collection<IItemJavaElementSourceContributor> getContributors() {
		for (IExtension extension : newExtensions)
			loadExtension(extension);

		newExtensions.clear();
		return new ArrayList<IItemJavaElementSourceContributor>(contributors);
	}
	
	private synchronized void invalidateContributor(IItemJavaElementSourceContributor contributor) {
		contributors.remove(contributor);
	}

	private void loadExtension(IExtension extension) {
		for (IConfigurationElement config : extension
				.getConfigurationElements()) {
			String uri = config.getAttribute("uri");
			if (uri != null && !uri.equals(GenModelPackage.eNS_URI))
				continue;
			
			Object obj;
			try {
				obj = config.createExecutableExtension("class");
			} catch (CoreException e) {
				MintCore.getInstance().logError(
						"Could not instantiate contributor "
								+ config.getAttribute("class") + ".", e);
				continue;
			}

			if (!(obj instanceof IItemJavaElementSourceContributor)) {
				String msg = "Invalid contributor "
						+ (obj == null ? "null" : obj.getClass()) + ".";
				MintCore.getInstance().logError(msg, null);
				continue;
			}

			IItemJavaElementSourceContributor contributor = (IItemJavaElementSourceContributor) obj;
			if (contributor instanceof IChangeNotifier)
				((IChangeNotifier) contributor)
						.addListener(new INotifyChangedListener() {
							public void notifyChanged(Notification notification) {
								GenModelItemProviderAdapter.this
										.fireNotifyChanged(notification);
							}
						});

			contributors.add(contributor);
			extensionTracker.registerObject(extension, obj,
					IExtensionTracker.REF_WEAK);
		}
	}

	public synchronized void addExtension(IExtensionTracker tracker,
			IExtension extension) {
		if (newExtensions.add(extension))
			fireNotifyChanged(createRefreshNotification());
	}

	public synchronized void removeExtension(IExtension extension,
			Object[] objects) {
		boolean notify;
		if (!newExtensions.remove(extension)
				&& contributors.removeAll(Arrays.asList(objects))) {
			notify = true;
			for (Object object : objects)
				if (object instanceof IDisposable)
					((IDisposable) object).dispose();
		} else {
			notify = false;
		}

		if (notify)
			fireNotifyChanged(createRefreshNotification());
	}

	@Override
	public void setTarget(Notifier target) {
		if (this.target != null) {
			if (this.target != target) {
				if (targets == null)
					targets = new ArrayList<Notifier>();

				targets.add(this.target);
				super.setTarget(target);
			}
		} else {
			super.setTarget(target);
		}
	}

	@Override
	public void unsetTarget(Notifier target) {
		if (target == this.target) {
			if (targets == null || targets.isEmpty())
				super.setTarget(null);
			else
				super.setTarget(targets.remove(targets.size() - 1));
		} else if (targets != null) {
			targets.remove(target);
		}
	}

	public void dispose() {
		Notifier oldTarget = target;
		target = null;

		List<Notifier> oldTargets = targets;
		targets = null;

		if (oldTarget != null)
			oldTarget.eAdapters().remove(this);

		if (oldTargets != null)
			for (Notifier otherTarget : oldTargets) {
				otherTarget.eAdapters().remove(this);
			}

		extensionTracker.close();
		disposeContributors();
	}

	private synchronized void disposeContributors() {
		for (IItemJavaElementSourceContributor contributor : contributors)
			if (contributor instanceof IDisposable)
				((IDisposable) contributor).dispose();
	}

	private static Notification createRefreshNotification() {
		return new RefreshNotification();
	}

	private static class RefreshNotification extends NotificationImpl implements
			IJavaElementNotification {

		public RefreshNotification() {
			super(REFRESH, false, false);
		}

		public Object getElement() {
			return null;
		}
	}
}
