/*****************************************************************************
 * Copyright (c) 2020 CEA LIST.
 *
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 *
 *****************************************************************************/

package org.eclipse.papyrus.robotics.ros2.codegen.component

import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.ConstInit
import org.eclipse.papyrus.robotics.profile.robotics.functions.Function
import org.eclipse.uml2.uml.Behavior
import org.eclipse.uml2.uml.Class
import org.eclipse.uml2.uml.Operation
import org.eclipse.uml2.uml.util.UMLUtil

import static extension org.eclipse.papyrus.robotics.ros2.codegen.component.Constructor.addConstrOp
import static extension org.eclipse.papyrus.robotics.ros2.codegen.utils.ActivityUtils.*

/**
 * Create a template for the code that needs to be filled by a developer.
 * 
 */
class CodeSkeleton {
	public static String POSTFIX = "_impl"

	def static createSkeleton(Class component) {
		val skeleton = component.nearestPackage.createOwnedClass(component.name + POSTFIX, false);
		val comment = skeleton.createOwnedComment()
		comment.body = '''
			This is a skeleton class generated for component «component.name»
			Copy it into the source folder as an initial base (or copy parts
			of it whenever you add modify the component).
		'''
		// generated code uses template - it needs to instantiate and block ...
		component.createUsage(skeleton);
		// template inherits from generated component code
		skeleton.createGeneralization(component);
		// create a suitable constructor
		val op = skeleton.addConstrOp
		if (op !== null) {
			val constInit = UMLUtil.getStereotypeApplication(op, ConstInit)
			constInit.initialisation = '''«component.name»(options)'''
		}
		for (activity : component.activities) {
			for (function : activity.functions) {
				if (!function.codeInModel) {
					moveFunction(skeleton, function)
				}
			}
		}
	}

	/**
	 *  Move function from generated code into skeleton (empty declaration)
	 */
	def static moveFunction(Class skeleton, Function function) {
		if (function.base_Class instanceof Behavior) {
			val ob = function.base_Class as Behavior
			skeleton.ownedBehaviors.add(ob);
			skeleton.ownedOperations.add(ob.specification as Operation)
		}
	}
}
