/**
 * <copyright> 
 * 
 * Copyright (c) 2004-2005 IBM Corporation and others. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License - v 1.0 
 * which accompanies this distribution, and is available at 
 * http://opensource.org/licenses/eclipse-1.0.txt 
 * 
 * Contributors: 
 *   IBM - Initial API and implementation 
 * 
 * </copyright> 
 * 
 * $Id: RDFLiteralElement.java,v 1.2 2007/03/18 08:39:02 lzhang Exp $
 */

package org.eclipse.eodm.rdf.resource.parser.element;

/**
 * The class of RDF literal.
 */
public class RDFLiteralElement implements RDFValue {

    private String literalText;

    private String literalLanguage;

    private URIReference literalDatatype;

    /**
     * Construct rdf literal.
     * 
     * @param text
     *            the content of the literal
     */
    public RDFLiteralElement(String text) {
        this(text, null, null);
    }

    /**
     * Construct rdf literal.
     * 
     * @param text
     *            the content of the literal
     * @param language
     *            the language of the literal
     */
    public RDFLiteralElement(String text, String language) {
        this(text, language, null);
    }

    /**
     * Construct rdf literal.
     * 
     * @param text
     *            the content of the literal
     * @param language
     *            the language of the literal
     * @param datatype
     *            the data type of the literal
     */
    public RDFLiteralElement(String text, String language, URIReference datatype) {
        /** the text of a literal can not be null */
        if (text == null) {
            throw new IllegalArgumentException(
                    "text of literal must NOT be null");
        }
        literalText = text;
        if (language != null) {
            literalLanguage = language.intern();
        }

        // if (datatype == null && language == null) {
        // literalDatatype = RDFSURIConstant.C_XSDString_URI;
        // }
        if (datatype != null && language != null) {
            literalLanguage = null;
            literalDatatype = datatype;
        } else
            literalDatatype = datatype;
    }

    /**
     * get the content of the literal
     * 
     * @return the content
     */
    public String getText() {
        return literalText;
    }

    /**
     * get the language of the literal
     * 
     * @return the language
     */
    public String getLanguage() {
        return literalLanguage;
    }

    /**
     * get the data type of the literal
     * 
     * @return the data type
     */
    public URIReference getDatatype() {
        return literalDatatype;
    }

    /**
     * set the content of the literal
     * 
     * @param text
     *            the content
     */
    public void setText(String text) {
        literalText = text;
    }

    /**
     * set the language of the literal
     * 
     * @param language
     *            the language
     */
    public void setLanguage(String language) {
        literalLanguage = language;
    }

    /**
     * set the data type of the literal
     * 
     * @param datatype
     *            the data type
     */
    public void setDatatype(URIReference datatype) {
        literalDatatype = datatype;
    }

    public boolean equals(Object o) {
        if (this == o)
            return true;
        if (!(o instanceof RDFLiteralElement))
            return false;

        final RDFLiteralElement literal = (RDFLiteralElement) o;

        boolean textEquivalent = literalText == null ? literal.literalText == null
                : literalText.equals(literal.literalText);
        if (!textEquivalent)
            return false;

        if (literalDatatype == null) {
            if (literal.literalDatatype != null) {
                return false;
            }
        } else {
            if (literal.literalDatatype == null)
                return false;
            else {
                if (!literalDatatype.equals(literal.literalDatatype))
                    return false;
            }
            if (literalDatatype.equals(RDFSURIConstant.C_XSDString_URI)
                && literalLanguage == null && literal.literalLanguage == null
                && literal.literalDatatype == null)
                return true;

        }

        boolean languageEquivalent = literalLanguage == null ? literal.literalLanguage == null
                : literalLanguage.equals(literal.literalLanguage);
        if (!languageEquivalent)
            return false;

        return true;
    }

    public int hashCode() {
        int result;
        result = literalText.hashCode();
        result = 29
                 * result
                 + (literalLanguage != null ? literalLanguage.hashCode() : 0);
        result = 29 * result + literalDatatype.hashCode();
        return result;
    }

    public String toString() {
        final StringBuffer buf = new StringBuffer();
        buf.append("LiteralImpl");
        buf.append("{Text=").append(literalText);
        buf.append(",Language=").append(literalLanguage);
        buf.append(",Datatype=").append(literalDatatype);
        buf.append('}');
        return buf.toString();

    }

}