/**
 * <copyright> 
 * 
 * Copyright (c) 2004-2005 IBM Corporation and others. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License - v 1.0 
 * which accompanies this distribution, and is available at 
 * http://opensource.org/licenses/eclipse-1.0.txt 
 * 
 * Contributors: 
 *   IBM - Initial API and implementation 
 * 
 * </copyright> 
 * 
 * $Id: OWLStructuralSubsumptionReasoner.java,v 1.2 2007/03/18 09:07:00 lzhang Exp $
 */

package org.eclipse.eodm.owl.reasoner.structural;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;


import org.eclipse.eodm.exceptions.UnsupportedViewTypeException;
import org.eclipse.eodm.owl.owlbase.AllValuesFromRestriction;
import org.eclipse.eodm.owl.owlbase.CardinalityRestriction;
import org.eclipse.eodm.owl.owlbase.ComplementClass;
import org.eclipse.eodm.owl.owlbase.EnumeratedClass;
import org.eclipse.eodm.owl.owlbase.HasValueRestriction;
import org.eclipse.eodm.owl.owlbase.Individual;
import org.eclipse.eodm.owl.owlbase.IntersectionClass;
import org.eclipse.eodm.owl.owlbase.MaxCardinalityRestriction;
import org.eclipse.eodm.owl.owlbase.MinCardinalityRestriction;
import org.eclipse.eodm.owl.owlbase.OWLClass;
import org.eclipse.eodm.owl.owlbase.OWLDataRange;
import org.eclipse.eodm.owl.owlbase.OWLGraph;
import org.eclipse.eodm.owl.owlbase.OWLObjectProperty;
import org.eclipse.eodm.owl.owlbase.OWLOntology;
import org.eclipse.eodm.owl.owlbase.Property;
import org.eclipse.eodm.owl.owlbase.SomeValuesFromRestriction;
import org.eclipse.eodm.owl.owlbase.UnionClass;
import org.eclipse.eodm.owl.reasoner.OWLReasonerException;
import org.eclipse.eodm.owl.reasoner.OWLTaxonomyReasoner;
import org.eclipse.eodm.rdf.rdfbase.RDFProperty;
import org.eclipse.eodm.rdf.rdfbase.RDFSLiteral;
import org.eclipse.eodm.rdf.rdfbase.RDFSResource;
import org.eclipse.eodm.rdf.rdfs.RDFSClass;
import org.eclipse.eodm.rdf.rdfs.RDFSDatatype;
import org.eclipse.eodm.vocabulary.OWL;


public class OWLStructuralSubsumptionReasoner implements OWLTaxonomyReasoner {


	public void initialize(OWLGraph graph) throws OWLReasonerException {
        List properties = null;
        List classes = null;
        
        try {
        	classes = graph.getTypeResources(OWL.CLASS_URI);
        	properties = graph.getTypeResources(OWL.OBJECT_PROPERTY_URI);
        	properties.addAll( graph.getTypeResources(OWL.DATATYPE_PROPERTY_URI) );
        	rbox = createPropertyHierarchy(properties);
            
            tbox = new TBox(rbox);
        
            
            if(properties != null) {
    	        for (Iterator iter = properties.iterator(); iter.hasNext();) {
    	        	Property p = (Property) ((RDFProperty)iter.next()).asType( Property.class );
    	            addDomainAndRangeAxioms(p);
    	        }
            }
	        
            
	        // classify every class in ontology
	        for (Iterator iter = classes.iterator(); iter.hasNext();) {
	            OWLClass clazz = (OWLClass) iter.next();
	
	            // classify owl class when necessary
	            classifyOWLClass(clazz);

	            // add axioms
	            for (Iterator iterC = clazz.getSubClass().iterator(); iterC.hasNext();) {
	            	OWLClass sub = (OWLClass) ((RDFSResource)iterC.next()).asType(OWLClass.class);
	                tbox.addSubOfAxiom(OWLClass2Concept(sub), OWLClass2Concept(clazz));
	            }
	
	            // add equivalent classes
	            for (Iterator iterC = clazz.getEquivalentClass().iterator(); iterC.hasNext();) {
	            	tbox.addEquivalentAxiom(OWLClass2Concept((OWLClass) iterC.next()), OWLClass2Concept(clazz));
	            }
	        
	
	            // add disjointWith
	            for (Iterator iterC = clazz.getOWLdisjointWith().iterator(); iterC.hasNext();) {
	            	tbox.addDisjoint(OWLClass2Concept((OWLClass) iterC.next()), OWLClass2Concept(clazz));
	            }            
	
	        }
	        
        } catch (Exception e){
        	throw new OWLReasonerException("Fail to retrieve resources from Graph:" + e.getMessage());
        }
	}
	
	public void initialize(OWLOntology ontology) throws OWLReasonerException {
        this.ontology = ontology;

        OWLGraph graph = (OWLGraph)ontology.getOwlGraph().get(0);
        
        initialize(graph);
        
	}

	public List getDescendantClasses(OWLClass clazz) {
        ArrayList result = new ArrayList();
        CTNode v;
        
        try {
        	v = tbox.classify( OWLClass2Concept(clazz)  );
        } catch (UnsupportedViewTypeException e) {
        	return result;
        }
        
        List l = tbox.getSubNodes(v);
        
        for (Iterator iter = l.iterator(); iter.hasNext();) {
        	CTNode v1 = (CTNode) iter.next();

            Iterator it = v1.getCIDSet().iterator();
            while (it.hasNext()) {
            	
                Concept c = tbox.getConceptByID(((Integer) it.next()).intValue());
                //if (isNamedObject(clazz) ) {
                    Object obj = getObjectByUri(c.getName());
                    if (obj != null) {
                        result.add(obj);
                    }
                //}
            }

        }

        return result;
	}

	public List getAncestorClasses(OWLClass c) {
		ArrayList result = new ArrayList();
		CTNode v;
		
		try {
			v = tbox.classify(OWLClass2Concept(c));
		} catch (UnsupportedViewTypeException e) {
			return result;
		}
		
        List l = tbox.getSuperNodes(v);
        for (Iterator iter = l.iterator(); iter.hasNext();) {
            CTNode v1 = (CTNode) iter.next();

            Iterator it = v1.getCIDSet().iterator();
            while (it.hasNext()) {
                Concept clazz = tbox.getConceptByID(((Integer) it.next()).intValue());
                //if (isNamedObject(clazz) && !isBuildInClass(clazz)) {
                    Object obj = getObjectByUri(clazz.getName());
                    if (obj != null) {
                        result.add(obj);
                    }
                //}
            }

        }

        return result;
	}

	public List getDescendantProperties(Property property) {
        List l = rbox.getSubProperties( Property2Role(property)	);
        ArrayList result = new ArrayList();
        for (Iterator iter = l.iterator(); iter.hasNext();) {
            Object p = getObjectByUri((String) iter.next());
            if (p != null) {
                result.add(p);
            }
        }
        return result;
	}

	public List getAncestorProperties(Property property) {
        List l = rbox.getSuperProperties(Property2Role(property));
        Vector result = new Vector();
        for (Iterator iter = l.iterator(); iter.hasNext();) {
            Object p = getObjectByUri((String) iter.next());
            if (p != null) {
                result.add(p);
            }
        }
        return result;
	}

	public boolean isSubClassOf(OWLClass c1, OWLClass c2) {
		CTNode v1, v2;
		try {
            v1 = tbox.classify( OWLClass2Concept(c1) );
            v2 = tbox.classify( OWLClass2Concept(c2) );
		} catch (UnsupportedViewTypeException e) {
			return false;
		}
        return tbox.isSubNodeOf(v1, v2);
	}

	public boolean isSuperClassOf(OWLClass c1, OWLClass c2) {
		CTNode v1, v2;
		try {
			v1 = tbox.classify( OWLClass2Concept(c1) );
			v2 = tbox.classify( OWLClass2Concept(c2) );
		} catch (UnsupportedViewTypeException e) {
			return false;
		}
		
        return tbox.isSubNodeOf(v2, v1);
	}

	/**
	 * return true if p1 is subproperty of p2
	 */
	public boolean isSubPropertyOf(Property p1, Property p2) {
		return rbox.isSubPropertyOf( Property2Role(p1),Property2Role(p2) );
	}

	public boolean isSuperPropertyOf(Property p1, Property p2) {
        return rbox.isSubPropertyOf( Property2Role(p2), Property2Role(p1) );
	}

	void classifyOWLClass(OWLClass clazz) throws UnsupportedViewTypeException {
        if (isNamedObject(clazz)) {
            // adding name clazz to obj map
            addObject(clazz.getURI(), clazz);
            
            if(isPrimitiveClass(clazz)) {
            	tbox.addConcept( OWLClass2Concept(clazz) );
            } else {
            	// create a primitive class
            	Concept c1 = TermFactory.instance().createPrim(clazz.getURI());
            	tbox.addEquivalentAxiom(c1 , OWLClass2Concept(clazz) );
            }

        } 
	}
	
    protected RBox createPropertyHierarchy(List props) throws UnsupportedViewTypeException {
        RBox rbox = new RBox();
        
        if(props!=null){
	        for (Iterator iter = props.iterator(); iter.hasNext();) {
	        	Property p = (Property) iter.next();
	            
	            Role father = Property2Role(p);
	
	            List sublst = p.getSubProperty();
	            for (Iterator iterP = sublst.iterator(); iterP.hasNext();) {
	            	//RDFProperty subP = (RDFProperty) iterP.next();
	            	Property subP = (Property) ((RDFProperty)iterP.next()).asType( Property.class );
	            	
	            	Role child = Property2Role(subP);
	                rbox.addSubPropertyOfAxiom(child, father);
	            }
	        }
        }

        return rbox;
    }
    
    protected void addDomainAndRangeAxioms(Property p) throws UnsupportedViewTypeException {
    	
        if ( p instanceof OWLObjectProperty ) {
        	Role r = Property2Role(p);
        	
        	SomeValueFrom someRT = TermFactory.instance().createSome(r,ReasonerUtil.TOP);
        	
        	for(Iterator it=p.getRDFSdomain().iterator();it.hasNext();){
        		OWLClass clz = (OWLClass) ((RDFSResource)it.next()).asType(OWLClass.class);
            	Concept domain =  OWLClass2Concept( clz );
            	tbox.addSubOfAxiom(someRT, domain);
        	}

        	for(Iterator it=p.getRDFSrange().iterator();it.hasNext();){
        		OWLClass clz = (OWLClass) ((RDFSResource)it.next()).asType(OWLClass.class);
            	Concept range =  OWLClass2Concept( clz);
                AllValueFrom allRrange = TermFactory.instance().createAll(r,range);
                tbox.addSubOfAxiom(ReasonerUtil.TOP, allRrange);
        	}
        }

    }
    
    Concept OWLClass2Concept(OWLClass clazz) throws UnsupportedViewTypeException {
    	Concept c = null;
    
    	switch(getOWLClassType(clazz)) {
    	case OWL_UNION:
    		Union un = TermFactory.instance().createUnion();
    		 
    		for(Iterator it=((UnionClass)clazz.asType(UnionClass.class)).getOWLunionOf().iterator();it.hasNext();){
    			un.addUnionOf( OWLClass2Concept( (OWLClass)it.next() ) );
    		}
    		c = un;
    		break;
    	
    	case OWL_INTERSECTION:
    		Intersection in = TermFactory.instance().createIntersect();
    		for(Iterator it=((IntersectionClass)clazz.asType(IntersectionClass.class)).getOWLintersectionOf().iterator();it.hasNext();){
    			in.addIntersectOf(OWLClass2Concept( (OWLClass)it.next()) );
    		}
    		c = in;
    		break;
    	
    	case OWL_SOME_VALUE_FORM:
    		SomeValuesFromRestriction some = (SomeValuesFromRestriction)clazz.asType(SomeValuesFromRestriction.class);
    		Role r = Property2Role(	some.getOWLonProperty() );
    		Concept range = OWLClass2Concept( some.getSomeValuesFromClass());
    		c = TermFactory.instance().createSome(r,range);
    		break;
    	
    	case OWL_ALL_VALUE_FROM:
    		AllValuesFromRestriction all = (AllValuesFromRestriction)clazz.asType(AllValuesFromRestriction.class);
    		r = Property2Role(	all.getOWLonProperty() );
    		range = OWLClass2Concept( all.getAllValuesFromClass());
    		c = TermFactory.instance().createAll(r,range);
    		break;
    	
    	case OWL_HAS_VALUE:
    		HasValueRestriction has = (HasValueRestriction)clazz.asType(HasValueRestriction.class);
       		r = Property2Role(	has.getOWLonProperty() );
       		
       		RDFSResource res = has.getHasIndividualValue();
       		
       		if ( res == null) {
       			res = has.getHasLiteralValue();
       		}
       		
       		Resource rsource = RDFSResource2Resource( res );
       		c = TermFactory.instance().createHasValue(r,rsource);
       		break;
    	
    	case OWL_CARDINARITY:
    		CardinalityRestriction card = (CardinalityRestriction)clazz.asType(CardinalityRestriction.class);
    		r = Property2Role(	card.getOWLonProperty() );
    		int n = Integer.parseInt( card.getOWLcardinality().getLexicalForm() );
    		c = TermFactory.instance().createCardinality(r, n, ReasonerUtil.TOP);
    		break;
    	
    	case OWL_MAX_CARDINARITY:
    		MaxCardinalityRestriction max = (MaxCardinalityRestriction)clazz.asType(MaxCardinalityRestriction.class); 
    		r = Property2Role(	max.getOWLonProperty() );
    		n = Integer.parseInt( max.getOWLmaxCardinality().getLexicalForm() );
    		c = TermFactory.instance().createMaxCardinality(r, n, ReasonerUtil.TOP);
    		break;
    	
    	case OWL_MIN_CARDINARITY:
    		MinCardinalityRestriction min = (MinCardinalityRestriction) clazz.asType(MinCardinalityRestriction.class);
    		r = Property2Role(	min.getOWLonProperty() );
    		n = Integer.parseInt( min.getOWLminCardinality().getLexicalForm() );
    		c = TermFactory.instance().createMinCardinality(r, n, ReasonerUtil.TOP);
    		break;
    	case OWL_COMPLEMENT:
    		ComplementClass comp = (ComplementClass)clazz.asType(ComplementClass.class);
    		c = TermFactory.instance().createNegation(
      			  OWLClass2Concept( comp.getOWLcomplementOf() )
      			);
    		break;
    	case OWL_ONE_OF:
    	case OWL_DATA_RANGE:
    	case RDFS_DATATYPE: 
    	case OWL_PRIMITIVE:
    		// primtive class
    		c = TermFactory.instance().createPrim(clazz.getURI());
    		break;
    	}
    	
    	return c;
    }
    
    Role Property2Role(RDFProperty p) {
		String uri = p.getURI();
		
    	Role r = (Role)propURI2roleMap.get(uri);
    	
    	if(r==null){
    		r = TermFactory.instance().createRole(uri);
    		propURI2roleMap.put(uri,r);
    		uri2objectMap.put(uri,p);
    	}
    	
    	return r;
    }
    
    Resource RDFSResource2Resource(RDFSResource r) throws UnsupportedViewTypeException {
    	Resource rsource = null;
    	
    	if( r.canAsType(Individual.class) ){
    		rsource = TermFactory.instance().createIndividual( ((Individual)r.asType(Individual.class)).getURI() );
    	} else if( r.canAsType(RDFSLiteral.class) ) {
    		rsource = TermFactory.instance().createLiteral( ((RDFSLiteral)r.asType(RDFSLiteral.class)).getLexicalForm() );
    	}
    	
    	return rsource;
    }
        
    HashMap propURI2roleMap = new HashMap();
    
    
    protected OWLOntology ontology = null;
    
    /**
     * HashMap of URI -> Ontology Objects if it is support by model, it can be
     * abandoned
     */
    protected HashMap uri2objectMap = new HashMap();

    protected Object getObjectByUri(String uri) {
        return uri2objectMap.get(uri);
    }

    protected void addObject(String key, RDFSResource obj) {
        uri2objectMap.put(key, obj);
    }
   
    // owl class type
    protected static final int OWL_PRIMITIVE = -1;
    protected static final int OWL_UNION = 0;
    protected static final int OWL_INTERSECTION = 1;
    protected static final int OWL_SOME_VALUE_FORM = 2;
    protected static final int OWL_ALL_VALUE_FROM = 3;
    protected static final int OWL_HAS_VALUE = 4;
    protected static final int OWL_CARDINARITY = 5;
    protected static final int OWL_MAX_CARDINARITY = 6;
    protected static final int OWL_MIN_CARDINARITY = 7;
    protected static final int OWL_COMPLEMENT = 8;
    protected static final int OWL_ONE_OF = 9;
    protected static final int OWL_DATA_RANGE = 10;
    protected static final int RDFS_DATATYPE = 11;
   
    protected static Class[] nonPrimitiveClass = { UnionClass.class,
        IntersectionClass.class, SomeValuesFromRestriction.class,
        AllValuesFromRestriction.class, HasValueRestriction.class,
        CardinalityRestriction.class, MaxCardinalityRestriction.class,
        MinCardinalityRestriction.class, ComplementClass.class,
        EnumeratedClass.class, OWLDataRange.class, RDFSDatatype.class 
    };
    
    protected static int getOWLClassType(RDFSClass c) {
        for (int i = 0; i < nonPrimitiveClass.length; i++) {
        	
            if ( c.canAsType(nonPrimitiveClass[i]))  // nonPrimitiveClass[i].isInstance(c))
                return i;
        }

        return OWL_PRIMITIVE;
    }
    
    boolean isNamedObject(RDFSResource c){
    	String nid = c.getNodeID();
    	return (nid.length()==0); 
    }
    
    boolean isPrimitiveClass(OWLClass clazz) {
    	return (getOWLClassType(clazz) == OWL_PRIMITIVE);
    }
    
    TBox tbox = null;
    
    RBox rbox = null;
    
	//private static String RDF_PROPERTY = "http://www.w3.org/1999/02/22-rdf-syntax-ns#Property";

}
