/**
 * <copyright> 
 * 
 * Copyright (c) 2004-2005 IBM Corporation and others. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License - v 1.0 
 * which accompanies this distribution, and is available at 
 * http://opensource.org/licenses/eclipse-1.0.txt 
 * 
 * Contributors: 
 *   IBM - Initial API and implementation 
 * 
 * </copyright> 
 * 
 * $Id: ReasonerUtil.java,v 1.2 2007/03/18 10:24:39 lzhang Exp $
 */

package org.eclipse.eodm.owl.reasoner.structural;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

public class ReasonerUtil {

    public static final String THING = "http://www.w3.org/2002/07/owl#Thing";
    
    public static final String NOTHING = "http://www.w3.org/2002/07/owl#Nothing";
	
    public static final PrimitiveConcept TOP = TermFactory.instance().createPrim(THING);

    public static final PrimitiveConcept BOTTOM = TermFactory.instance().createPrim(NOTHING);
    // public static final Negation BOTTOM = TermFactory.instance().createNegation(TOP);

    public static Concept[] getAllForms(Concept c, RBox rbox){
    	Concept[] forms = new Concept[2];
    	
    	forms[0] = toMinimalDNF(c,rbox);
    	forms[1] = toMaximalCNF(c,rbox);
    	
    	return forms;
    }
    
    /**
     * Normalize concept to Disjunctive Normal Form(DNF)
     * 
     * @param c
     * @param normalizeRestriction
     *            if true, it will normalize the expression within restrictions
     * @return concept in DNF
     */
    public static Concept toDNF(Concept c, boolean normalizeRestriction) {
        Concept term = c;
        List clist = null;
        int size = -1;
        
        int type = (c==null) ? -1 : c.getCType();
        switch (type) {
        case TermUtil.C_INTERSECT:
            clist = ((Intersection) c).getIntersectOf();
            size = clist.size();
            if (size > 1) {
                // transform the whole express to DNF
                // the algorithm is based on building an id list by applying
                // de-morgan exchange
                // First it will transform all sub or expression to DNF,
                // e.g. (P_1 or P_2) and (Q_1 or Q_2), each P_i, Q_j is a
                // conjunction of elementary propositions
                // Then the final CNF will be disjunction of all possible pairs
                // of distributed conjunctions:
                // ... Or ( P_i and Q_j ) Or ...
                List list = new ArrayList();
                Intersection prefix = TermFactory.instance().createIntersect();
                int length = 1;
                for (int i = 0; i < size; i++) {
                    Concept child = toDNF((Concept) clist.get(i),
                            normalizeRestriction);

                    if (child.getCType() == TermUtil.C_UNION) {
                        List or = ((Union) child).getUnionOf();
                        list.add(or);
                        length *= or.size();
                    } else {
                        prefix.addIntersectOf(child);
                    }
                }

                int slength = list.size(); // length of list
                if (slength == 0) {
                    term = prefix; // all children is in prefix
                } else {
                    // building ID list
                    int idlist[][] = new int[length][slength];
                    int head = 1;
                    for (int i = 0; i < slength; i++) {
                        int len = ((List)list.get(i)).size(); // length of every sub
                        // "Or list"
                        int index = 0;
                        for (int j = 0; j < head; j++) {
                            int tail = length / (head * len);
                            for (int val = 0; val < len; val++) {
                                for (int k = 0; k < tail; k++) {
                                    idlist[index++][i] = val;
                                }
                            }
                        }
                        head *= len;
                    }

                    // create DNF according to id tree
                    Union dnf = TermFactory.instance().createUnion();
                    for (int i = 0; i < length; i++) {
                        Intersection sub = TermFactory.instance()
                                .createIntersect();
                        for (int j = 0; j < slength; j++) {
                            sub.addIntersectOf(prefix);
                            sub.addIntersectOf((Concept) ((List)list.get(j)).get(
                                    idlist[i][j]));
                        }
                        dnf.addUnionOf(sub);
                    }
                    term = dnf;
                }
            } else if (size == 1) {
                term = toDNF((Concept) clist.get(0), normalizeRestriction);
            } else {
                term = ReasonerUtil.BOTTOM;
            }
            break;
        case TermUtil.C_UNION:
            clist = ((Union) c).getUnionOf();
            size = clist.size();
            if (size > 1) {
                term = TermFactory.instance().createUnion();
                for (int i = 0; i < size; i++) {
                	Concept c1 = (Concept) clist.get(i);
                    ((Union) term).addUnionOf(toDNF(c1, normalizeRestriction));
                }
            } else if (size == 1) {
                term = toDNF((Concept) clist.get(0), normalizeRestriction);
            } else {
                term = ReasonerUtil.BOTTOM;
            }
            break;
        case TermUtil.C_SOME:
            if (normalizeRestriction) {
                SomeValueFrom s = (SomeValueFrom) c;
                term = TermFactory.instance().createSome(s.getOnProperty(),
                        toDNF(s.getRange(), normalizeRestriction));
            } else {
                term = cloneConcept(c);
            }
            break;
        case TermUtil.C_ALL:
            if (normalizeRestriction) {
                AllValueFrom a = (AllValueFrom) c;
                term = TermFactory.instance().createAll(a.getOnProperty(),
                        toDNF(a.getRange(), normalizeRestriction));
            } else {
                term = cloneConcept(c);
            }
            break;
//        case TermUtil.C_CARD:
//            if (normalizeRestriction) {
//                Cardinality a = (Cardinality) c;
//                term = TermFactory.instance().createCardinality(
//                        a.getOnProperty(), a.getValRange(),
//                        toDNF(a.getRange(), normalizeRestriction));
//            } else {
//                term = cloneConcept(c);
//            }
//            break;
//        case TermUtil.C_MAX:
//            if (normalizeRestriction) {
//                MaxCardinality a = (MaxCardinality) c;
//                term = TermFactory.instance().createMaxCardinality(
//                        a.getOnProperty(), a.getValRange(),
//                        toDNF(a.getRange(), normalizeRestriction));
//            } else {
//                term = cloneConcept(c);
//            }
//            break;
//        case TermUtil.C_MIN:
//            if (normalizeRestriction) {
//                MinCardinality a = (MinCardinality) c;
//                term = TermFactory.instance().createMinCardinality(
//                        a.getOnProperty(), a.getValRange(),
//                        toDNF(a.getRange(), normalizeRestriction));
//            } else {
//                term = cloneConcept(c);
//            }
//            break;
        case TermUtil.C_VALUE:
        case TermUtil.C_NEGATION:
        default:
            term = cloneConcept(c);
        }

        return optimizeDNF(term);
    }

    /**
     * Optimize an expression in DNF. It will remove redundant elements in DNF,
     * E.g. (or (and 1 2) (and 1 2 3) ) will be optimized as (and 1 2)
     * 
     * @param dnf
     *            expression to be optimized
     * @return optimized DNF
     */
    private static Concept optimizeDNF(Concept dnf) {
        Concept term = dnf;
        List clist = null;

        int type = (dnf==null) ? -1 : dnf.getCType();
        switch (type) {
        case TermUtil.C_INTERSECT:
            clist = ((Intersection) dnf).getIntersectOf();
            if (clist.size() == 1) {
                term = (Concept) clist.get(0);
            }
            break;

        case TermUtil.C_UNION:
            clist = ((Union) dnf).getUnionOf();

            // find redundant elements
            for(int i=clist.size()-1;i>=0;i--){
            	Concept c1 = (Concept) clist.get(i);
            	if(c1.getCType() == TermUtil.C_INTERSECT) {
            		for(int j=i-1;j>=0;j--) {
            			Concept c2 = (Concept)clist.get(j);
            			if(c2.getCType() == TermUtil.C_INTERSECT) {
            				if(((Intersection) c1).getIntersectOf()
                               .containsAll( ((Intersection) c2) .getIntersectOf()) ) {
                				clist.remove(i);
                				break;
            				} else if(((Intersection) c2).getIntersectOf()
                               .containsAll( ((Intersection) c1) .getIntersectOf()) ) {
            					clist.remove(j);
            					i--;
            				}
            			} else if (((Intersection) c1).getIntersectOf().contains(c2)){
            				clist.remove(i);
            				break;
            			}
            		}
            	} else {
            		for(int j=i-1;j>=0;j--) {
            			Concept c2 = (Concept)clist.get(j);
            			if(c2.getCType() == TermUtil.C_INTERSECT && 
            		       ((Intersection) c2).getIntersectOf().contains(c1) ) {
            				clist.remove(j);
            				i--;
            			}
            		}
            	}
            }
            
//            int i = 0;
//            while (i < clist.size()) {
//                Concept c1 = (Concept) clist.get(i);
//                boolean forward = true;
//                if (c1.getCType() == TermUtil.C_INTERSECT) {
//                    for (Iterator it = clist.iterator(); it.hasNext();) {
//                        Concept c2 = (Concept) it.next();
//                        if (c1 == c2) {
//                            continue;
//                        } else if (c2.getCType() == TermUtil.C_INTERSECT) {
//                            if (((Intersection) c1).getIntersectOf()
//                                    .containsAll(
//                                            ((Intersection) c2)
//                                                    .getIntersectOf())) {
//                                clist.remove(c1);
//                                forward = false;
//                                break;
//                            }
//                        } else if (((Intersection) c1).getIntersectOf()
//                                .contains(c2)) {
//                            clist.remove(c1);
//                            forward = false;
//                            break;
//                        }
//                    }
//                }
//
//                if (forward) {
//                    i++;
//                }
//            }

            // remove non-sense conjunctions, for example, (and c) will be
            // replace as c
            for (int i = 0; i < clist.size(); i++) {
                Concept c1 = (Concept) clist.get(i);
                if (c1.getCType() == TermUtil.C_INTERSECT && ((Intersection) c1).getIntersectOf().size() == 1) {
                    ((ArrayList) clist).set(i, ((Intersection) c1).getIntersectOf().get(0));
                }
            }

            if (clist.size() == 1) {
                term = (Concept) clist.get(0);
            }
            break;

        default:
            break;
        }

        return term;
    }

    /**
     * Normalize concept to Conjunctive Normal Form(CNF)
     * 
     * @param c
     * @return concept in CNF
     */
    public static Concept toCNF(Concept c, boolean normalizeRestriction) {
        Concept term = c;
        List clist = null;
        int size = -1;
        
        int type = (c==null) ? -1 : c.getCType();
        switch (type) {
        case TermUtil.C_INTERSECT:
            clist = ((Intersection) c).getIntersectOf();
            size = clist.size();
            if (size > 1) {
                term = TermFactory.instance().createIntersect();
                for (int i = 0; i < size; i++) {
                    ((Intersection) term).addIntersectOf(toCNF((Concept) clist
                            .get(i), normalizeRestriction));
                }
            } else if (size == 1) {
                term = toCNF((Concept) clist.get(0), normalizeRestriction);
            } else {
                term = ReasonerUtil.BOTTOM;
            }
            break;
        case TermUtil.C_UNION:
            clist = ((Union) c).getUnionOf();
            size = clist.size();
            if (size > 1) {
                // transform the whole express to CNF
                // the algorithm is based on building an id list by applying
                // de-morgan exchange
                // First it will transform all sub or expression to CNF,
                // e.g. (P_1 and P_2) or (Q_1 and Q_2), each P_i, Q_j is a
                // disjunction of elementary propositions
                // Then the final CNF will be conjunction of all possible pairs
                // of distributed disjunctions:
                // ... And ( P_i or Q_j ) And ...
                Union prefix = TermFactory.instance().createUnion();
                List list = new ArrayList();
                int length = 1; // denote the length of DNF expression
                for (int i = 0; i < size; i++) {
                    Concept child = toCNF((Concept) clist.get(i),
                            normalizeRestriction);
                    if (child.getCType() == TermUtil.C_INTERSECT) {
                        List and = ((Intersection) child).getIntersectOf();
                        list.add(and);
                        length *= and.size();
                    } else {
                        prefix.addUnionOf(child);
                    }
                }

                int slength = list.size(); // length of list
                if (slength == 0) {
                    term = prefix; //c;
                } else {
                    // building ID list
                    int idlist[][] = new int[length][slength];
                    int head = 1;
                    for (int i = 0; i < slength; i++) {
                        int len = ((List)list.get(i)).size(); // length of every sub
                        // "Or list"
                        int index = 0;
                        for (int j = 0; j < head; j++) {
                            int tail = length / (head * len);
                            for (int val = 0; val < len; val++) {
                                for (int k = 0; k < tail; k++) {
                                    idlist[index++][i] = val;
                                }
                            }
                        }
                        head *= len;
                    }

                    // create DNF according to id tree
                    Intersection cnf = TermFactory.instance().createIntersect();
                    for (int i = 0; i < length; i++) {
                        Union sub = TermFactory.instance().createUnion();
                        sub.addUnionOf(prefix);
                        for (int j = 0; j < slength; j++) {
                            sub.addUnionOf((Concept) ((List)list.get(j)).get(
                                    idlist[i][j]));
                        }
                        cnf.addIntersectOf(sub);
                    }
                    term = cnf;
                }

            } else if (size == 1) {
                term = toCNF((Concept) clist.get(0), normalizeRestriction);
            } else {
                term = ReasonerUtil.BOTTOM;
            }
            break;
        case TermUtil.C_SOME:
            if (normalizeRestriction) {
                SomeValueFrom s = (SomeValueFrom) c;
                term = TermFactory.instance().createSome(s.getOnProperty(),
                        toCNF(s.getRange(), normalizeRestriction));
            } else {
                term = cloneConcept(c);
            }

            break;
        case TermUtil.C_ALL:
            if (normalizeRestriction) {
                AllValueFrom a = (AllValueFrom) c;
                term = TermFactory.instance().createAll(a.getOnProperty(),
                        toCNF(a.getRange(), normalizeRestriction));
            } else {
                term = cloneConcept(c);
            }

            break;
//        case TermUtil.C_CARD:
//            if (normalizeRestriction) {
//                Cardinality a = (Cardinality) c;
//                term = TermFactory.instance().createCardinality(
//                        a.getOnProperty(), a.getValRange(),
//                        toCNF(a.getRange(), normalizeRestriction));
//            } else {
//                term = cloneConcept(c);
//            }
//
//            break;
//        case TermUtil.C_MAX:
//            if (normalizeRestriction) {
//                MaxCardinality a = (MaxCardinality) c;
//                term = TermFactory.instance().createMaxCardinality(
//                        a.getOnProperty(), a.getValRange(),
//                        toCNF(a.getRange(), normalizeRestriction));
//            } else {
//                term = cloneConcept(c);
//            }
//
//            break;
//        case TermUtil.C_MIN:
//            if (normalizeRestriction) {
//                MinCardinality a = (MinCardinality) c;
//                term = TermFactory.instance().createMinCardinality(
//                        a.getOnProperty(), a.getValRange(),
//                        toCNF(a.getRange(), normalizeRestriction));
//            } else {
//                term = cloneConcept(c);
//            }
//
//            break;
        case TermUtil.C_NEGATION:
        case TermUtil.C_VALUE:
        default:
            term = cloneConcept(c);
        }
        return optimizeCNF(term);
    }

    /**
     * Optimize an expression in CNF. It will remove redundant elements in CNF,
     * E.g. (and (or 1 2) (or 1 2 3)) will be optimized as (or 1 2)
     * 
     * @param cnf
     *            expression to be optimized
     * @return optimized CNF
     */
    private static Concept optimizeCNF(Concept cnf) {
        Concept term = cnf;
        List clist = null;

        int type = (cnf==null) ? -1 : cnf.getCType();
        switch (type) {
        case TermUtil.C_UNION:
            clist = ((Union) cnf).getUnionOf();
            if (clist.size() == 1) {
                term = (Concept) clist.get(0);
            }
            break;

        case TermUtil.C_INTERSECT:
            clist = ((Intersection) cnf).getIntersectOf();
            
            // find redundant elements
            for(int i=clist.size()-1;i>=0;i--){
            	Concept c1 = (Concept) clist.get(i);
            	if(c1.getCType() == TermUtil.C_UNION) {
            		for(int j=i-1;j>=0;j--) {
            			Concept c2 = (Concept)clist.get(j);
            			if(c2.getCType() == TermUtil.C_UNION) {
            				if(((Union)c1).getUnionOf().containsAll( ((Union)c2).getUnionOf()) ) {
                				clist.remove(i);
                				break;
            				} else if(((Union)c2).getUnionOf().containsAll( ((Union)c1).getUnionOf()) ) {
            					clist.remove(j);
            					i--;
            				}
            			} else if (((Union)c1).getUnionOf().contains(c2)){
            				clist.remove(i);
            				break;
            			}
            		}
            	} else {
            		for(int j=i-1;j>=0;j--) {
            			Concept c2 = (Concept)clist.get(j);
            			if(c2.getCType() == TermUtil.C_UNION && ((Union)c2).getUnionOf().contains(c1) ) {
            				clist.remove(j);
            				i--;
            			}
            		}
            	}
            }
            
//            int i = 0;
//            while (i < clist.size()) {
//                Concept c1 = (Concept) clist.get(i);
//                boolean forward = true;
//
//                if (c1.getCType() == TermUtil.C_UNION) {
//                    for (Iterator it = clist.iterator(); it.hasNext();) {
//                        Concept c2 = (Concept) it.next();
//                        if (c1 == c2) {
//                            continue;
//                        } else if (c2.getCType() == TermUtil.C_UNION) {
//                            if (((Union) c1).getUnionOf().containsAll(
//                                    ((Union) c2).getUnionOf())) {
//                                clist.remove(c1);
//                                forward = false;
//                                break;
//                            }
//                        } else if (((Union) c1).getUnionOf().contains(c2)) {
//                            clist.remove(c1);
//                            forward = false;
//                            break;
//                        }
//                    }
//                }
//
//                if (forward) {
//                    i++;
//                }
//            }

            // remove non-sense disjunctions, for example, (or c) will be
            // replace as c
            for (int i = 0; i < clist.size(); i++) {
                Concept c1 = (Concept) clist.get(i);
                if (c1.getCType() == TermUtil.C_UNION  && ((Union) c1).getUnionOf().size() == 1) {
                    ((ArrayList) clist).set(i, ((Union) c1).getUnionOf().get(0));
                }
            }

            if (clist.size() == 1) {
                term = (Concept) clist.get(0);
            }
            break;

        default:
            break;
        }

        return term;
    }

    /**
     * Applying transformation transformation rules on concept c which in form
     * of DNF It will excute the following transformation rules:
     * 
     * @param c
     * @return
     */
    public static Concept toMinimalDNF(Concept c, RBox rbox) {
        Concept term = toDNF(c, false);

        int type = (term==null) ? -1 : term.getCType();
        switch (type) {
        case TermUtil.C_INTERSECT:
            applyIRules((Intersection) term, rbox);

            // normalize expressions in restrictions
            for (Iterator it = ((Intersection) term).getIntersectOf()
                    .iterator(); it.hasNext();) {
                Concept c1 = (Concept) it.next();
                if (c1.isRestriction() && c1.getCType() != TermUtil.C_VALUE) {
                    Restriction rest = (Restriction) c1;
                    rest.setRange(toMinimalDNF(rest.getRange(), rbox));
                }
            }
            break;

        case TermUtil.C_UNION:
            // for each conjunction of union, apply IRules
            List ulist = ((Union) term).getUnionOf();
            for (Iterator it = ulist.iterator(); it.hasNext();) {
                Concept subc = (Concept) it.next();
                switch (subc.getCType()) {
                case TermUtil.C_INTERSECT:
                    applyIRules((Intersection) subc, rbox);

                    // normalize expressions in restrictions
                    for (Iterator iter = ((Intersection) subc).getIntersectOf()
                            .iterator(); iter.hasNext();) {
                        Concept c1 = (Concept) iter.next();
                        if (c1.isRestriction()
                            && c1.getCType() != TermUtil.C_VALUE) {
                            Restriction rest = (Restriction) c1;
                            rest.setRange(toMinimalDNF(rest.getRange(), rbox));
                        }
                    }
                    break;
                case TermUtil.C_ALL:
                case TermUtil.C_SOME:
                case TermUtil.C_CARD:
                case TermUtil.C_MAX:
                case TermUtil.C_MIN:
                    Restriction rest = (Restriction) subc;
                    rest.setRange(toMinimalDNF(rest.getRange(), rbox));
                    break;
                default:
                    break;
                }
            }
            break;

        case TermUtil.C_ALL:
        case TermUtil.C_CARD:
        case TermUtil.C_MAX:
        case TermUtil.C_MIN:
        case TermUtil.C_SOME:
            // normalize expressions in restrictions
            Restriction rest = (Restriction) term;
            rest.setRange(toMinimalDNF(rest.getRange(), rbox));
            break;

        case TermUtil.C_NEGATION:
        case TermUtil.C_VALUE:
        default:
            break;
        }

        return term;
    }

    /**
     * Apply the following transformation rules on an intersection concept.
     * Expressions within restrictions will not be dealt with.
     * 
     * R1. (and (all R A) (all S B) ) --> (and (all R (and A B)) (all S B) where
     * S<R R3. (and (all R A) (some S B) ) --> (and (all R A) (some S (and A
     * B)) where S<R R3' (and (all R A) (min n S B)) --> (and (all R A) (min n
     * S(and A B)) where S<R
     * 
     * @param c
     * @param rbox
     * @return
     */
    private static Concept applyIRules(Intersection c, RBox rbox) {
        List clist = c.getIntersectOf();

        for (Iterator it = clist.iterator(); it.hasNext();) {
            Concept c1 = (Concept) it.next();

            if (c1.getCType() == TermUtil.C_ALL) {
                // otherwise, it will not influent any other constructs
                for (Iterator iter = clist.iterator(); iter.hasNext();) {
                    Concept c2 = (Concept) iter.next();

                    if (c2 != c1) {
                        switch (c2.getCType()) {
                        case TermUtil.C_ALL:
                        case TermUtil.C_MIN:
                        case TermUtil.C_SOME:
                        case TermUtil.C_CARD:
                        case TermUtil.C_MAX:
                            Restriction rest = (Restriction) c2;
                            Role p = rest.getOnProperty();
                            Role r = ((AllValueFrom) c1).getOnProperty();

                            if (rbox.isSubPropertyOf(p, r)) {
                                Concept range = rest.getRange();
                                if (range.getCType() == TermUtil.C_INTERSECT) {
                                    ((Intersection) range)
                                            .addIntersectOf(((AllValueFrom) c1)
                                                    .getRange());
                                } else {
                                    rest.setRange(TermFactory.instance()
                                            .createIntersect(
                                                    range,
                                                    ((AllValueFrom) c1)
                                                            .getRange()));
                                }
                            }
                            break;
                        default:
                            break;
                        }
                    }
                }
            }
        }

        return c;
    }

    public static Concept toMaximalCNF(Concept c, RBox rbox) {
        Concept term = toCNF(c, false);

        int type = (term==null) ? -1 : term.getCType();
        switch (type) {
        case TermUtil.C_INTERSECT:
            // for each disjunction of intersection, apply URules
            List ilist = ((Intersection) term).getIntersectOf();
            for (Iterator it = ilist.iterator(); it.hasNext();) {
                Concept subc = (Concept) it.next();

                switch (subc.getCType()) {
                case TermUtil.C_UNION:
                    applyURules((Union) subc, rbox);

                    // normalize expressions in restrictions
                    for (Iterator iter = ((Union) subc).getUnionOf().iterator(); iter
                            .hasNext();) {
                        Concept c1 = (Concept) iter.next();
                        if (c1.isRestriction()
                            && c1.getCType() != TermUtil.C_VALUE) {
                            Restriction rest = (Restriction) c1;
                            rest.setRange(toMaximalCNF(rest.getRange(), rbox));
                        }
                    }
                    break;
                case TermUtil.C_ALL:
                case TermUtil.C_SOME:
                case TermUtil.C_CARD:
                case TermUtil.C_MAX:
                case TermUtil.C_MIN:
                    Restriction rest = (Restriction) subc;
                    rest.setRange(toMaximalCNF(rest.getRange(), rbox));
                    break;
                default:
                    break;
                }
            }
            break;

        case TermUtil.C_UNION:
            applyURules((Union) term, rbox);

            // normalize expressions in restrictions
            for (Iterator it = ((Union) term).getUnionOf().iterator(); it
                    .hasNext();) {
                Concept c1 = (Concept) it.next();
                if (c1.isRestriction() && c1.getCType() != TermUtil.C_VALUE) {
                    Restriction rest = (Restriction) c1;
                    rest.setRange(toMaximalCNF(rest.getRange(), rbox));
                }
            }
            break;

        case TermUtil.C_ALL:
        case TermUtil.C_CARD:
        case TermUtil.C_MAX:
        case TermUtil.C_MIN:
        case TermUtil.C_SOME:
            // normalize expressions in restrictions
            Restriction rest = (Restriction) term;
            rest.setRange(toMaximalCNF(rest.getRange(), rbox));
            break;

        case TermUtil.C_NEGATION:
        case TermUtil.C_VALUE:
        default:
            break;

        }

        return term;
    }

    /**
     * Applying transformation transformation rules on a union concept c which
     * in form of CNF It will excute the following transformation rules:
     * 
     * R2. (or (some R A) (some S B) ) --> (or (some R (or A B)) (some S B)
     * where S<R R2' (or (min 1 R A) (min n S B)) --> (or (min 1 R A) (min n S
     * (or A B)) ) where S<R R4. (or (some R A) (all S B) ) --> (or (some R A)
     * (all S (or A B)) ) where S<R R4' (or (min 1 R A) (all S B)) --> (or (min
     * 1 R A)(all S (or A B)) ) where S<R
     * 
     * @param c
     * @return
     */
    private static Concept applyURules(Union c, RBox rbox) {
        List clist = c.getUnionOf();

        for (Iterator it = clist.iterator(); it.hasNext();) {
            Concept c1 = (Concept) it.next();
            if ((c1.getCType() == TermUtil.C_SOME)
//                || ((c1.getCType() == TermUtil.C_MIN) && ((MinCardinality) c1).getValRange() == 1)
                ) {

                // otherwise it will not influent any other constructs
                for (Iterator iter = clist.iterator(); iter.hasNext();) {
                    Concept c2 = (Concept) iter.next();

                    if (c2 != c1) {
                        switch (c2.getCType()) {
                        case TermUtil.C_ALL:
                        case TermUtil.C_SOME:
                        case TermUtil.C_MIN:
                        case TermUtil.C_CARD:
                        case TermUtil.C_MAX:
                            Restriction rest = (Restriction) c2;
                            Role p = rest.getOnProperty();
                            Role r = ((Restriction) c1).getOnProperty();

                            if (rbox.isSubPropertyOf(p, r)) {
                                Concept range = rest.getRange();
                                if (range.getCType() == TermUtil.C_UNION) {
                                    ((Union) range).addUnionOf(((Restriction) c1).getRange());
                                } else {
                                    rest.setRange(TermFactory.instance().createUnion(
                                                    range, ((Restriction) c1).getRange()));
                                }
                            }
                            break;

                        default:
                            break;
                        }
                    }
                }
            }
        }

        return c;
    }

    /**
     * 
     * @param c
     * @return (not c)
     */
    public static Concept negate(Concept c) {
        return (c.getCType() == TermUtil.C_NEGATION) ? ((Negation) c).getNot()
                : TermFactory.instance().createNegation(c);
    }

    /**
     * Normalize concept to negation normal form
     * 
     * 
     * @return normalized concept
     */
    public static Concept toNNF(Concept c) {
        Concept term = null;

        int type = (c==null) ? -1 : c.getCType();
        switch (type) {
        case TermUtil.C_NEGATION:
            Concept t = ((Negation) c).getNot();
            switch (t.getCType()) {
            case TermUtil.C_NEGATION:
                term = toNNF(((Negation) t).getNot()); // double not at front
                break;
            case TermUtil.C_ALL:
                term = TermFactory.instance().createSome(
                        ((AllValueFrom) t).getOnProperty(),
                        toNNF(negate(((AllValueFrom) t).getRange())));
                break;
            case TermUtil.C_SOME:
                term = TermFactory.instance().createAll(
                        ((SomeValueFrom) t).getOnProperty(),
                        toNNF(negate(((SomeValueFrom) t).getRange())));
                break;
            case TermUtil.C_INTERSECT:
                term = TermFactory.instance().createUnion();
                for (Iterator it = ((Intersection) t).getIntersectOf()
                        .iterator(); it.hasNext();) {
                    ((Union) term)
                            .addUnionOf(toNNF(negate((Concept) it.next())));
                }
                break;
            case TermUtil.C_UNION:
                term = TermFactory.instance().createIntersect();
                for (Iterator it = ((Union) t).getUnionOf().iterator(); it
                        .hasNext();) {
                    ((Intersection) term)
                            .addIntersectOf(toNNF(negate((Concept) it.next())));
                }
                break;
            case TermUtil.C_CARD:
            case TermUtil.C_MAX:
            case TermUtil.C_MIN:
            case TermUtil.C_VALUE:
            // to do Min,Max, hasValue...
            default:
                term = c;
            }
            break;
        case TermUtil.C_ALL:
            term = TermFactory.instance().createAll(
                    ((AllValueFrom) c).getOnProperty(),
                    toNNF(((AllValueFrom) c).getRange()));
            break;
        case TermUtil.C_SOME:
            term = TermFactory.instance().createSome(
                    ((SomeValueFrom) c).getOnProperty(),
                    toNNF(((SomeValueFrom) c).getRange()));
            break;
        case TermUtil.C_INTERSECT:
            term = TermFactory.instance().createIntersect();
            for (Iterator it = ((Intersection) c).getIntersectOf().iterator(); it
                    .hasNext();) {
                ((Intersection) term)
                        .addIntersectOf(toNNF((Concept) it.next()));
            }
            break;
        case TermUtil.C_UNION:
            term = TermFactory.instance().createUnion();
            for (Iterator it = ((Union) c).getUnionOf().iterator(); it
                    .hasNext();) {
                ((Union) term).addUnionOf(toNNF((Concept) it.next()));
            }
            break;
        default:
            term = c;
        // to do Min,Max, hasValue...

        }

        return term;
    }

    public static String toString(Term t) {
        StringBuffer buffer = new StringBuffer();
        List l = null;
        String[] tmp = null;

        if (t != null) {
            switch (t.getType()) {
            case TermUtil.T_ROLE:
                buffer.append(((Role) t).getName());
                break;

            case TermUtil.T_RESOURCE:
                if (((Resource) t).getRType() == TermUtil.R_IND)
                    buffer.append(((Individual) t).getName());
                break;

            case TermUtil.T_CONCEPT:
                switch (((Concept) t).getCType()) {
                case TermUtil.C_PRIM:
                    buffer.append(((PrimitiveConcept) t).getName());
                    break;

                case TermUtil.C_NEGATION:
                    buffer.append("(NOT ");
                    buffer.append(toString(((Negation) t).getNot()));
                    buffer.append(")");
                    break;

                case TermUtil.C_INTERSECT:
                    buffer.append("(AND");
                    l = ((Intersection) t).getIntersectOf();
                    tmp = new String[l.size()];
                    for (int i = 0; i < tmp.length; i++) {
                        tmp[i] = toString((Concept) l.get(i));
                    }
                    Arrays.sort(tmp);
                    for (int i = 0; i < tmp.length; i++) {
                        buffer.append(" ");
                        buffer.append(tmp[i]);
                    }
                    buffer.append(")");
                    break;

                case TermUtil.C_UNION:
                    buffer.append("(OR");
                    l = ((Union) t).getUnionOf();
                    tmp = new String[l.size()];
                    for (int i = 0; i < tmp.length; i++) {
                        tmp[i] = toString((Concept) l.get(i));
                    }
                    Arrays.sort(tmp);
                    for (int i = 0; i < tmp.length; i++) {
                        buffer.append(" ");
                        buffer.append(tmp[i]);
                    }
                    buffer.append(")");
                    break;

                case TermUtil.C_SOME:
                    buffer.append("(SOME ");
                    buffer.append(((SomeValueFrom) t).getOnProperty().getName());
                    buffer.append(" ");
                    buffer.append(toString(((SomeValueFrom) t).getRange()));
                    buffer.append(")");
                    break;

                case TermUtil.C_ALL:
                    buffer.append("(ALL ");
                    buffer.append(((AllValueFrom) t).getOnProperty().getName());
                    buffer.append(" ");
                    buffer.append(toString(((AllValueFrom) t).getRange()));
                    buffer.append(")");
                    break;

                case TermUtil.C_MIN:
                    buffer.append("(MIN ");
                    buffer.append(((MinCardinality) t).getValRange());
                    buffer.append(" ");
                    buffer.append(((MinCardinality) t).getOnProperty().getName());
                    buffer.append(" ");
                    buffer.append(toString(((MinCardinality) t).getRange()));
                    buffer.append(")");
                    break;

                case TermUtil.C_MAX:
                    buffer.append("(MAX ");
                    buffer.append(((MaxCardinality) t).getValRange());
                    buffer.append(" ");
                    buffer.append(((MaxCardinality) t).getOnProperty().getName());
                    buffer.append(" ");
                    buffer.append(toString(((MaxCardinality) t).getRange()));
                    buffer.append(")");
                    break;

                case TermUtil.C_CARD:
                    buffer.append("(Card ");
                    buffer.append(((Cardinality) t).getValRange());
                    buffer.append(" ");
                    buffer.append(((Cardinality) t).getOnProperty().getName());
                    buffer.append(" ");
                    buffer.append(toString(((Cardinality) t).getRange()));
                    buffer.append(")");
                    break;

                case TermUtil.C_VALUE:
                    buffer.append("(HasValue ");
                    buffer.append(((HasValue) t).getOnProperty().getName());
                    buffer.append(" ");
                    buffer.append(toString(((HasValue) t).getHasValue()));
                    buffer.append(")");
                    break;

                }
                break;
            }
        }

        return buffer.toString();
    }

    public static Concept cloneConcept(Concept c) {
        Concept clone = null;

        int type = (c==null) ? -1 : c.getCType();
        
        switch (type) {
        case TermUtil.C_NEGATION:
            clone = TermFactory.instance().createNegation(
                    ((Negation) c).getNot());
            break;

        case TermUtil.C_ALL:
            clone = TermFactory.instance().createAll(
                    ((AllValueFrom) c).getOnProperty(),
                    cloneConcept(((AllValueFrom) c).getRange()));
            break;

        case TermUtil.C_SOME:
            clone = TermFactory.instance().createSome(
                    ((SomeValueFrom) c).getOnProperty(),
                    cloneConcept(((SomeValueFrom) c).getRange()));
            break;

        case TermUtil.C_CARD:
            clone = TermFactory.instance().createCardinality(
                    ((Cardinality) c).getOnProperty(),
                    ((Cardinality) c).getValRange(),
                    cloneConcept(((Cardinality) c).getRange()));
            break;

        case TermUtil.C_MAX:
            clone = TermFactory.instance().createMaxCardinality(
                    ((MaxCardinality) c).getOnProperty(),
                    ((MaxCardinality) c).getValRange(),
                    cloneConcept(((MaxCardinality) c).getRange()));
            break;

        case TermUtil.C_MIN:
            clone = TermFactory.instance().createMinCardinality(
                    ((MinCardinality) c).getOnProperty(),
                    ((MinCardinality) c).getValRange(),
                    cloneConcept(((MinCardinality) c).getRange()));
            break;

        case TermUtil.C_VALUE: // c.getHasValue will not clone
            clone = TermFactory.instance().createHasValue(
                    ((HasValue) c).getOnProperty(),
                    ((HasValue) c).getHasValue());
            break;

        case TermUtil.C_INTERSECT:
            clone = TermFactory.instance().createIntersect();
            for (Iterator it = ((Intersection) c).getIntersectOf().iterator(); it
                    .hasNext();) {
                ((Intersection) clone).addIntersectOf(cloneConcept((Concept) it
                        .next()));
            }
            break;

        case TermUtil.C_UNION:
            clone = TermFactory.instance().createUnion();
            for (Iterator it = ((Union) c).getUnionOf().iterator(); it
                    .hasNext();) {
                ((Union) clone).addUnionOf(cloneConcept((Concept) it.next()));
            }
            break;

        default: // Prim
            clone = c;

        }

        return clone;
    }


    public static final boolean DEBUG = true;

}
