/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.internal.logging.aeri.ide.utils;

import static java.util.Objects.requireNonNull;
import static org.apache.commons.lang3.SystemUtils.getUserHome;

import java.io.File;
import java.io.IOException;
import java.util.UUID;

import org.eclipse.jdt.annotation.Nullable;

import com.google.common.base.Charsets;
import com.google.common.base.Throwables;
import com.google.common.io.Files;

public class AnonymousId {

    @Nullable
    private static UUID anonymousId;

    public static synchronized UUID getId() {
        if (anonymousId == null) {
            try {
                anonymousId = readOrCreateUUID();
            } catch (IOException e) {
                Throwables.propagate(e);
            }
        }
        return requireNonNull(anonymousId);
    }

    private static UUID readOrCreateUUID() throws IOException {
        File idFile = new File(getUserHome(), ".eclipse/org.eclipse.epp.logging.aeri/anonymousId"); //$NON-NLS-1$
        if (idFile.exists()) {
            String uuid = Files.readFirstLine(idFile, Charsets.UTF_8);
            return UUID.fromString(uuid);
        } else {
            idFile.getParentFile().mkdirs();
            File oldIdFile = new File(getUserHome(), ".eclipse/org.eclipse.recommenders/anonymousId"); //$NON-NLS-1$
            if (!oldIdFile.exists()) {
                UUID uuid = UUID.randomUUID();
                Files.write(uuid.toString(), idFile, Charsets.UTF_8);
                return uuid;
            } else {
                // we moved the code to epp. Change that file location as well:
                idFile.getParentFile().mkdirs();
                Files.copy(oldIdFile, idFile);
                String uuid = Files.readFirstLine(idFile, Charsets.UTF_8);
                return UUID.fromString(uuid);
            }
        }
    }
}
