/* --------------------------------------------------------------------------------------------
 * Copyright (c) Remy Suen. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for license information.
 * ------------------------------------------------------------------------------------------ */
'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
const vscode_languageserver_types_1 = require("vscode-languageserver-types");
const dockerfile_ast_1 = require("dockerfile-ast");
class DockerFormatter {
    getIndentation(formattingOptions) {
        let indentation = "\t";
        if (formattingOptions && formattingOptions.insertSpaces) {
            indentation = "";
            for (let i = 0; i < formattingOptions.tabSize; i++) {
                indentation = indentation + " ";
            }
        }
        return indentation;
    }
    /**
     * Creates a TextEdit for formatting the given document.
     *
     * @param document the document being formatted
     * @param start the start offset of the document's content to be replaced
     * @param end the end offset of the document's content to be replaced
     * @param indent true if this block should be replaced with an indentation, false otherwise
     * @param indentation the string to use for an indentation
     */
    createFormattingEdit(document, start, end, indent, indentation) {
        if (indent) {
            return vscode_languageserver_types_1.TextEdit.replace({
                start: document.positionAt(start),
                end: document.positionAt(end)
            }, indentation);
        }
        else {
            return vscode_languageserver_types_1.TextEdit.del({
                start: document.positionAt(start),
                end: document.positionAt(end)
            });
        }
    }
    formatOnType(document, position, ch, options) {
        const dockerfile = dockerfile_ast_1.DockerfileParser.parse(document.getText());
        // check that the inserted character is the escape character
        if (dockerfile.getEscapeCharacter() === ch) {
            for (let comment of dockerfile.getComments()) {
                // ignore if we're in a comment
                if (comment.getRange().start.line === position.line) {
                    return [];
                }
            }
            const directive = dockerfile.getDirective();
            // ignore if we're in the parser directive
            if (directive && position.line === 0) {
                return [];
            }
            const content = document.getText();
            validityCheck: for (let i = document.offsetAt(position); i < content.length; i++) {
                switch (content.charAt(i)) {
                    case ' ':
                    case '\t':
                        break;
                    case '\r':
                    case '\n':
                        break validityCheck;
                    default:
                        // not escaping a newline, no need to format the next line
                        return [];
                }
            }
            const lines = [position.line + 1];
            const indentedLines = [];
            indentedLines[lines[0]] = true;
            return this.formatLines(document, document.getText(), lines, indentedLines, options);
        }
        return [];
    }
    formatRange(document, range, options) {
        const lines = [];
        for (let i = range.start.line; i <= range.end.line; i++) {
            lines.push(i);
        }
        return this.format(document, lines, options);
    }
    formatDocument(document, options) {
        const lines = [];
        for (let i = 0; i < document.lineCount; i++) {
            lines.push(i);
        }
        return this.format(document, lines, options);
    }
    /**
     * Formats the specified lines of the given document based on the
     * provided formatting options.
     *
     * @param document the text document to format
     * @param lines the lines to format
     * @param options the formatting options to use to perform the format
     * @return the text edits to apply to format the lines of the document
     */
    format(document, lines, options) {
        let content = document.getText();
        let dockerfile = dockerfile_ast_1.DockerfileParser.parse(content);
        const indentedLines = [];
        for (let i = 0; i < document.lineCount; i++) {
            indentedLines[i] = false;
        }
        for (let instruction of dockerfile.getInstructions()) {
            let range = instruction.getRange();
            indentedLines[range.start.line] = false;
            for (let i = range.start.line + 1; i <= range.end.line; i++) {
                indentedLines[i] = true;
            }
        }
        return this.formatLines(document, content, lines, indentedLines, options);
    }
    formatLines(document, content, lines, indentedLines, options) {
        const indentation = this.getIndentation(options);
        const edits = [];
        lineCheck: for (let line of lines) {
            let startOffset = document.offsetAt(vscode_languageserver_types_1.Position.create(line, 0));
            for (let i = startOffset; i < content.length; i++) {
                switch (content.charAt(i)) {
                    case ' ':
                    case '\t':
                        break;
                    case '\r':
                    case '\n':
                        if (i !== startOffset) {
                            // only whitespace on this line, trim it
                            let edit = vscode_languageserver_types_1.TextEdit.del({
                                start: document.positionAt(startOffset),
                                end: document.positionAt(i)
                            });
                            edits.push(edit);
                        }
                        // process the next line
                        continue lineCheck;
                    default:
                        // non-whitespace encountered
                        if (i !== startOffset || indentedLines[line]) {
                            let edit = this.createFormattingEdit(document, startOffset, i, indentedLines[line], indentation);
                            edits.push(edit);
                        }
                        // process the next line
                        continue lineCheck;
                }
            }
            if (startOffset < content.length) {
                // only whitespace on the last line, trim it
                let edit = vscode_languageserver_types_1.TextEdit.del({
                    start: document.positionAt(startOffset),
                    end: document.positionAt(content.length)
                });
                edits.push(edit);
            }
        }
        return edits;
    }
}
exports.DockerFormatter = DockerFormatter;
