import { TextDocument, Range, Position, Diagnostic, DiagnosticSeverity } from 'vscode-languageserver-types';
import { Dockerfile } from 'dockerfile-ast';
import { ValidationCode, ValidatorSettings } from './main';
export declare const KEYWORDS: string[];
export declare class Validator {
    private document;
    private settings;
    constructor(settings?: ValidatorSettings);
    parseDirective(dockerfile: Dockerfile, problems: Diagnostic[]): void;
    /**
     * Checks the arguments of the given instruction.
     *
     * @param instruction the instruction to validate
     * @param problems an array of identified problems in the document
     * @param expectedArgCount an array of expected number of arguments
     *                         for the instruction, if its length is 1
     *                         and its value is -1, any number of
     *                         arguments greater than zero is valid
     * @param validate the function to use to validate an argument
     * @param createIncompleteDiagnostic the function to use to create a diagnostic
     *                                   if the number of arguments is incorrect
     */
    private checkArguments(instruction, problems, expectedArgCount, validate, createIncompleteDiagnostic?);
    private checkVariables(instruction, problems);
    private checkProperty(document, escapeChar, keyword, property, firstProperty, optionalValue, problems);
    validate(document: TextDocument): Diagnostic[];
    private validateInstruction(document, escapeChar, instruction, keyword, isTrigger, problems);
    private checkDestinationIsDirectory(instruction, requiresTwoArgumentsFunction, notDirectoryFunction);
    private checkJsonDestinationIsDirectory(instruction, requiresTwoArgumentsFunction, notDirectoryFunction);
    private checkFlagValue(flags, validFlagNames, problems);
    private checkFlagDuration(flags, validFlagNames, problems);
    private static isNumberRelated(character);
    private checkDuplicateFlags(flags, validFlags, problems);
    private checkJSON(document, instruction, problems);
    private checkJSONQuotes(instruction, problems);
    private static dockerProblems;
    private static formatMessage(text, ...variables);
    static getDiagnosticMessage_DirectiveCasing(): string;
    static getDiagnosticMessage_DirectiveEscapeInvalid(value: string): string;
    static getDiagnosticMessage_NoSourceImage(): string;
    static getDiagnosticMessage_EmptyContinuationLine(): string;
    static getDiagnosticMessage_DuplicateBuildStageName(name: string): string;
    static getDiagnosticMessage_InvalidBuildStageName(name: string): string;
    static getDiagnosticMessage_FlagAtLeastOne(flagName: string, flagValue: string): string;
    static getDiagnosticMessage_FlagDuplicate(flag: string): string;
    static getDiagnosticMessage_FlagInvalidDuration(flag: string): string;
    static getDiagnosticMessage_FlagLessThan1ms(flag: string): string;
    static getDiagnosticMessage_FlagMissingDuration(duration: string): string;
    static getDiagnosticMessage_FlagInvalidFromValue(value: string): string;
    static getDiagnosticMessage_FlagMissingValue(flag: string): string;
    static getDiagnosticMessage_FlagUnknown(flag: string): string;
    static getDiagnosticMessage_FlagUnknownUnit(unit: string, duration: string): string;
    static getDiagnosticMessage_BaseNameEmpty(name: string): string;
    static getDiagnosticMessage_InvalidAs(): string;
    static getDiagnosticMessage_InvalidPort(port: string): string;
    static getDiagnosticMessage_InvalidProto(protocol: string): string;
    static getDiagnosticMessage_InvalidReferenceFormat(): string;
    static getDiagnosticMessage_InvalidSignal(signal: string): string;
    static getDiagnosticMessage_InvalidSyntax(syntax: string): string;
    static getDiagnosticMessage_InstructionExtraArgument(): string;
    static getDiagnosticMessage_InstructionMissingArgument(): string;
    static getDiagnosticMessage_ADDDestinationNotDirectory(): string;
    static getDiagnosticMessage_ADDRequiresAtLeastTwoArguments(): string;
    static getDiagnosticMessage_ARGRequiresOneArgument(): string;
    static getDiagnosticMessage_COPYDestinationNotDirectory(): string;
    static getDiagnosticMessage_COPYRequiresAtLeastTwoArguments(): string;
    static getDiagnosticMessage_HEALTHCHECKRequiresAtLeastOneArgument(): string;
    static getDiagnosticMessage_ENVRequiresTwoArguments(): string;
    static getDiagnosticMessage_InstructionRequiresOneOrThreeArguments(): string;
    static getDiagnosticMessage_HealthcheckNoneUnnecessaryArgument(): string;
    static getDiagnosticMessage_InstructionMultiple(instruction: string): string;
    static getDiagnosticMessage_InstructionUnknown(instruction: string): string;
    static getDiagnosticMessage_SyntaxMissingEquals(argument: string): string;
    static getDiagnosticMessage_SyntaxMissingNames(instruction: string): string;
    static getDiagnosticMessage_SyntaxMissingSingleQuote(key: string): string;
    static getDiagnosticMessage_SyntaxMissingDoubleQuote(key: string): string;
    static getDiagnosticMessage_InstructionCasing(): string;
    static getDiagnosticMessage_InstructionJSONInSingleQuotes(): string;
    static getDiagnosticMessage_OnbuildChainingDisallowed(): string;
    static getDiagnosticMessage_OnbuildTriggerDisallowed(trigger: string): string;
    static getDiagnosticMessage_VariableModifierUnsupported(variable: string, modifier: string): string;
    static getDiagnosticMessage_ShellJsonForm(): string;
    static getDiagnosticMessage_ShellRequiresOne(): string;
    static getDiagnosticMessage_DeprecatedMaintainer(): string;
    static getDiagnosticMessage_HealthcheckCmdArgumentMissing(): string;
    static getDiagnosticMessage_HealthcheckTypeUnknown(type: string): string;
    static getDiagnosticMessage_WORKDIRPathNotAbsolute(): string;
    static createInvalidEscapeDirective(start: Position, end: Position, value: string): Diagnostic;
    private static createDuplicateBuildStageName(range, name);
    private static createInvalidBuildStageName(range, name);
    static createFlagAtLeastOne(start: Position, end: Position, flagName: string, flagValue: string): Diagnostic;
    static createFlagDuplicate(start: Position, end: Position, flag: string): Diagnostic;
    private static createFlagInvalidDuration(start, end, flag);
    private static createFlagLessThan1ms(start, end, flag);
    private static createFlagMissingDuration(start, end, duration);
    private static createFlagInvalidFrom(start, end, flag);
    static createFlagMissingValue(start: Position, end: Position, flag: string): Diagnostic;
    static createUnknownAddFlag(start: Position, end: Position, flag: string): Diagnostic;
    static createUnknownCopyFlag(start: Position, end: Position, flag: string): Diagnostic;
    static createUnknownHealthcheckFlag(start: Position, end: Position, flag: string): Diagnostic;
    private static createFlagUnknownUnit(range, unit, duration);
    private static createBaseNameEmpty(range, name);
    static createInvalidAs(start: Position, end: Position): Diagnostic;
    static createInvalidPort(range: Range, port: string): Diagnostic;
    private static createInvalidProto(start, end, protocol);
    private static createInvalidReferenceFormat(range);
    static createInvalidStopSignal(start: Position, end: Position, signal: string): Diagnostic;
    static createInvalidSyntax(start: Position, end: Position, syntax: string): Diagnostic;
    static createMissingArgument(start: Position, end: Position): Diagnostic;
    static createExtraArgument(start: Position, end: Position): Diagnostic;
    private static createHealthcheckNoneUnnecessaryArgument(start, end);
    static createARGRequiresOneArgument(start: Position, end: Position): Diagnostic;
    private static createADDDestinationNotDirectory(range);
    private static createADDRequiresAtLeastTwoArguments(range);
    private static createCOPYDestinationNotDirectory(range);
    private static createCOPYRequiresAtLeastTwoArguments(range);
    static createENVRequiresTwoArguments(start: Position, end: Position): Diagnostic;
    private static createHEALTHCHECKRequiresAtLeastOneArgument(range);
    private static createHealthcheckCmdArgumentMissing(start, end);
    private static createHealthcheckTypeUnknown(range, type);
    private static createOnbuildChainingDisallowed(range);
    private static createOnbuildTriggerDisallowed(range, trigger);
    private static createShellJsonForm(range);
    private static createShellRequiresOne(range);
    static createRequiresOneOrThreeArguments(start: Position, end: Position): Diagnostic;
    static createNoSourceImage(start: Position, end: Position): Diagnostic;
    static createSyntaxMissingEquals(start: Position, end: Position, argument: string): Diagnostic;
    private static createSyntaxMissingSingleQuote(start, end, argument);
    private static createSyntaxMissingDoubleQuote(start, end, argument);
    private static createSyntaxMissingNames(start, end, instruction);
    private static createVariableUnsupportedModifier(range, variable, modifier);
    static createUnknownInstruction(start: Position, end: Position, instruction: string): Diagnostic;
    static createError(start: Position, end: Position, description: string, code?: ValidationCode): Diagnostic;
    private static createJSONInSingleQuotes(range, severity);
    private static createEmptyContinuationLine(start, end, severity);
    private createMultipleInstructions(range, severity, instruction);
    private createLowercaseDirective(start, end);
    createUppercaseInstruction(start: Position, end: Position): Diagnostic | null;
    createMaintainerDeprecated(start: Position, end: Position): Diagnostic | null;
    private createWORKDIRNotAbsolute(range);
    static createWarning(start: Position, end: Position, description: string, code?: ValidationCode): Diagnostic;
    static createDiagnostic(severity: DiagnosticSeverity, start: Position, end: Position, description: string, code?: ValidationCode): Diagnostic;
}
