'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
function findArgument(argv, argName) {
    const index = argv.indexOf(argName);
    if (index < 0 || index === argv.length - 1) {
        return;
    }
    return argv[index + 1];
}
function parseStringArray(argv, argName) {
    const arg = findArgument(argv, argName);
    if (!arg) {
        return [];
    }
    return arg.split(',');
}
function hasArgument(argv, argName) {
    return argv.includes(argName);
}
function parseCommandLine(argv) {
    return {
        help: hasArgument(argv, '--help'),
        logFile: findArgument(argv, '--logFile'),
        logVerbosity: findArgument(argv, '--logVerbosity'),
        ngProbeLocations: parseStringArray(argv, '--ngProbeLocations'),
        tsProbeLocations: parseStringArray(argv, '--tsProbeLocations'),
    };
}

/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const MIN_TS_VERSION = '3.6';
function resolve(packageName, location, rootPackage) {
    rootPackage = rootPackage || packageName;
    try {
        const packageJsonPath = require.resolve(`${rootPackage}/package.json`, {
            paths: [location],
        });
        const packageJson = require(packageJsonPath);
        const resolvedPath = require.resolve(packageName, {
            paths: [location],
        });
        return {
            resolvedPath,
            version: new Version(packageJson.version),
        };
    }
    catch (_a) {
    }
}
/**
 * Resolve the node module with the specified `packageName` that satisfies
 * the specified minimum version.
 * @param packageName name of package to be resolved
 * @param minVersionStr minimum version
 * @param probeLocations locations to initiate node module resolution
 * @param rootPackage location of package.json, if different from `packageName`
 */
function resolveWithMinVersion(packageName, minVersionStr, probeLocations, rootPackage) {
    if (rootPackage && !packageName.startsWith(rootPackage)) {
        throw new Error(`${packageName} must be in the root package`);
    }
    const minVersion = new Version(minVersionStr);
    for (const location of probeLocations) {
        const nodeModule = resolve(packageName, location, rootPackage);
        if (nodeModule && nodeModule.version.greaterThanOrEqual(minVersion)) {
            return nodeModule;
        }
    }
    throw new Error(`Failed to resolve '${packageName}' with minimum version '${minVersion}' from ` +
        JSON.stringify(probeLocations, null, 2));
}
/**
 * Resolve `typescript/lib/tsserverlibrary` from the given locations.
 * @param probeLocations
 */
function resolveTsServer(probeLocations) {
    const tsserver = 'typescript/lib/tsserverlibrary';
    return resolveWithMinVersion(tsserver, MIN_TS_VERSION, probeLocations, 'typescript');
}
/**
 * Converts the specified string `a` to non-negative integer.
 * Returns -1 if the result is NaN.
 * @param a
 */
function parseNonNegativeInt(a) {
    // parseInt() will try to convert as many as possible leading characters that
    // are digits. This means a string like "123abc" will be converted to 123.
    // For our use case, this is sufficient.
    const i = parseInt(a, 10 /* radix */);
    return isNaN(i) ? -1 : i;
}
class Version {
    constructor(versionStr) {
        this.versionStr = versionStr;
        const [major, minor, patch] = Version.parseVersionStr(versionStr);
        this.major = major;
        this.minor = minor;
        this.patch = patch;
    }
    greaterThanOrEqual(other) {
        if (this.major < other.major) {
            return false;
        }
        if (this.major > other.major) {
            return true;
        }
        if (this.minor < other.minor) {
            return false;
        }
        if (this.minor > other.minor) {
            return true;
        }
        return this.patch >= other.patch;
    }
    toString() {
        return this.versionStr;
    }
    /**
     * Converts the specified `versionStr` to its number constituents. Invalid
     * number value is represented as negative number.
     * @param versionStr
     */
    static parseVersionStr(versionStr) {
        const [major, minor, patch] = versionStr.split('.').map(parseNonNegativeInt);
        return [
            major === undefined ? 0 : major,
            minor === undefined ? 0 : minor,
            patch === undefined ? 0 : patch,
        ];
    }
}

/**
 * This method provides a custom implementation for the AMD loader to resolve
 * `typescript` module at runtime.
 * @param modules modules to resolve
 * @param cb function to invoke with resolved modules
 */
function define(modules, cb) {
    const TSSERVER = 'typescript/lib/tsserverlibrary';
    const resolvedModules = modules.map(m => {
        if (m === 'typescript') {
            throw new Error(`Import '${TSSERVER}' instead of 'typescript'`);
        }
        if (m === TSSERVER) {
            const { tsProbeLocations } = parseCommandLine(process.argv);
            m = resolveTsServer(tsProbeLocations).resolvedPath;
        }
        return require(m);
    });
    cb(...resolvedModules);
}

exports.define = define;
