"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const vscode_uri_1 = require("vscode-uri");
const request_light_1 = require("request-light");
const fs = require("fs");
const requestTypes_1 = require("../../requestTypes");
const paths_1 = require("../utils/paths");
/**
 * Handles schema content requests given the schema URI
 * @param uri can be a local file, vscode request, http(s) request or a custom request
 */
exports.schemaRequestHandler = (connection, uri) => {
    if (!uri) {
        return Promise.reject('No schema specified');
    }
    // If the requested schema URI is a relative file path
    // Convert it into a proper absolute path URI
    if (paths_1.isRelativePath(uri)) {
        uri = paths_1.relativeToAbsolutePath(this.workspaceFolders, this.workspaceRoot, uri);
    }
    const scheme = vscode_uri_1.URI.parse(uri).scheme.toLowerCase();
    // If the requested schema is a local file, read and return the file contents
    if (scheme === 'file') {
        const fsPath = vscode_uri_1.URI.parse(uri).fsPath;
        return new Promise((c, e) => {
            fs.readFile(fsPath, 'UTF-8', (err, result) => 
            // If there was an error reading the file, return empty error message
            // Otherwise return the file contents as a string
            err ? e('') : c(result.toString()));
        });
    }
    // vscode schema content requests are forwarded to the client through LSP
    // This is a non-standard LSP extension introduced by the JSON language server
    // See https://github.com/microsoft/vscode/blob/master/extensions/json-language-features/server/README.md
    if (scheme === 'vscode') {
        return connection.sendRequest(requestTypes_1.VSCodeContentRequest.type, uri)
            .then(responseText => responseText, error => error.message);
    }
    // HTTP(S) requests are sent and the response result is either the schema content or an error
    if (scheme === 'http' || scheme === 'https') {
        // If it's an HTTP(S) request to Microsoft Azure, log the request
        if (uri.indexOf('//schema.management.azure.com/') !== -1) {
            connection.telemetry.logEvent({
                key: 'json.schema',
                value: {
                    schemaURL: uri
                }
            });
        }
        // Send the HTTP(S) schema content request and return the result
        const headers = { 'Accept-Encoding': 'gzip, deflate' };
        return request_light_1.xhr({ url: uri, followRedirects: 5, headers })
            .then(response => response.responseText, (error) => Promise.reject(error.responseText || request_light_1.getErrorStatusDescription(error.status) || error.toString()));
    }
    // Neither local file nor vscode, nor HTTP(S) schema request, so send it off as a custom request
    return connection.sendRequest(requestTypes_1.CustomSchemaContentRequest.type, uri);
};
//# sourceMappingURL=schemaRequestHandler.js.map