package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link ConfigFile}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableConfigFile.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableConfigFile implements ConfigFile {
  private final String name;
  private final @Nullable String uid;
  private final @Nullable String gid;
  private final @Nullable Long mode;

  private ImmutableConfigFile(
      String name,
      @Nullable String uid,
      @Nullable String gid,
      @Nullable Long mode) {
    this.name = name;
    this.uid = uid;
    this.gid = gid;
    this.mode = mode;
  }

  /**
   * @return The value of the {@code name} attribute
   */
  @JsonProperty("Name")
  @Override
  public String name() {
    return name;
  }

  /**
   * @return The value of the {@code uid} attribute
   */
  @JsonProperty("UID")
  @Override
  public @Nullable String uid() {
    return uid;
  }

  /**
   * @return The value of the {@code gid} attribute
   */
  @JsonProperty("GID")
  @Override
  public @Nullable String gid() {
    return gid;
  }

  /**
   * @return The value of the {@code mode} attribute
   */
  @JsonProperty("Mode")
  @Override
  public @Nullable Long mode() {
    return mode;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ConfigFile#name() name} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for name
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableConfigFile withName(String value) {
    String newValue = Objects.requireNonNull(value, "name");
    if (this.name.equals(newValue)) return this;
    return new ImmutableConfigFile(newValue, this.uid, this.gid, this.mode);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ConfigFile#uid() uid} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for uid (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableConfigFile withUid(@Nullable String value) {
    if (Objects.equals(this.uid, value)) return this;
    return new ImmutableConfigFile(this.name, value, this.gid, this.mode);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ConfigFile#gid() gid} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for gid (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableConfigFile withGid(@Nullable String value) {
    if (Objects.equals(this.gid, value)) return this;
    return new ImmutableConfigFile(this.name, this.uid, value, this.mode);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ConfigFile#mode() mode} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for mode (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableConfigFile withMode(@Nullable Long value) {
    if (Objects.equals(this.mode, value)) return this;
    return new ImmutableConfigFile(this.name, this.uid, this.gid, value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableConfigFile} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableConfigFile
        && equalTo((ImmutableConfigFile) another);
  }

  private boolean equalTo(ImmutableConfigFile another) {
    return name.equals(another.name)
        && Objects.equals(uid, another.uid)
        && Objects.equals(gid, another.gid)
        && Objects.equals(mode, another.mode);
  }

  /**
   * Computes a hash code from attributes: {@code name}, {@code uid}, {@code gid}, {@code mode}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + name.hashCode();
    h += (h << 5) + Objects.hashCode(uid);
    h += (h << 5) + Objects.hashCode(gid);
    h += (h << 5) + Objects.hashCode(mode);
    return h;
  }

  /**
   * Prints the immutable value {@code ConfigFile} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "ConfigFile{"
        + "name=" + name
        + ", uid=" + uid
        + ", gid=" + gid
        + ", mode=" + mode
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link ConfigFile} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable ConfigFile instance
   */
  public static ImmutableConfigFile copyOf(ConfigFile instance) {
    if (instance instanceof ImmutableConfigFile) {
      return (ImmutableConfigFile) instance;
    }
    return ImmutableConfigFile.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableConfigFile ImmutableConfigFile}.
   * <pre>
   * ImmutableConfigFile.builder()
   *    .name(String) // required {@link ConfigFile#name() name}
   *    .uid(String | null) // nullable {@link ConfigFile#uid() uid}
   *    .gid(String | null) // nullable {@link ConfigFile#gid() gid}
   *    .mode(Long | null) // nullable {@link ConfigFile#mode() mode}
   *    .build();
   * </pre>
   * @return A new ImmutableConfigFile builder
   */
  public static ImmutableConfigFile.Builder builder() {
    return new ImmutableConfigFile.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableConfigFile ImmutableConfigFile}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements ConfigFile.Builder {
    private static final long INIT_BIT_NAME = 0x1L;
    private long initBits = 0x1L;

    private String name;
    private String uid;
    private String gid;
    private Long mode;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code ConfigFile} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(ConfigFile instance) {
      Objects.requireNonNull(instance, "instance");
      name(instance.name());
      @Nullable String uidValue = instance.uid();
      if (uidValue != null) {
        uid(uidValue);
      }
      @Nullable String gidValue = instance.gid();
      if (gidValue != null) {
        gid(gidValue);
      }
      @Nullable Long modeValue = instance.mode();
      if (modeValue != null) {
        mode(modeValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link ConfigFile#name() name} attribute.
     * @param name The value for name 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Name")
    public final Builder name(String name) {
      this.name = Objects.requireNonNull(name, "name");
      initBits &= ~INIT_BIT_NAME;
      return this;
    }

    /**
     * Initializes the value for the {@link ConfigFile#uid() uid} attribute.
     * @param uid The value for uid (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("UID")
    public final Builder uid(@Nullable String uid) {
      this.uid = uid;
      return this;
    }

    /**
     * Initializes the value for the {@link ConfigFile#gid() gid} attribute.
     * @param gid The value for gid (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("GID")
    public final Builder gid(@Nullable String gid) {
      this.gid = gid;
      return this;
    }

    /**
     * Initializes the value for the {@link ConfigFile#mode() mode} attribute.
     * @param mode The value for mode (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Mode")
    public final Builder mode(@Nullable Long mode) {
      this.mode = mode;
      return this;
    }

    /**
     * Builds a new {@link ImmutableConfigFile ImmutableConfigFile}.
     * @return An immutable instance of ConfigFile
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableConfigFile build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableConfigFile(name, uid, gid, mode);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_NAME) != 0) attributes.add("name");
      return "Cannot build ConfigFile, some of required attributes are not set " + attributes;
    }
  }
}
