/**
 * Copyright (c) 2010-2012, Mark Czotter, Istvan Rath and Daniel Varro
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-v20.html.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.viatra.query.patternlanguage.emf.jvmmodel;

import com.google.inject.Inject;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.viatra.query.patternlanguage.emf.jvmmodel.EMFPatternLanguageJvmModelInferrerUtil;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Pattern;
import org.eclipse.viatra.query.patternlanguage.emf.vql.PatternModel;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Variable;
import org.eclipse.viatra.query.runtime.api.IQueryGroup;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.naming.IQualifiedNameProvider;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * @noreference
 */
@SuppressWarnings("all")
public class JavadocInferrer {
  @Inject
  @Extension
  private EMFPatternLanguageJvmModelInferrerUtil _eMFPatternLanguageJvmModelInferrerUtil;
  
  @Inject
  @Extension
  private IQualifiedNameProvider _iQualifiedNameProvider;
  
  /**
   * Infers javadoc for Match class based on the input 'pattern'.
   */
  public CharSequence javadocMatchClass(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Pattern-specific match representation of the ");
    QualifiedName _fullyQualifiedName = this._iQualifiedNameProvider.getFullyQualifiedName(pattern);
    _builder.append(_fullyQualifiedName);
    _builder.append(" pattern,");
    _builder.newLineIfNotEmpty();
    _builder.append("to be used in conjunction with {@link ");
    JvmType _findMatcherClass = this._eMFPatternLanguageJvmModelInferrerUtil.findMatcherClass(pattern);
    String _simpleName = null;
    if (_findMatcherClass!=null) {
      _simpleName=_findMatcherClass.getSimpleName();
    }
    _builder.append(_simpleName);
    _builder.append("}.");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("<p>Class fields correspond to parameters of the pattern. Fields with value null are considered unassigned.");
    _builder.newLine();
    _builder.append("Each instance is a (possibly partial) substitution of pattern parameters,");
    _builder.newLine();
    _builder.append("usable to represent a match of the pattern in the result of a query,");
    _builder.newLine();
    _builder.append("or to specify the bound (fixed) input parameters when issuing a query.");
    _builder.newLine();
    _builder.newLine();
    _builder.append("@see ");
    JvmType _findMatcherClass_1 = this._eMFPatternLanguageJvmModelInferrerUtil.findMatcherClass(pattern);
    String _simpleName_1 = null;
    if (_findMatcherClass_1!=null) {
      _simpleName_1=_findMatcherClass_1.getSimpleName();
    }
    _builder.append(_simpleName_1);
    _builder.newLineIfNotEmpty();
    {
      JvmType _findInferredClass = this._eMFPatternLanguageJvmModelInferrerUtil.findInferredClass(pattern, Consumer.class);
      boolean _tripleNotEquals = (_findInferredClass != null);
      if (_tripleNotEquals) {
        _builder.append(" @see ");
        JvmType _findInferredClass_1 = this._eMFPatternLanguageJvmModelInferrerUtil.findInferredClass(pattern, Consumer.class);
        String _simpleName_2 = null;
        if (_findInferredClass_1!=null) {
          _simpleName_2=_findInferredClass_1.getSimpleName();
        }
        _builder.append(_simpleName_2);
      }
    }
    _builder.newLineIfNotEmpty();
    return _builder;
  }
  
  public CharSequence javadocMatcherClass(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Generated pattern matcher API of the ");
    QualifiedName _fullyQualifiedName = this._iQualifiedNameProvider.getFullyQualifiedName(pattern);
    _builder.append(_fullyQualifiedName);
    _builder.append(" pattern,");
    _builder.newLineIfNotEmpty();
    _builder.append("providing pattern-specific query methods.");
    _builder.newLine();
    _builder.newLine();
    _builder.append("<p>Use the pattern matcher on a given model via {@link #on(ViatraQueryEngine)},");
    _builder.newLine();
    _builder.append("e.g. in conjunction with {@link ViatraQueryEngine#on(QueryScope)}.");
    _builder.newLine();
    _builder.newLine();
    _builder.append("<p>Matches of the pattern will be represented as {@link ");
    JvmType _findMatchClass = this._eMFPatternLanguageJvmModelInferrerUtil.findMatchClass(pattern);
    String _simpleName = null;
    if (_findMatchClass!=null) {
      _simpleName=_findMatchClass.getSimpleName();
    }
    _builder.append(_simpleName);
    _builder.append("}.");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("<p>Original source:");
    _builder.newLine();
    _builder.append("<code><pre>");
    _builder.newLine();
    String _serializeToJavadoc = this._eMFPatternLanguageJvmModelInferrerUtil.serializeToJavadoc(pattern);
    _builder.append(_serializeToJavadoc);
    _builder.newLineIfNotEmpty();
    _builder.append("</pre></code>");
    _builder.newLine();
    _builder.newLine();
    _builder.append("@see ");
    JvmType _findMatchClass_1 = this._eMFPatternLanguageJvmModelInferrerUtil.findMatchClass(pattern);
    String _simpleName_1 = null;
    if (_findMatchClass_1!=null) {
      _simpleName_1=_findMatchClass_1.getSimpleName();
    }
    _builder.append(_simpleName_1);
    _builder.newLineIfNotEmpty();
    {
      JvmType _findInferredClass = this._eMFPatternLanguageJvmModelInferrerUtil.findInferredClass(pattern, Consumer.class);
      boolean _tripleNotEquals = (_findInferredClass != null);
      if (_tripleNotEquals) {
        _builder.append(" @see ");
        JvmType _findInferredClass_1 = this._eMFPatternLanguageJvmModelInferrerUtil.findInferredClass(pattern, Consumer.class);
        String _simpleName_2 = null;
        if (_findInferredClass_1!=null) {
          _simpleName_2=_findInferredClass_1.getSimpleName();
        }
        _builder.append(_simpleName_2);
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("@see ");
    JvmType _findInferredSpecification = this._eMFPatternLanguageJvmModelInferrerUtil.findInferredSpecification(pattern);
    String _simpleName_3 = null;
    if (_findInferredSpecification!=null) {
      _simpleName_3=_findInferredSpecification.getSimpleName();
    }
    _builder.append(_simpleName_3);
    _builder.newLineIfNotEmpty();
    return _builder;
  }
  
  public CharSequence javadocQuerySpecificationClass(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("A pattern-specific query specification that can instantiate ");
    JvmType _findMatcherClass = this._eMFPatternLanguageJvmModelInferrerUtil.findMatcherClass(pattern);
    String _simpleName = null;
    if (_findMatcherClass!=null) {
      _simpleName=_findMatcherClass.getSimpleName();
    }
    _builder.append(_simpleName);
    _builder.append(" in a type-safe way.");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("<p>Original source:");
    _builder.newLine();
    _builder.append("        ");
    _builder.append("<code><pre>");
    _builder.newLine();
    _builder.append("        ");
    String _serializeToJavadoc = this._eMFPatternLanguageJvmModelInferrerUtil.serializeToJavadoc(pattern);
    _builder.append(_serializeToJavadoc, "        ");
    _builder.newLineIfNotEmpty();
    _builder.append("</pre></code>");
    _builder.newLine();
    _builder.newLine();
    _builder.append("@see ");
    JvmType _findMatcherClass_1 = this._eMFPatternLanguageJvmModelInferrerUtil.findMatcherClass(pattern);
    String _simpleName_1 = null;
    if (_findMatcherClass_1!=null) {
      _simpleName_1=_findMatcherClass_1.getSimpleName();
    }
    _builder.append(_simpleName_1);
    _builder.newLineIfNotEmpty();
    _builder.append("@see ");
    JvmType _findMatchClass = this._eMFPatternLanguageJvmModelInferrerUtil.findMatchClass(pattern);
    String _simpleName_2 = null;
    if (_findMatchClass!=null) {
      _simpleName_2=_findMatchClass.getSimpleName();
    }
    _builder.append(_simpleName_2);
    _builder.newLineIfNotEmpty();
    return _builder;
  }
  
  public CharSequence javadocProcessorClass(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("A match processor tailored for the ");
    QualifiedName _fullyQualifiedName = this._iQualifiedNameProvider.getFullyQualifiedName(pattern);
    _builder.append(_fullyQualifiedName);
    _builder.append(" pattern.");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("Clients should derive an (anonymous) class that implements the abstract process().");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocMatcherConstructorEngine(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Initializes the pattern matcher within an existing VIATRA Query engine.");
    _builder.newLine();
    _builder.append("If the pattern matcher is already constructed in the engine, only a light-weight reference is returned.");
    _builder.newLine();
    _builder.newLine();
    _builder.append("@param engine the existing VIATRA Query engine in which this matcher will be created.");
    _builder.newLine();
    _builder.append("@throws ViatraQueryRuntimeException if an error occurs during pattern matcher creation");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocMatcherStaticOnEngine(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Initializes the pattern matcher within an existing VIATRA Query engine.");
    _builder.newLine();
    _builder.append("If the pattern matcher is already constructed in the engine, only a light-weight reference is returned.");
    _builder.newLine();
    _builder.newLine();
    _builder.append("@param engine the existing VIATRA Query engine in which this matcher will be created.");
    _builder.newLine();
    _builder.append("@throws ViatraQueryRuntimeException if an error occurs during pattern matcher creation");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocMatcherStaticCreate(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("@throws ViatraQueryRuntimeException if an error occurs during pattern matcher creation");
    _builder.newLine();
    _builder.append("@return an initialized matcher");
    _builder.newLine();
    _builder.append("@noreference This method is for internal matcher initialization by the framework, do not call it manually.");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocGetAllMatchesMethod(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Returns the set of all matches of the pattern that conform to the given fixed values of some parameters.");
    _builder.newLine();
    {
      EList<Variable> _parameters = pattern.getParameters();
      for(final Variable p : _parameters) {
        _builder.append("@param ");
        String _parameterName = this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
        _builder.append(_parameterName);
        _builder.append(" the fixed value of pattern parameter ");
        String _name = p.getName();
        _builder.append(_name);
        _builder.append(", or null if not bound.");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("@return matches represented as a ");
    JvmType _findMatchClass = this._eMFPatternLanguageJvmModelInferrerUtil.findMatchClass(pattern);
    String _simpleName = null;
    if (_findMatchClass!=null) {
      _simpleName=_findMatchClass.getSimpleName();
    }
    _builder.append(_simpleName);
    _builder.append(" object.");
    _builder.newLineIfNotEmpty();
    return _builder;
  }
  
  public CharSequence javadocStreamAllMatchesMethod(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Returns a stream of all matches of the pattern that conform to the given fixed values of some parameters.");
    _builder.newLine();
    _builder.append("</p>");
    _builder.newLine();
    _builder.append("<strong>NOTE</strong>: It is important not to modify the source model while the stream is being processed.");
    _builder.newLine();
    _builder.append("If the match set of the pattern changes during processing, the contents of the stream is <strong>undefined</strong>.");
    _builder.newLine();
    _builder.append("In such cases, either rely on {@link #getAllMatches()} or collect the results of the stream in end-user code.");
    _builder.newLine();
    {
      EList<Variable> _parameters = pattern.getParameters();
      for(final Variable p : _parameters) {
        _builder.append("@param ");
        String _parameterName = this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
        _builder.append(_parameterName);
        _builder.append(" the fixed value of pattern parameter ");
        String _name = p.getName();
        _builder.append(_name);
        _builder.append(", or null if not bound.");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("@return a stream of matches represented as a ");
    JvmType _findMatchClass = this._eMFPatternLanguageJvmModelInferrerUtil.findMatchClass(pattern);
    String _simpleName = null;
    if (_findMatchClass!=null) {
      _simpleName=_findMatchClass.getSimpleName();
    }
    _builder.append(_simpleName);
    _builder.append(" object.");
    _builder.newLineIfNotEmpty();
    return _builder;
  }
  
  public CharSequence javadocGetOneArbitraryMatchMethod(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Returns an arbitrarily chosen match of the pattern that conforms to the given fixed values of some parameters.");
    _builder.newLine();
    _builder.append("Neither determinism nor randomness of selection is guaranteed.");
    _builder.newLine();
    {
      EList<Variable> _parameters = pattern.getParameters();
      for(final Variable p : _parameters) {
        _builder.append("@param ");
        String _parameterName = this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
        _builder.append(_parameterName);
        _builder.append(" the fixed value of pattern parameter ");
        String _name = p.getName();
        _builder.append(_name);
        _builder.append(", or null if not bound.");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("@return a match represented as a ");
    JvmType _findMatchClass = this._eMFPatternLanguageJvmModelInferrerUtil.findMatchClass(pattern);
    String _simpleName = null;
    if (_findMatchClass!=null) {
      _simpleName=_findMatchClass.getSimpleName();
    }
    _builder.append(_simpleName);
    _builder.append(" object, or null if no match is found.");
    _builder.newLineIfNotEmpty();
    return _builder;
  }
  
  public CharSequence javadocHasMatchMethod(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Indicates whether the given combination of specified pattern parameters constitute a valid pattern match,");
    _builder.newLine();
    _builder.append("under any possible substitution of the unspecified parameters (if any).");
    _builder.newLine();
    {
      EList<Variable> _parameters = pattern.getParameters();
      for(final Variable p : _parameters) {
        _builder.append("@param ");
        String _parameterName = this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
        _builder.append(_parameterName);
        _builder.append(" the fixed value of pattern parameter ");
        String _name = p.getName();
        _builder.append(_name);
        _builder.append(", or null if not bound.");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("@return true if the input is a valid (partial) match of the pattern.");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocHasMatchMethodNoParameter(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Indicates whether the (parameterless) pattern matches or not.");
    _builder.newLine();
    _builder.append("@return true if the pattern has a valid match.");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocCountMatchesMethod(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Returns the number of all matches of the pattern that conform to the given fixed values of some parameters.");
    _builder.newLine();
    {
      EList<Variable> _parameters = pattern.getParameters();
      for(final Variable p : _parameters) {
        _builder.append("@param ");
        String _parameterName = this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
        _builder.append(_parameterName);
        _builder.append(" the fixed value of pattern parameter ");
        String _name = p.getName();
        _builder.append(_name);
        _builder.append(", or null if not bound.");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("@return the number of pattern matches found.");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocForEachMatchMethod(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Executes the given processor on each match of the pattern that conforms to the given fixed values of some parameters.");
    _builder.newLine();
    {
      EList<Variable> _parameters = pattern.getParameters();
      for(final Variable p : _parameters) {
        _builder.append("@param ");
        String _parameterName = this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
        _builder.append(_parameterName);
        _builder.append(" the fixed value of pattern parameter ");
        String _name = p.getName();
        _builder.append(_name);
        _builder.append(", or null if not bound.");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("@param processor the action that will process each pattern match.");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocForOneArbitraryMatchMethod(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Executes the given processor on an arbitrarily chosen match of the pattern that conforms to the given fixed values of some parameters.");
    _builder.newLine();
    _builder.append("Neither determinism nor randomness of selection is guaranteed.");
    _builder.newLine();
    {
      EList<Variable> _parameters = pattern.getParameters();
      for(final Variable p : _parameters) {
        _builder.append("@param ");
        String _parameterName = this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
        _builder.append(_parameterName);
        _builder.append(" the fixed value of pattern parameter ");
        String _name = p.getName();
        _builder.append(_name);
        _builder.append(", or null if not bound.");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("@param processor the action that will process the selected match.");
    _builder.newLine();
    _builder.append("@return true if the pattern has at least one match with the given parameter values, false if the processor was not invoked");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocProcessMethod(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Defines the action that is to be executed on each match.");
    _builder.newLine();
    {
      EList<Variable> _parameters = pattern.getParameters();
      for(final Variable p : _parameters) {
        _builder.append("@param ");
        String _parameterName = this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
        _builder.append(_parameterName);
        _builder.append(" the value of pattern parameter ");
        String _name = p.getName();
        _builder.append(_name);
        _builder.append(" in the currently processed match");
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder;
  }
  
  public CharSequence javadocNewMatchMethod(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Returns a new (partial) match.");
    _builder.newLine();
    _builder.append("This can be used e.g. to call the matcher with a partial match.");
    _builder.newLine();
    _builder.append("<p>The returned match will be immutable. Use {@link #newEmptyMatch()} to obtain a mutable match object.");
    _builder.newLine();
    {
      EList<Variable> _parameters = pattern.getParameters();
      for(final Variable p : _parameters) {
        _builder.append("@param ");
        String _parameterName = this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
        _builder.append(_parameterName);
        _builder.append(" the fixed value of pattern parameter ");
        String _name = p.getName();
        _builder.append(_name);
        _builder.append(", or null if not bound.");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("@return the (partial) match object.");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocNewMutableMatchMethod(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Returns a mutable (partial) match.");
    _builder.newLine();
    _builder.append("Fields of the mutable match can be filled to create a partial match, usable as matcher input.");
    _builder.newLine();
    _builder.newLine();
    {
      EList<Variable> _parameters = pattern.getParameters();
      for(final Variable p : _parameters) {
        _builder.append("@param ");
        String _parameterName = this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
        _builder.append(_parameterName);
        _builder.append(" the fixed value of pattern parameter ");
        String _name = p.getName();
        _builder.append(_name);
        _builder.append(", or null if not bound.");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("@return the new, mutable (partial) match object.");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocNewEmptyMatchMethod(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Returns an empty, mutable match.");
    _builder.newLine();
    _builder.append("Fields of the mutable match can be filled to create a partial match, usable as matcher input.");
    _builder.newLine();
    _builder.newLine();
    _builder.append("@return the empty match.");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocGetAllValuesOfMethod(final Variable parameter) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Retrieve the set of values that occur in matches for ");
    String _name = parameter.getName();
    _builder.append(_name);
    _builder.append(".");
    _builder.newLineIfNotEmpty();
    _builder.append("@return the Set of all values or empty set if there are no matches");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocStreamAllValuesOfMethod(final Variable parameter) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Retrieve the set of values that occur in matches for ");
    String _name = parameter.getName();
    _builder.append(_name);
    _builder.append(".");
    _builder.newLineIfNotEmpty();
    _builder.append("</p>");
    _builder.newLine();
    _builder.append("<strong>NOTE</strong>: It is important not to modify the source model while the stream is being processed.");
    _builder.newLine();
    _builder.append("If the match set of the pattern changes during processing, the contents of the stream is <strong>undefined</strong>.");
    _builder.newLine();
    _builder.append("In such cases, either rely on {@link #getAllMatches()} or collect the results of the stream in end-user code.");
    _builder.newLine();
    _builder.append("     ");
    _builder.newLine();
    _builder.append("@return the Stream of all values or empty set if there are no matches");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocQuerySpecificationMethod(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("@return the singleton instance of the query specification of this pattern");
    _builder.newLine();
    _builder.append("@throws ViatraQueryRuntimeException if the pattern definition could not be loaded");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocQuerySpecificationInstanceMethod(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("@return the singleton instance of the query specification");
    _builder.newLine();
    _builder.append("@throws ViatraQueryRuntimeException if the pattern definition could not be loaded");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocGroupClass(final PatternModel model, final boolean includePrivate) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("A pattern group formed of all");
    {
      if ((!includePrivate)) {
        _builder.append(" public");
      }
    }
    _builder.append(" patterns defined in ");
    String _modelFileName = this._eMFPatternLanguageJvmModelInferrerUtil.modelFileName(model);
    _builder.append(_modelFileName);
    _builder.append(".vql.");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    {
      if (includePrivate) {
        _builder.append("<p>A private group that includes private patterns as well. Only intended use case is for pattern testing.");
        _builder.newLine();
      } else {
        _builder.append("<p>Use the static instance as any {@link ");
        _builder.append(IQueryGroup.class);
        _builder.append("}, to conveniently prepare");
        _builder.newLineIfNotEmpty();
        _builder.append("a VIATRA Query engine for matching all patterns originally defined in file ");
        String _modelFileName_1 = this._eMFPatternLanguageJvmModelInferrerUtil.modelFileName(model);
        _builder.append(_modelFileName_1);
        _builder.append(".vql,");
        _builder.newLineIfNotEmpty();
        _builder.append("in order to achieve better performance than one-by-one on-demand matcher initialization.");
        _builder.newLine();
      }
    }
    _builder.newLine();
    _builder.append("<p> From package ");
    String _packageName = model.getPackageName();
    _builder.append(_packageName);
    _builder.append(", the group contains the definition of the following patterns: <ul>");
    _builder.newLineIfNotEmpty();
    {
      final Function1<Pattern, Boolean> _function = (Pattern it) -> {
        return Boolean.valueOf((includePrivate || (!it.getModifiers().isPrivate())));
      };
      Iterable<Pattern> _filter = IterableExtensions.<Pattern>filter(model.getPatterns(), _function);
      for(final Pattern p : _filter) {
        _builder.append("<li>");
        String _name = p.getName();
        _builder.append(_name);
        _builder.append("</li>");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("</ul>");
    _builder.newLine();
    _builder.newLine();
    _builder.append("@see IQueryGroup");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocGroupClassInstanceMethod(final PatternModel model) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Access the pattern group.");
    _builder.newLine();
    _builder.newLine();
    _builder.append("@return the singleton instance of the group");
    _builder.newLine();
    _builder.append("@throws ViatraQueryRuntimeException if there was an error loading the generated code of pattern specifications");
    _builder.newLine();
    return _builder;
  }
}
