"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.create = void 0;
const language_core_1 = require("@vue/language-core");
const shared_1 = require("@vue/shared");
const path = require("path-browserify");
const vue_extract_file_1 = require("../plugins/vue-extract-file");
const types_1 = require("../types");
const getUserPreferences_1 = require("volar-service-typescript/lib/configs/getUserPreferences");
function create(ts, getVueOptions, getTsPluginClient) {
    return {
        name: 'vue-document-drop',
        create(context) {
            let casing = types_1.TagNameCasing.Pascal; // TODO
            const tsPluginClient = getTsPluginClient?.(context);
            const vueCompilerOptions = getVueOptions(context.env);
            return {
                async provideDocumentDropEdits(document, _position, dataTransfer) {
                    if (document.languageId !== 'html') {
                        return;
                    }
                    const decoded = context.decodeEmbeddedDocumentUri(document.uri);
                    const sourceScript = decoded && context.language.scripts.get(decoded[0]);
                    const virtualCode = decoded && sourceScript?.generated?.embeddedCodes.get(decoded[1]);
                    const vueVirtualCode = sourceScript?.generated?.root;
                    if (!sourceScript || !virtualCode || !(vueVirtualCode instanceof language_core_1.VueVirtualCode)) {
                        return;
                    }
                    let importUri;
                    for (const [mimeType, item] of dataTransfer) {
                        if (mimeType === 'text/uri-list') {
                            importUri = item.value;
                        }
                    }
                    if (!importUri || !vueCompilerOptions.extensions.some(ext => importUri.endsWith(ext))) {
                        return;
                    }
                    let baseName = importUri.substring(importUri.lastIndexOf('/') + 1);
                    baseName = baseName.substring(0, baseName.lastIndexOf('.'));
                    const newName = (0, shared_1.capitalize)((0, shared_1.camelize)(baseName));
                    const { sfc } = vueVirtualCode;
                    const script = sfc.scriptSetup ?? sfc.script;
                    if (!script) {
                        return;
                    }
                    const additionalEdit = {};
                    const code = [...(0, language_core_1.forEachEmbeddedCode)(vueVirtualCode)].find(code => code.id === (sfc.scriptSetup ? 'scriptSetupFormat' : 'scriptFormat'));
                    const lastImportNode = (0, vue_extract_file_1.getLastImportNode)(ts, script.ast);
                    const incomingFileName = context.env.typescript.uriToFileName(importUri);
                    let importPath;
                    const serviceScript = sourceScript.generated?.languagePlugin.typescript?.getServiceScript(vueVirtualCode);
                    if (tsPluginClient && serviceScript) {
                        const tsDocumentUri = context.encodeEmbeddedDocumentUri(sourceScript.id, serviceScript.code.id);
                        const tsDocument = context.documents.get(tsDocumentUri, serviceScript.code.languageId, serviceScript.code.snapshot);
                        const preferences = await (0, getUserPreferences_1.getUserPreferences)(context, tsDocument);
                        const importPathRequest = await tsPluginClient.getImportPathForFile(vueVirtualCode.fileName, incomingFileName, preferences);
                        if (importPathRequest) {
                            importPath = importPathRequest;
                        }
                    }
                    if (!importPath) {
                        importPath = path.relative(path.dirname(vueVirtualCode.fileName), incomingFileName)
                            || importUri.substring(importUri.lastIndexOf('/') + 1);
                        if (!importPath.startsWith('./') && !importPath.startsWith('../')) {
                            importPath = './' + importPath;
                        }
                    }
                    additionalEdit.changes ??= {};
                    additionalEdit.changes[context.encodeEmbeddedDocumentUri(sourceScript.id, code.id)] = [];
                    additionalEdit.changes[context.encodeEmbeddedDocumentUri(sourceScript.id, code.id)].push({
                        range: lastImportNode ? {
                            start: script.ast.getLineAndCharacterOfPosition(lastImportNode.end),
                            end: script.ast.getLineAndCharacterOfPosition(lastImportNode.end),
                        } : {
                            start: script.ast.getLineAndCharacterOfPosition(0),
                            end: script.ast.getLineAndCharacterOfPosition(0),
                        },
                        newText: `\nimport ${newName} from '${importPath}'`
                            + (lastImportNode ? '' : '\n'),
                    });
                    if (sfc.script) {
                        const edit = (0, vue_extract_file_1.createAddComponentToOptionEdit)(ts, sfc.script.ast, newName);
                        if (edit) {
                            additionalEdit.changes[context.encodeEmbeddedDocumentUri(sourceScript.id, code.id)].push({
                                range: {
                                    start: document.positionAt(edit.range.start),
                                    end: document.positionAt(edit.range.end),
                                },
                                newText: edit.newText,
                            });
                        }
                    }
                    return {
                        insertText: `<${casing === types_1.TagNameCasing.Kebab ? (0, shared_1.hyphenate)(newName) : newName}$0 />`,
                        insertTextFormat: 2,
                        additionalEdit,
                    };
                },
            };
        },
    };
}
exports.create = create;
//# sourceMappingURL=vue-document-drop.js.map