/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import javax.persistence.FetchType;
import org.eclipse.persistence.annotations.JoinFetchType;
import org.eclipse.persistence.tools.mapping.orm.ExternalBasicMapMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalBatchFetch;
import org.eclipse.persistence.tools.mapping.orm.ExternalCollectionTable;
import org.eclipse.persistence.tools.mapping.orm.ExternalConverter;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalObjectTypeConverter;
import org.eclipse.persistence.tools.mapping.orm.ExternalStructConverter;
import org.eclipse.persistence.tools.mapping.orm.ExternalTypeConverter;

/**
 * The external form for a basic map mapping, which is a child of an entity.
 *
 * @see EmbeddableEntity
 *
 * @version 2.5
 * @author Les Davis
 */
@SuppressWarnings("nls")
final class BasicMapMapping extends ConvertibleMapping
                            implements ExternalBasicMapMapping {

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String BASIC_MAP = "basic-map";

	/**
	 * The attribute name used to store and retrieve the fetch property.
	 */
	static final String FETCH = "fetch";

	/**
	 * The element name used to store and retrieve the join-fetch child text node.
	 */
	static final String JOIN_FETCH = "join-fetch";

	/**
	 * The element name used to store and retrieve the key-column child node.
	 */
	static final String KEY_COLUMN = "key-column";

	/**
	 * The element name used to store and retrieve the key-converter child text node.
	 */
	static final String KEY_CONVERTER = "key-converter";

	/**
	 * The element name used to store and retrieve the value-column child node.
	 */
	static final String VALUE_COLUMN = "value-column";

	/**
	 * The element name used to store and retrieve the value-converter child text node.
	 */
	static final String VALUE_CONVERTER = "value-converter";

	/**
	 * Creates a new <code>BasicMapMapping</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	BasicMapMapping(EmbeddableEntity parent, int index) {
		super(parent, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void addBatchFetch() {
		BatchFetch batchFetch = buildBatchFetch();
		batchFetch.addSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addCollectionTable(String name) {
		CollectionTable collectionTable = buildCollectionTable();
		collectionTable.addSelf();
		collectionTable.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addKeyColumn(String columnName) {
		EntityColumn column = buildColumn(KEY_COLUMN);
		column.addSelf();
		column.setName(columnName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addKeyConverterString(String converter) {
		updateChildTextNode(KEY_CONVERTER, converter);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addValueColumn(String columnName) {
		EntityColumn column = buildColumn(VALUE_COLUMN);
		column.addSelf();
		column.setName(columnName);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(FETCH);
		names.add(ACCESS);
		return names;
	}


	private BatchFetch buildBatchFetch() {
		return new BatchFetch(this);
	}

	private CollectionTable buildCollectionTable() {
		return new CollectionTable(this);
	}

	private EntityColumn buildColumn(String elementName) {
		return new EntityColumn(this, elementName);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(KEY_COLUMN);
		names.add(KEY_CONVERTER);
		names.add(VALUE_COLUMN);
		names.add(VALUE_CONVERTER);
		names.add(Converter.CONVERTER);
		names.add(TypeConverter.TYPE_CONVERTER);
		names.add(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		names.add(StructConverter.STRUCT_CONVERTER);
		names.add(CollectionTable.COLLECTION_TABLE);
		names.add(JOIN_FETCH);
		names.add(BatchFetch.BATCH_FETCH);
		names.add(Property.PROPERTY);
		names.add(AccessMethods.ACCESS_METHODS);
		return names;
	}

	private Converter buildKeyConverter() {
		return new Converter(this);
	}

	private ObjectTypeConverter buildKeyObjectTypeConverter() {
		return new ObjectTypeConverter(this);
	}

	private StructConverter buildKeyStructConverter() {
		return new StructConverter(this);
	}

	private TypeConverter buildKeyTypeConverter() {
		return new TypeConverter(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalBatchFetch getBatchFetch() {

		if (hasChild(BatchFetch.BATCH_FETCH)) {
			return buildBatchFetch();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalCollectionTable getCollectionTable() {

		if (hasChild(CollectionTable.COLLECTION_TABLE)) {
			return buildCollectionTable();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getConvert() {
		return getChildTextNode(VALUE_CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return BASIC_MAP;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FetchType getFetchType() {
		return getEnumAttribute(FETCH, FetchType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public JoinFetchType getJoinFetchType() {
		return getChildEnumNode(JOIN_FETCH, JoinFetchType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn getKeyColumn() {

		if (hasChild(KEY_COLUMN)) {
			return buildColumn(KEY_COLUMN);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalConverter getKeyConverter() {

		if (hasChild(Converter.CONVERTER)) {
			return buildKeyConverter();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getKeyConveter() {
		return getChildTextNode(KEY_CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalObjectTypeConverter getKeyObjectTypeConverter() {

		if (hasChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER)) {
			return buildKeyObjectTypeConverter();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalStructConverter getKeyStructConverter() {

		if (hasChild(KEY_CONVERTER)) {
			return buildKeyStructConverter();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalTypeConverter getKeyTypeConverter() {

		if (hasChild(TypeConverter.TYPE_CONVERTER)) {
			return buildKeyTypeConverter();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn getValueColumn() {

		if (hasChild(VALUE_COLUMN)) {
			return buildColumn(VALUE_COLUMN);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeBatchFetch() {
		removeChild(BatchFetch.BATCH_FETCH);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeCollectionTable() {
		CollectionTable collectionTable = buildCollectionTable();
		collectionTable.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeKeyColumn() {
		EntityColumn column = buildColumn(KEY_COLUMN);
		column.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeKeyConverter() {
		removeChild(Converter.CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeKeyConverterString() {
		EntityColumn column = buildColumn(KEY_CONVERTER);
		column.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeKeyObjectTypeConverter() {
		removeChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeKeyStructConverter() {
		removeChild(StructConverter.STRUCT_CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeKeyTypeConverter() {
		removeChild(TypeConverter.TYPE_CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeValueColumn() {
		EntityColumn column = buildColumn(VALUE_COLUMN);
		column.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAsKeyConverter(String name, String className) {

		removeChild(TypeConverter.TYPE_CONVERTER);
		removeChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		removeChild(StructConverter.STRUCT_CONVERTER);

		Converter valueConverter = buildKeyConverter();
		valueConverter.addSelf();
		valueConverter.setName(name);
		valueConverter.setConverterClassName(className);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAsKeyObjectTypeConverter(String name) {

		removeChild(TypeConverter.TYPE_CONVERTER);
		removeChild(Converter.CONVERTER);
		removeChild(StructConverter.STRUCT_CONVERTER);

		ObjectTypeConverter valueConverter = buildKeyObjectTypeConverter();
		valueConverter.addSelf();
		valueConverter.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAsKeyStructConverter(String name, String converter) {

		removeChild(TypeConverter.TYPE_CONVERTER);
		removeChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		removeChild(Converter.CONVERTER);

		StructConverter valueConverter = buildKeyStructConverter();
		valueConverter.addSelf();
		valueConverter.setName(name);
		valueConverter.setConverterClassName(converter);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAsKeyTypeConverter(String name) {

		removeChild(Converter.CONVERTER);
		removeChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		removeChild(StructConverter.STRUCT_CONVERTER);

		TypeConverter valueConverter = buildKeyTypeConverter();
		valueConverter.addSelf();
		valueConverter.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setConvert(String value) {
		updateChildTextNode(VALUE_CONVERTER, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setFetchType(FetchType type) {
		setAttribute(FETCH, type);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setJoinFetchType(JoinFetchType type) {
		updateChildTextNode(JOIN_FETCH, type);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setKeyConverter(String value) {
		updateChildTextNode(KEY_CONVERTER, value);
	}
}