/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm;

import org.eclipse.persistence.annotations.ExistenceType;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;

/**
 * This interface and inherited behavior describe the configurable properties defined for a
 * mapped superclass entity in the ORM xml. One or more mapped superclass entity may be declared
 * at the root of the ORM xml.
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.<p>
 *
 * @see javax.persistence.MappedSuperclass
 * @see http://jcp.org/aboutJava/communityprocess/final/jsr220/index.html, persistence specificaton
 *
 * @version 2.5
 * @author John Bracken
 */
public interface ExternalMappedSuperClassEntity extends ExternalEmbeddableEntity,
                                                        ExternalEntityListener,
                                                        ExternalEntityListenerHolder {

	/**
	 * Adds a cache to this entity.
	 */
	ExternalCache addCache();

	/**
	 * Adds an embedded id mapping with the given name.
	 */
	ExternalEmbeddedIDMapping addEmbeddedIdMapping(String name);

	/**
	 * Adds a fetch group with the given name.
	 */
	ExternalFetchGroup addFetchGroup(int index, String name);

	/**
	 * Adds multitenancy
	 */
	ExternalMultitenancyPolicy addMultitenancy();

	/**
	 * Create new Optimistic locking for this entity replacing the old one.
	 */
	ExternalOptimisticLocking addOptimisticLocking();

	/**
	 * Creates new primary key for this entity replacing the old one.
	 */
	ExternalPrimaryKey addPrimaryKey();

	/**
	 * Creates a new sequence generator for this entity.
	 */
	ExternalSequenceGenerator addSequenceGenerator();

	/**
	 * Creates a new table generator for this entity.
	 */
	ExternalTableGenerator addTableGenerator();

	/**
	 * Adds a transformation mapping with the given name.
	 */
	ExternalTransformationMapping addTransformationMapping(String name);

	/**
	 * Adds a variable one-to-one mapping with the given name.
	 */
	ExternalVariableOneToOneMapping addVariableOneToOneMapping(String name);

	/**
	 * Adds a version mapping with the given name.
	 */
	ExternalVersionMapping addVersionMapping(String name);

	/**
	 * Returns a list of the fetch groups defined for this entity.
	 */
	ListIterable<ExternalFetchGroup> fetchGroups();

	/**
	 * Returns the count of fetch groups defined for this entity.
	 */
	int fetchGroupsSize();

	/**
	 * Returns the cache for this entity.
	 */
	ExternalCache getCache();

	/**
	 * Returns the cache intercepter class name defined for this entity.
	 */
	String getCacheInterceptorClassName();

	/**
	 * Returns the existence type for this entity
	 */
	ExistenceType getExistenceCheckingType();

	/**
	 * Returns the fetch group at the given index.
	 */
	ExternalFetchGroup getFetchGroup(int index);

	/**
	 * Returns the ID class name defined for this entity.
	 */
	String getIdClassName();

	/**
	 * Returns the description of the mappings in this entity
	 */
	String getMappingsDescription();

	/**
	 * Returns the multitenancy policy for this entity
	 */
	ExternalMultitenancyPolicy getMultitenant();

	/**
	 * Returns the optimistic locking definition for this entity.
	 */
	ExternalOptimisticLocking getOptimisticLocking();

	/**
	 * Returns the name of the parent class of this entity.
	 */
	String getParentClassName();

	/**
	 * Returns the primary key definition for this entity.
	 */
	ExternalPrimaryKey getPrimaryKey();

	/**
	 * Returns the sequence generator defined for this entity if one exists.
	 */
	ExternalSequenceGenerator getSequenceGenerator();

	/**
	 * Returns the table generator for this entity if one exists.
	 */
	ExternalTableGenerator getTableGenerator();

	/**
	 * Returns whether this entity is cacheable.
	 */
	Boolean isCacheable();

	/**
	 * Returns whether this eneity has multitenant
	 */
	Boolean isMultitenant();

	/**
	 * Returns whether this entity is read only.
	 */
	Boolean isReadOnly();

	/**
	 * Removes the cache from this entity if it exists.
	 */
	void removeCache();

	/**
	 * Removes the fetch group specified at the given position.
	 */
	void removeFetchGroup(int index);

	/**
	 * Removes multitenancy
	 */
	void removeMultiTenancy();

	/**
	 * Removes the optimistic locking from this entity if it exists.
	 */
	void removeOptimisticLocking();

	/**
	 * Removes the primary Key definition from this entity if it exists.
	 */
	void removePrimaryKey();

	/**
	 * Sets whether this entity is cacheable
	 */
	void setCacheable(Boolean cacheable);

	/**
	 * Sets the name of the cache intercepter class declared for this entity.
	 */
	void setCacheInterceptorClassName(String className);

	/**
	 * Sets if this entity should exclude default class listener methods.
	 */
	void setExcludesClassListeners(Boolean excludeClassListeners);

	/**
	 * Sets if this entity has declared to exclude listener methods defined on mapped superclasses.
	 */
	void setExcludesSuperClassListeners(Boolean excludesSuperClassListeners);

	/**
	 * Sets the existence checking type for this entity.
	 */
	void setExistenceCheckingType(ExistenceType type);

	/**
	 * Sets the name of the ID class declared for this entity.
	 */
	void setIdClassName(String className);

	/**
	 * Sets the description for the mappings on this entity
	 */
	void setMappingsDescription(String description);

	/**
	 * Sets the name of the parent class of this entity
	 */
	void setParentClassName(String className);

	/**
	 * Sets whether this entity is read only
	 */
	void setReadOnly(Boolean readOnly);

	/**
	 * Returns if this entity has declared to exclude default listener methods.
	 */
	Boolean shouldExcludesClassListeners();

	/**
	 * Returns if this entity has declared to exclude listener methods defined on mapped superclasses.
	 */
	Boolean shouldExcludesSuperClassListeners();
}