/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm;

import org.eclipse.persistence.annotations.ChangeTrackingType;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;

/**
 * This interface describes the configurable properties for an embeddable entity. One or more
 * embeddable entities can be declared as part of an ORM configuration.
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.<p>
 *
 * @see javax.persistence.Embeddable
 * @see http://jcp.org/aboutJava/communityprocess/final/jsr220/index.html, persistence specificaton
 *
 * @version 2.5
 * @author John Bracken
 */
public interface ExternalEmbeddableEntity extends ExternalForm {

	/**
	 * Adds the get and Sets method names for this mapping.
	 */
	void addAccessMethods(String getMethodName, String setMethodName);

	/**
	 * Adds a basic collection mapping with the given name.
	 */
	ExternalBasicCollectionMapping addBasicCollectionMapping(String name);

	/**
	 * Adds a basic map mapping with the given name.
	 */
	ExternalBasicMapMapping addBasicMapMapping(String name);

	/**
	 * Adds a basic mapping with the given name.
	 */
	ExternalBasicMapping addBasicMapping(String name);

	/**
	 * Adds clone copy policy to this entity
	 */
	ExternalCloneCopyPolicy addCloneCopyPolicy();

	/**
	 * Adds a converter to this entity
	 */
	ExternalConverter addConverter(int index, String name);

	/**
	 * Adds copy policy to this entity
	 */
	ExternalCopyPolicy addCopyPolicy();

	/**
	 * Adds a element collection mapping with the given name.
	 */
	ExternalElementCollectionMapping addElementCollectionMapping(String name);

	/**
	 * Adds an embeddable mapping with the given name.
	 */
	ExternalEmbeddedMapping addEmbeddedMapping(String name);

	/**
	 * Adds an id mapping with the given name.
	 */
	ExternalIDMapping addIdMapping(String name);

	
	/**
	 * Adds instantiation copy policy to this entity
	 */
	ExternalInstantiationCopyPolicy addInstantiationCopyPolicy();

	/**
	 * Adds a many-to-many mapping with the given name.
	 */
	ExternalManyToManyMapping addManyToManyMapping(String name);

	/**
	 * Adds a many-to-one mapping with the given name.
	 */
	ExternalManyToOneMapping addManyToOneMapping(String name);

	/**
	 * Adds the NoSQL annotation.
	 */
	ExternalNoSql addNoSql();

	/**
	 * Adds a object type converter to this entity
	 */
	ExternalObjectTypeConverter addObjectTypeConverter(int index, String name);

	/**
	 * Adds a one-to-many mapping with the given name.
	 */
	ExternalOneToManyMapping addOneToManyMapping(String name);

	/**
	 * Adds a one-to-one mapping with the given name.
	 */
	ExternalOneToOneMapping addOneToOneMapping(String name);

	/**
	 * Adds a property to this entity.
	 */
	void addProperty(int index, String name, String value);

	/**
	 * Adds a struct converter to this entity.
	 */
	ExternalStructConverter addStructConverter(int index, String name);

	/**
	 * Adds a transient mapping with the given name.
	 */
	ExternalTransientMapping addTransientMapping(String name);

	/**
	 * Adds a type converter to this entity.
	 */
	ExternalTypeConverter addTypeConverter(int index, String name);

	/**
	 * Returns a list of the converters defined for this entity.
	 */
	ListIterable<ExternalConverter> converters();

	/**
	 * Returns the count of converters defined for this entity.
	 */
	int convertersSize();

	/**
	 * Returns the access methods for this mapping.
	 */
	ExternalAccessMethods getAccessMethods();

	/**
	 * Returns the to be used in this entity.
	 */
	String getAccessType();

	/**
	 * Returns the change tracking type for this entity
	 */
	ChangeTrackingType getChangeTrackingType();

	/**
	 * Returns the class name of the entity.
	 */
	String getClassName();

	/**
	 * Returns the short class name of the entity.
	 */
	String getClassShortName();

	
	/**
	 * Returns the clone copy policy for this entity.
	 */
	ExternalCloneCopyPolicy getCloneCopyPolicy();

	/**
	 * Returns the converter at the given index
	 * @param index
	 */
	ExternalConverter getConverter(int index);

	/**
	 * Returns the copy policy for this entity.
	 */
	ExternalCopyPolicy getCopyPolicy();

	/**
	 * Returns the customizer class name.
	 */
	String getCustomizerClassName();

	/**
	 * Returns the description text for this entity.
	 */
	String getDescription();

	/**
	 * Returns the position of this entity within the list of entities owned by the parent.
	 */
	int getIndex();

	/**
	 * Returns the instantiation policy for this entity.
	 */
	ExternalInstantiationCopyPolicy getInstantiationCopyPolicy();

	/**
	 * Returns the mapping with the same name as the name parameter.
	 */
	ExternalMapping getMapping(String name);

	/**
	 * Returns the external form of NoSql.
	 */
	ExternalNoSql getNoSql();

	/**
	 * Returns the object type converter at the given index.
	 */
	ExternalObjectTypeConverter getObjectTypeConverter(int index);

	/**
	 * Returns the property at the given index.
	 */
	ExternalProperty getProperty(int index);

	/**
	 * Returns the struct type converter at the given index.
	 */
	ExternalStructConverter getStructConverter(int index);

	/**
	 * Returns the converter at the given index.
	 */
	ExternalTypeConverter getTypeConverter(int index);

	/**
	 * Returns a list of the id mappings defined for this entity.
	 */
	ListIterable<ExternalIDMapping> idMappings();

	/**
	 * Indicates whether default mappings should be excluded for this entity.
	 */
	Boolean isExcludeDefaultMappings();

	/**
	 * Indicates whether the meta data for this entity is completely described by the ORM definition.
	 */
	Boolean isMetadataComplete();

	/**
	 * Returns a list of the attribute mappings defined for this entity.
	 */
	ListIterable<ExternalMapping> mappings();

	/**
	 * Returns the count of attribute mappings defined for this entity.
	 */
	int mappingsSize();

	/**
	 * Returns a list of the object type converters defined for this entity.
	 */
	ListIterable<ExternalObjectTypeConverter> objectTypeConverters();

	/**
	 * Returns the count of the object type converters defined for this entity.
	 */
	int objectTypeConvertersSize();

	/**
	 * Returns a list of the properties defined for this entity.
	 */
	ListIterable<ExternalProperty> properties();

	/**
	 * Returns the count of properties defined for this entity.
	 */
	int propertiesSize();

	/**
	 * Removes the access methods element from the mapping.
	 */
	void removeAccessMethods();

	/**
	 * Removes the clone copy policy from this entity.
	 */
	void removeCloneCopyPolicy();

	/**
	 * Removes the converter specified at the given position.
	 */
	void removeConverter(int index);

	/**
	 * Removes the copy policy from this entity.
	 */
	void removeCopyPolicy();

	/**
	 * Removes the instantiation copy policy from this entity.
	 */
	void removeInstantiationCopyPolicy();

	/**
	 * Removes the mapping specified at the given position.
	 */
	void removeMapping(String name);

	/**
	 * Removes the NoSQL annotation.
	 */
	void removeNoSql();

	/**
	 * Removes the object type converter specified at the given position.
	 */
	void removeObjectTypeConverter(int index);

	/**
	 * Removes the property specified at the given position.
	 */
	void removeProperty(int index);

	/**
	 * Removes the struct converter specified at the given position.
	 */
	void removeStructConverter(int index);

	/**
	 * Removes the type converter specified at the given position.
	 */
	void removeTypeConverter(int index);

	/**
	 * Sets the {@link AccessType} to be used for this entity. String
	 * value used to support "VIRTUAL".
	 */
	void setAccessType(String type);

	/**
	 * Sets the change tracking type for this entity.
	 */
	void setChangeTrackingType(ChangeTrackingType type);

	/**
	 * Sets the class name.
	 */
	void setClassName(String className);

	/**
	 * Sets the customizer class name.
	 */
	void setCustomizerClassName(String name);

	/**
	 * Sets the description.
	 */
	void setDescription(String description);

	/**
	 * Sets whether default mappings should be excluded for this entity.
	 */
	void setExcludeDefaultMappings(Boolean excludeDefaultMappings);

	/**
	 * Sets whether the meta data for this entity is completely described by the ORM definition
	 */
	void setMetadataComplete(Boolean metaDataComplete);

	/**
	 * Returns a list of the struct converters defined for this entity.
	 */
	ListIterable<ExternalStructConverter> structConverters();

	/**
	 * Returns the count of converters defined for this entity.
	 */
	int structConvertersSize();

	/**
	 * Returns a list of the converters defined for this entity.
	 */
	ListIterable<ExternalTypeConverter> typeConverters();

	/**
	 * Returns the count of type converters defined for this entity.
	 */
	int typeConvertersSize();
}