/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.net.URL;
import java.util.List;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import org.eclipse.persistence.tools.mapping.orm.ORMDocumentType;
import org.eclipse.persistence.tools.mapping.orm.dom.ORMConfiguration;
import org.eclipse.persistence.tools.mapping.persistence.PersistenceDocumentType;
import org.eclipse.persistence.tools.mapping.persistence.dom.PersistenceConfiguration;
import org.eclipse.persistence.tools.utility.TextRange;
import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.Text;
import static org.eclipse.persistence.tools.mapping.AbstractExternalForm.*;
import static org.eclipse.persistence.tools.mapping.orm.ORMXmlConstants.*;
import static org.eclipse.persistence.tools.mapping.persistence.PersistenceXmlConstants.*;

/**
 * The default implementation of {@link ExternalFormHelper} that directly manipulate the document.
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.<p>
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
public class DefaultFormHelper implements ExternalFormHelper {

	/**
	 * The XML document to manipulate.
	 */
	private final Document document;

	/**
	 * The location of the XML document on the file system.
	 */
	private final URL location;

	/**
	 * Creates a new <code>FormHelper</code>.
	 *
	 * @param document The XML document to manipulate
	 * @param location The location of the XML document on the file system
	 */
	public DefaultFormHelper(Document document, URL location) {
		super();
		this.document = document;
		this.location = location;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void acquireReadLock() {
		// Nothing to do
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element addChild(AbstractExternalForm externalForm,
	                        Element element,
	                        String elementName,
	                        List<String> elementNamesOrder) {

		// Create the child node
		Element childElement = document.createElement(elementName);

		// Insert the new child node at the right location
		Element elementOfInsertion = externalForm.elementOfInsertion(
			element,
			elementName,
			elementNamesOrder
		);

		element.insertBefore(childElement, elementOfInsertion);

		return childElement;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element addChildTextNode(AbstractExternalForm externalForm,
	     	                          Element element,
	                                String elementName,
	                                String value,
	                                List<String> elementNamesOrder) {

		// Create the child text node
		Element childElement = getDocument().createElement(elementName);

		Text text = getDocument().createTextNode(value);
		text.setNodeValue(value);

		childElement.appendChild(text);

		// Insert the new child text node at the right location
		Element elementOfInsertion = externalForm.elementOfInsertion(element, elementName, elementNamesOrder);
		element.insertBefore(childElement, elementOfInsertion);

		return childElement;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element buildORMConfiguration(ORMConfiguration orm) {

		Element element = document.createElementNS(ECLIPSELINK_ORM_NAMESPACE_URI, orm.getElementName());
		element.setAttributeNS(XMLNS_URI, XMLNS_ATTRIBUTE, ECLIPSELINK_ORM_NAMESPACE_URI);
		orm.addXmlns(element, "xsi", XSI_URI);
		element.setAttributeNS(XSI_URI, XSI_SCHEMA_LOCATION, orm.buildSchemaLocation(ORMDocumentType.ECLIPELINK_2_6));
		element.setAttribute(XSD_VERSION, orm.getBuildVersion());

		document.appendChild(element);

		return element;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element buildPersistenceConfiguration(PersistenceConfiguration persistence) {

		Element element = document.createElementNS(PERSISTENCE_NAMESPACE_URI, persistence.getElementName());
		element.setAttributeNS(XMLNS_URI, XMLNS_ATTRIBUTE, PERSISTENCE_NAMESPACE_URI);
		persistence.addXmlns(element, "xsi", XSI_URI);
		element.setAttributeNS(XSI_URI, XSI_SCHEMA_LOCATION, persistence.buildSchemaLocation(PersistenceDocumentType.JPA_2_1));
		element.setAttribute(XSD_VERSION, persistence.getBuildVersion());

		document.appendChild(element);

		return element;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Document getDocument() {
		return document;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getElementNameTextRange(AbstractExternalForm externalForm, Element element) {
		throw new IllegalAccessError("Not supported");
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public URL getLocation() {
		return location;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public URL getSourceRoot() {
		throw new IllegalAccessError("Not supported");
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getTextNodeTextRange(AbstractExternalForm externalForm, Node node) {
		throw new IllegalAccessError("Not supported");
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getTextRange(AbstractExternalForm externalForm, Node node) {
		throw new IllegalAccessError("Not supported");
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void releaseReadLock() {
		// Nothing to do
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void remove(AbstractExternalForm externalForm, Element element, Element childElement) {
		element.removeChild(childElement);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeChildren(AbstractExternalForm externalForm, Node node, String elementName) {

		for (Element childElement : externalForm.getChildren(node, elementName)) {
			node.removeChild(childElement);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void save() throws Exception {

		TransformerFactory factory = TransformerFactory.newInstance();
		Transformer transformer = factory.newTransformer();

		FileOutputStream outputStream = new FileOutputStream(new File(location.toURI()));
		BufferedOutputStream output = new BufferedOutputStream(outputStream);

		try {
			DOMSource source = new DOMSource(document);
			StreamResult result = new StreamResult(outputStream);
			transformer.transform(source, result);
		}
		finally {
			output.close();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAttribute(AbstractExternalForm externalForm,
		                      Element element,
	                         String attributeName,
	                         String value,
	                         List<String> attributeNamesOrder) {

		// Remove the attribute
		if (value == null) {
			element.removeAttribute(attributeName);
		}
		// Update the attribute's value
		else if (externalForm.hasAttribute(attributeName)) {
			element.setAttribute(attributeName, value);
		}
		// Add a new attribute
		else {

			// Create the attribute node
			Attr newAttribute = getDocument().createAttribute(attributeName);
			newAttribute.setValue(value);

			// Insert the new attribute node at the right location
			List<Attr> attributes = externalForm.getAttributes(element);
			Attr elementOfInsertion = (Attr) externalForm.nodeOfInsertion(element, attributes, attributeName, attributeNamesOrder);

			// The attribute needs to be inserted before another attribute.
			// Remove the attributes so they can be re-added in proper order
			if (elementOfInsertion != null) {
				int indexOfInsertion = attributes.indexOf(elementOfInsertion);

				// Remove the attributes
				for (int index = attributes.size(); --index >= 0; ) {
					Node attributeNode = attributes.get(index);
					element.removeAttribute(attributeNode.getNodeName());
				}

				// Inserts the new attribute at the right location
				attributes.add(indexOfInsertion, newAttribute);

				// Re-add the attributes
				for (int index = 0; index < attributes.size(); index++) {
					Attr attribute = attributes.get(index);
					element.setAttributeNode(attribute);
				}
			}
				// This will insert the attribute at the end of the list of attributes
			else {
				element.setAttributeNode(newAttribute);
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAttributeNS(AbstractExternalForm externalForm,
		                        Element element,
	                           String attributeName,
	                           String value,
	                           List<String> attributeNamesOrder) {

		// Remove the attribute
		if (value == null) {
			element.removeAttributeNS(XSI_URI, attributeName);
		}
		// Update the attribute's value
		else if (externalForm.hasAttribute(attributeName)) {
			attributeName = externalForm.buildQualifiedName(attributeName);
			element.setAttributeNS(XSI_URI, attributeName, value);
		}
		// Add a new attribute
		else {

			// Create the attribute node
			attributeName = externalForm.buildQualifiedName(attributeName);
			Attr newAttribute = getDocument().createAttributeNS(XSI_URI, attributeName);
			newAttribute.setValue(value);

			// Insert the new attribute node at the right location
			List<Attr> attributes = externalForm.getAttributes(element);

			Attr elementOfInsertion = (Attr) externalForm.nodeOfInsertion(element, attributes, attributeName, attributeNamesOrder);

			// The attribute needs to be inserted before another attribute.
			// Remove the attributes so they can be re-added in proper order
			if (elementOfInsertion != null) {
				int indexOfInsertion = attributes.indexOf(elementOfInsertion);

				// Remove the attributes
				for (int index = attributes.size(); --index >= 0; ) {
					Node attributeNode = attributes.get(index);
					element.removeAttribute(attributeNode.getNodeName());
				}

				// Inserts the new attribute at the right location
				attributes.add(indexOfInsertion, newAttribute);

				// Re-add the attributes
				for (int index = 0, count = attributes.size(); index < count; index++) {
					Attr attribute = attributes.get(index);
					element.setAttributeNode(attribute);
				}
			}
			// This will insert the attribute at the end of the list of attributes
			else {
				element.setAttributeNS(XSI_URI, attributeName, value);
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setElementName(AbstractExternalForm externalForm,
		                        Element element,
		                        String elementName,
		                        List<String> elementNamesOrder) {

		// First remove the existing node and add the new node
		getDocument().removeChild(element);
		Element newElement = externalForm.addSelf(elementName, elementNamesOrder);

		// Now add the new node, which will have the new element name

		// Copy all the children to the new node
		for (Node childNode : externalForm.getAllChildren(element)) {
			newElement.appendChild(childNode);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setNamespace(AbstractExternalForm externalForm, String uri) {
		Element rootElement = externalForm.getRootElement();
		rootElement.setAttributeNS(XMLNS_URI, XMLNS_ATTRIBUTE, uri);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setSchemaLocation(AbstractExternalForm externalForm, String uri) {
		Element rootElement = externalForm.getRootElement();
		rootElement.setAttributeNS(XSI_URI, XSI_SCHEMA_LOCATION, uri);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void updateTextNode(AbstractExternalForm externalForm,
	                           String elementName,
	                           Element childElement,
	                           String value) {

		Text text = externalForm.findTextNode(childElement);

		if (text == null) {
			text = getDocument().createTextNode(value);
			childElement.appendChild(text);
		}
		else {
			text.setNodeValue(value);
		}
	}
}