/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping;

import java.net.URL;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.dom.ORMConfiguration;
import org.eclipse.persistence.tools.mapping.persistence.dom.PersistenceConfiguration;
import org.eclipse.persistence.tools.utility.TextRange;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

/**
 * This helper is used by {@link AbstractExternalForm} and completes its behavior when it needs to
 * modify the document.
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.<p>
 *
 * @version 2.6
 */
public interface ExternalFormHelper {

	/**
	 * Acquires a lock before reading the document.
	 */
	void acquireReadLock();

	/**
	 * Adds a new child element to the given element. The child element will be added at the end of
	 * the list of children.
	 *
	 * @param externalForm Reference to the {@link AbstractExternalForm} invoking this method
	 * @param element The element to which a new child element will be added
	 * @param elementName The name of the new child element
	 * @param elementNamesOrder The list of element names used to determine the insertion point
	 * @return The newly created child element
	 */
	Element addChild(AbstractExternalForm externalForm,
	                 Element element,
	                 String elementName,
	                 List<String> elementNamesOrder);

	/**
	 * Adds a child text node to the given parent element.
	 *
	 * @param externalForm Reference to the {@link AbstractExternalForm} invoking this method
	 * @param element The element to which a new child element will be added
	 * @param elementName The name of the new child element
	 * @param value The node's text
	 * @param index The index of insertion within the collection of children
	 * @return The newly created element
	 */
	Element addChildTextNode(AbstractExternalForm externalForm,
		                      Element element,
	                         String elementName,
	                         String value,
	                         List<String> elementNamesOrder);

	/**
	 * Creates the root {@link Element} defining the persistence object/relational mapping file.
	 *
	 * @param ormConfiguration The root of the external form
	 * @return The newly created {@link Element}
	 */
	Element buildORMConfiguration(ORMConfiguration ormConfiguration);

	/**
	 * Creates the root {@link Element} defining the persistence configuration file.
	 *
	 * @param persistence The root of the external form
	 * @return The newly created {@link Element}
	 */
	Element buildPersistenceConfiguration(PersistenceConfiguration persistence);

	/**
	 * Returns the document being modify by the external form.
	 *
	 * @return The document being modify by the external form
	 */
	Document getDocument();

	/**
	 * Retrieves the {@link TextRange} for the name of the given {@link Element} within the document.
	 *
	 * @param externalForm Reference to the {@link AbstractExternalForm} invoking this method
	 * @param attribute The {@link Element} to retrieve the {@link TextRange} for its name
	 * @return The {@link TextRange} of the given {@link Element}'s name
	 */
	TextRange getElementNameTextRange(AbstractExternalForm externalForm, Element element);

	/**
	 * Returns the location on the file system of the document.
	 *
	 * @return The location on the file system of the document
	 */
	URL getLocation();

	/**
	 * Returns the source root of the underlying XML document.
	 *
	 * @return The path of the source container
	 */
	URL getSourceRoot();

	/**
	 * Retrieves the {@link TextRange} of the given node's text value.
	 *
	 * @param externalForm Reference to the {@link AbstractExternalForm} invoking this method
	 * @param node The node from which the {@link TextRange} is retrieved
	 * @return The {@link TextRange} of the node's text
	 */
	TextRange getTextNodeTextRange(AbstractExternalForm externalForm, Node node);

	/**
	 * Retrieves the {@link TextRange} of the given {@link Node}.
	 *
	 * @param externalForm Reference to the {@link AbstractExternalForm} invoking this method
	 * @param node The node from which the {@link TextRange} is retrieved, which can be either an
	 * {@link Element} or an {@link Attr}
	 * @return The {@link TextRange} of the given {@link Node}
	 */
	TextRange getTextRange(AbstractExternalForm externalForm, Node node);

	/**
	 * Releases the previously acquired lock from the document.
	 */
	void releaseReadLock();

	/**
	 * Removes the given {@link Element} from its parent {@link Element}.
	 *
	 * @param externalForm Reference to the {@link AbstractExternalForm} invoking this method
	 * @param element The parent of the {@link Element} to remove
	 * @param childElement The child to remove from its parent
	 */
	void remove(AbstractExternalForm externalForm, Element element, Element childElement);

	/**
	 * Removes the child {@link Element} from the given {@link Element}.
	 *
	 * @param externalForm Reference to the {@link AbstractExternalForm} invoking this method
	 * @param node The node from which the children with the given name will be removed
	 * @param elementName The name of the child element to remove
	 */
	void removeChildren(AbstractExternalForm externalForm, Node node, String elementName);

	/**
	 * Saves the content of the document on disk.
	 *
	 * @throws Exception If an error occurred while persisting the document
	 */
	void save() throws Exception;

	/**
	 * Sets an attribute on the given element with the given value.
	 *
	 * @param externalForm Reference to the {@link AbstractExternalForm} invoking this method
	 * @param element The element to have one of its attributes updated
	 * @param attributeName The name of the attribute
	 * @param value The value of the attribute. If the value is <code>null</code>, then the attribute will be removed
	 * @param attributeNamesOrder The list of attribute names used to determine the insertion
	 */
	void setAttribute(AbstractExternalForm externalForm,
                     Element element,
                     String attributeName,
                     String value,
                     List<String> attributeNamesOrder);

	/**
	 * Sets an attribute on the given element with the given value. The name will be formatted with
	 * the namespace URI: "&lt;namespaceURI&gt;:attributeName".
	 *
	 * @param externalForm Reference to the {@link AbstractExternalForm} invoking this method
	 * @param element The element to have one of its attributes updated
	 * @param attributeName The name of the attribute
	 * @param value The value of the attribute. If the value is <code>null</code>, then the attribute will be removed
	 * @param attributeNamesOrder The list of attribute names used to determine the insertion
	 */
	void setAttributeNS(AbstractExternalForm externalForm,
                       Element element,
                       String attributeName,
                       String value,
                       List<String> attributeNamesOrder);

	/**
	 * Renames the given {@link Element}.
	 *
	 * @param externalForm Reference to the {@link AbstractExternalForm} invoking this method
	 * @param element The {@link Element} to rename
	 * @param elementName The new name of the given {@link Element}
	 * @param elementNamesOrder
	 */
	void setElementName(AbstractExternalForm externalForm,
                       Element element,
                       String elementName,
                       List<String> elementNamesOrder);

	/**
	 * Sets the namespace with the given one.
	 *
	 * @param externalForm Reference to the {@link AbstractExternalForm} invoking this method
	 * @param uri The URI of the namespace
	 */
	void setNamespace(AbstractExternalForm externalForm, String uri);

	/**
	 * Sets the schema location with the given one.
	 *
	 * @param externalForm Reference to the {@link AbstractExternalForm} invoking this method
	 * @param uri The URI of the XML schema
	 */
	void setSchemaLocation(AbstractExternalForm externalForm, String uri);

	/**
	 * Updates the child text node with the given value. If the value is <code>null</code>, then the
	 * child element will be removed. If the value is not <code>null</code> but the element is
	 * <code>null</code> then the child will be added.
	 *
	 * @param externalForm Reference to the {@link AbstractExternalForm} invoking this method
	 * @param elementName The child element's name
	 * @param childElement The text node to change its value
	 * @param value The new element's text
	 */
	void updateTextNode(AbstractExternalForm externalForm,
                       String elementName,
                       Element childElement,
                       String value);
}