/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping;

import java.util.List;
import org.eclipse.persistence.tools.utility.TextRange;

/**
 * Defines an external ORM object that holds on to {@link ExternalProperty}. Implementors of this
 * interface know how to retrieve, remove, and create a property.
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.<p>
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
public interface ExternalPropertyHolder extends ExternalForm {

	/**
	 * The node name used to store and retrieve the element encapsulated by this external form.
	 */
	String PROPERTIES = "properties";

	/**
	 * Adds a property to this form.
	 */
	ExternalProperty addProperty(String name, String value);

	/**
	 * Retrieves the external properties with the given name.
	 *
	 * @param name The name of the properties to retrieve
	 * @return The list of properties with the given name as its key. A list is returned since it's
	 * possible to have more than one properties with the same key
	 */
	List<ExternalProperty> getProperties(String name);

	/**
	 * Returns the a property with the given name and value. Callers of this method assume that only
	 * one property exists with the given name. If multiple properties exist, acquire the property by
	 * name and value.
	 *
	 * @param index The position of the property to retrieve within the list of properties
	 * @return The external form of the property
	 */
	ExternalProperty getProperty(int index);

	/**
	 * Retrieves the external property, if one was found, with the given name.
	 *
	 * @param name The name of the key used to retrieve the property
	 * @return The property with the given name or <code>null</code> if none could be found
	 */
	ExternalProperty getProperty(String name);

	/**
	 * Retrieves the property at the given index from the list of properties with the given name.
	 *
	 * @param name The name used to retrieve the property at the given index within the list of
	 * properties with that same name
	 * @param index The index of the property to retrieve
	 * @return The property at the given index with the given name
	 */
	ExternalProperty getProperty(String name, int index);

	/**
	 * Retrieves the external property, if one was found, with the given name and value.
	 *
	 * @param name The name used to retrieve the property
	 * @param value The value used to retrieve the property
	 * @return The external property matching the given name and value if one could be found
	 */
	ExternalProperty getProperty(String name, String value);

	/**
	 * Returns the {@link TextRange} for the value of the "name" attribute defined in a {@link
	 * Element} with the element name "property".
	 * <p>
	 * &lt;<b>property</b> name="name" value="value"/&gt;
	 *
	 * @param name The value of the name attribute
	 * @return The {@link TextRange} of the value of the "value" attribute
	 */
	TextRange getPropertyNameTextRange(String name);

	/**
	 * Returns the {@link TextRange} for the {@link Element} with the attribute with the given name.
	 * <p>
	 * &lt;<b>property</b> name="name" value="value"/&gt;
	 *
	 * @param name The value of the name attribute
	 * @return The {@link TextRange} of the {@link Element} itself
	 */
	TextRange getPropertyTextRange(String name);

	/**
	 * Returns the {@link TextRange} for the property with the given name and value.
	 */
	TextRange getPropertyTextRange(String name, String value);

	/**
	 * Returns the {@link TextRange} for the value of the "value" attribute associated with the
	 * attribute with the given name defined in a {@link Element} with the element name "property".
	 * <p>
	 * &lt;<b>property</b> name="name" value="value"/&gt;
	 *
	 * @param name The value of the name attribute
	 * @return The {@link TextRange} of the value of the "value" attribute
	 */
	TextRange getPropertyValueTextRange(String name);

	/**
	 * Returns a list of the properties defined for this form.
	 */
	List<ExternalProperty> properties();

	/**
	 * Returns the count of properties defined for this form.
	 */
	int propertiesSize();

	/**
	 * Returns the count of properties defined for this form that has the given name.
	 *
	 * @param name The name used to determine the count of properties where their key matches that name
	 * @return The count of properties defined with the given name
	 */
	int propertiesSize(String name);

	/**
	 * Removes the given property from this form.
	 *
	 * @param index The position of the external property to remove
	 */
	void removeProperty(int index);

	/**
	 * Removes the first occurrence of the given property from this form.
	 *
	 * @param name The name of the external property to remove
	 */
	void removeProperty(String name);

	/**
	 * Removes the first occurrence of the given property from this form.
	 *
	 * @param name The name of the external property to remove
	 * @param value The value of the external property to remove
	 */
	void removeProperty(String name, String value);
}