/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import javax.persistence.FetchType;
import org.eclipse.persistence.annotations.JoinFetchType;
import org.eclipse.persistence.tools.mapping.orm.ExternalBasicCollectionMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalBatchFetch;
import org.eclipse.persistence.tools.mapping.orm.ExternalCollectionTable;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalMappingVisitor;
import org.eclipse.persistence.tools.utility.TextRange;

/**
 * The external form for a basic collection mapping, which is a child of an entity.
 *
 * @see Entity
 *
 * @version 2.6
 */
final class BasicCollectionMapping extends ConvertibleMapping
                                   implements ExternalBasicCollectionMapping {

	/**
	 * Creates a new <code>BasicCollectionMapping</code>.
	 *
	 * @param parent The parent of this external form
	 */
	BasicCollectionMapping(Embeddable parent) {
		super(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void accept(ExternalMappingVisitor visitor) {
		visitor.visit(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalBatchFetch addBatchFetch() {
		BatchFetch batchFetch = buildBatchFetch();
		batchFetch.addSelf();
		return batchFetch;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalCollectionTable addCollectionTable(String name) {
		CollectionTable collectionTable = buildCollectionTable();
		collectionTable.addSelf();
		collectionTable.setName(name);
		return collectionTable;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn addValueColumn(String columnName) {
		EntityColumn valueColumn = buildValueColumn();
		valueColumn.addSelf();
		valueColumn.setName(columnName);
		return valueColumn;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(FETCH);
		names.add(ACCESS);
		return names;
	}

	private BatchFetch buildBatchFetch() {
		return new BatchFetch(this);
	}

	private CollectionTable buildCollectionTable() {
		return new CollectionTable(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(VALUE_COLUMN);
		names.add(CONVERT);
		names.add(ClassConverter.CONVERTER);
		names.add(TypeConverter.TYPE_CONVERTER);
		names.add(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		names.add(StructConverter.STRUCT_CONVERTER);
		names.add(CollectionTable.COLLECTION_TABLE);
		names.add(JOIN_FETCH);
		names.add(BatchFetch.BATCH_FETCH);
		names.add(Property.PROPERTY);
		names.add(AccessMethods.ACCESS_METHODS);
		return names;
	}

	private EntityColumn buildValueColumn() {
		return new EntityColumn(this, VALUE_COLUMN, -1);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalBatchFetch getBatchFetch() {

		if (hasChild(BatchFetch.BATCH_FETCH)) {
			return buildBatchFetch();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalCollectionTable getCollectionTable() {

		if (hasChild(CollectionTable.COLLECTION_TABLE)) {
			return buildCollectionTable();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return BASIC_COLLECTION;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FetchType getFetchType() {
		return getEnumAttribute(FETCH, FetchType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getFetchTypeTextRange() {
		return getAttributeTextRange(FETCH);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public JoinFetchType getJoinFetchType() {
		return getChildEnumNode(JOIN_FETCH, JoinFetchType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getJoinFetchTypeTextRange() {
		return getChildTextNodeTextRange(JOIN_FETCH);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn getValueColumn() {

		if (hasChild(VALUE_COLUMN)) {
			return buildValueColumn();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeBatchFetch() {
		removeChild(BatchFetch.BATCH_FETCH);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeCollectionTable() {
		CollectionTable collectionTable = buildCollectionTable();
		collectionTable.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeValueColumn() {
		EntityColumn valueColumn = buildValueColumn();
		valueColumn.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setFetchType(FetchType type) {
		setAttribute(FETCH, type);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setJoinFetchType(JoinFetchType type) {
		updateChildTextNode(JOIN_FETCH, type);
	}
}