/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import javax.persistence.EnumType;
import javax.persistence.FetchType;
import javax.persistence.TemporalType;
import org.eclipse.persistence.tools.mapping.orm.ExternalBasicMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityIDGeneratedValue;
import org.eclipse.persistence.tools.mapping.orm.ExternalMappingVisitor;
import org.eclipse.persistence.tools.mapping.orm.ExternalNoSqlField;
import org.eclipse.persistence.tools.mapping.orm.ExternalSequenceGenerator;
import org.eclipse.persistence.tools.mapping.orm.ExternalTableGenerator;
import org.eclipse.persistence.tools.utility.TextRange;
import org.w3c.dom.Element;

/**
 * The external form of a basic mapping, which is a child of an entity.
 *
 * @see EmbeddableEntity
 *
 * @version 2.6
 */
final class BasicMapping extends ConvertibleMapping
                         implements ExternalBasicMapping {

	/**
	 * Creates a new <code>BasicMapping</code>.
	 *
	 * @param parent The parent of this external form
	 */
	BasicMapping(Embeddable parent) {
		super(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void accept(ExternalMappingVisitor visitor) {
		visitor.visit(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn addColumn() {
		EntityColumn column = buildColumn();
		column.addSelf();
		return column;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityIDGeneratedValue addGeneratedValue() {
		EntityIDGeneratedValue generatedValue = buildGeneratedValue();
		generatedValue.addSelf();
		return generatedValue;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalSequenceGenerator addSequenceGenerator() {
		SequenceGenerator sequenceGenerator = buildSequenceGenerator();
		sequenceGenerator.addSelf();
		return sequenceGenerator;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalTableGenerator addTableGenerator() {
		TableGenerator tableGenerator = buildTableGenerator();
		tableGenerator.addSelf();
		return tableGenerator;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(FETCH);
		names.add(OPTIONAL);
		names.add(ACCESS);
		names.add(MUTABLE);
		names.add(ATTRIBUTE_TYPE);
		return names;
	}

	private EntityColumn buildColumn() {
		return new EntityColumn(this, EntityColumn.COLUMN, -1);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(EntityColumn.COLUMN);
		names.add(NoSqlField.FIELD);
		names.add(EntityIDGeneratedValue.GENERATED_VALUE);
		names.add(LOB);
		names.add(TEMPORAL);
		names.add(ENUMERATED);
		names.add(CONVERT);
		names.add(ClassConverter.CONVERTER);
		names.add(TypeConverter.TYPE_CONVERTER);
		names.add(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		names.add(StructConverter.STRUCT_CONVERTER);
		names.add(TableGenerator.TABLE_GENERATOR);
		names.add(SequenceGenerator.SEQUENCE_GENERATOR);
		names.add(Property.PROPERTY);
		names.add(AccessMethods.ACCESS_METHODS);
		names.add(RETURN_INSERT);
		names.add(RETURN_UPDATE);
		return names;
	}

	private EntityIDGeneratedValue buildGeneratedValue() {
		return new EntityIDGeneratedValue(this);
	}

	private ExternalNoSqlField buildNoSqlField() {
		return new NoSqlField(this);
	}

	private SequenceGenerator buildSequenceGenerator() {
		return new SequenceGenerator(this, -1);
	}

	private TableGenerator buildTableGenerator() {
		return new TableGenerator(this, -1);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn getColumn() {
		return buildColumn();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return BASIC;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public EnumType getEnumeratedType() {
		return getChildEnumNode(ENUMERATED, EnumType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getEnumeratedTypeTextRange() {
		return getChildTextNodeTextRange(ENUMERATED);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FetchType getFetchType() {
		return getEnumAttribute(FETCH, FetchType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getFetchTypeTextRange() {
		return getAttributeTextRange(FETCH);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityIDGeneratedValue getGeneratedValue() {

		if (hasChild(EntityIDGeneratedValue.GENERATED_VALUE)) {
			return buildGeneratedValue();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getLobTextRange() {
		return getChildTextNodeTextRange(LOB);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getMutableTextRange() {
		return getAttributeTextRange(MUTABLE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getNoSqlField() {
		ExternalNoSqlField field = buildNoSqlField();
		return field.getName();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getOptionalTextRange() {
		return getAttributeTextRange(OPTIONAL);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getReturnOnInsertTextRange() {
		return getChildTextRange(RETURN_INSERT);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getReturnOnlyOnInsertTextRange() {
		return getChildAttributeTextRange(RETURN_INSERT, RETURN_ONLY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getReturnOnUpdateTextRange() {
		return getChildTextRange(RETURN_UPDATE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalSequenceGenerator getSequenceGenerator() {

		if (hasChild(SequenceGenerator.SEQUENCE_GENERATOR)) {
			return buildSequenceGenerator();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalTableGenerator getTableGenerator() {

		if (hasChild(TableGenerator.TABLE_GENERATOR)) {
			return buildTableGenerator();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TemporalType getTemporalType() {
		return getChildEnumNode(TEMPORAL, TemporalType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getTemporalTypeTextRange() {
		return getChildTextNodeTextRange(TEMPORAL);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean hasColumn() {
		return hasChild(EntityColumn.COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isBasicMapping() {
		return true;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isLob() {
		return getChildBooleanNode(LOB);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isMutable() {
		return getBooleanAttribute(MUTABLE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isOptional() {
		return getBooleanAttribute(OPTIONAL);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeColumn() {
		EntityColumn column = buildColumn();
		column.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeGeneratedValue() {
		EntityIDGeneratedValue generatedValue = buildGeneratedValue();
		generatedValue.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSequenceGenerator() {
		SequenceGenerator generator = buildSequenceGenerator();
		generator.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeTableGenerator() {
		TableGenerator generator = buildTableGenerator();
		generator.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean returnOnInsert() {
		return hasChild(RETURN_INSERT);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean returnOnlyOnInsert() {
		return getChildBooleanAttribute(RETURN_INSERT, RETURN_ONLY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean returnOnUpdate() {
		return hasChild(RETURN_UPDATE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setColumn(String columnName) {

		removeChild(EntityColumn.COLUMN);

		if (columnName != null) {
			EntityColumn entityColumn = buildColumn();
			entityColumn.setName(columnName);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setEnumeratedType(EnumType type) {
		updateChildTextNode(ENUMERATED, type);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setFetchType(FetchType type) {
		setAttribute(FETCH, type);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setLob(Boolean lob) {
		updateChildTextNode(LOB, lob);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMutable(Boolean mutable) {
		setAttribute(MUTABLE, mutable);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setNoSqlField(String field) {
		ExternalNoSqlField noSqlField = buildNoSqlField();
		noSqlField.setName(field);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setOptional(Boolean optional) {
		setAttribute(OPTIONAL, optional);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setReturnOnInsert(Boolean returnOnInsert) {
		if (returnOnInsert == Boolean.TRUE) {
			addChild(RETURN_INSERT);
		}
		else {
			removeChild(RETURN_INSERT);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setReturnOnlyOnInsert(Boolean returnOnlyOnInsert) {

		Element element = getChild(RETURN_INSERT);

		if ((element == null) && (returnOnlyOnInsert == null)) {
			return;
		}

		if ((element != null) && (returnOnlyOnInsert == null)) {
			remove(element);
		}
		else {

			if (element == null) {
				element = addChild(RETURN_INSERT);
			}

			setAttribute(element, RETURN_ONLY, returnOnlyOnInsert);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setReturnOnUpdate(Boolean returnOnUpdate) {
		if (returnOnUpdate == Boolean.TRUE) {
			addChild(RETURN_UPDATE);
		}
		else {
			removeChild(RETURN_UPDATE);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setSequenceGenerator(String generatorName) {
		updateChildTextNode(SequenceGenerator.SEQUENCE_GENERATOR, generatorName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setTableGenerator(String generatorName) {
		updateChildTextNode(TableGenerator.TABLE_GENERATOR, generatorName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setTemporalType(TemporalType type) {
		updateChildTextNode(TEMPORAL, type);
	}
}