/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.AbstractExternalForm;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityResult;
import org.eclipse.persistence.tools.mapping.orm.ExternalFieldResult;
import org.eclipse.persistence.tools.utility.TextRange;
import org.w3c.dom.Element;

/**
 * The external form of an entity result.
 *
 * @see SQLResultSetMapping
 *
 * @version 2.6
 */
final class EntityResult extends AbstractExternalForm
                         implements ExternalEntityResult {

	/**
	 * The position of the element within the list of children with the same type owned by the parent.
	 */
	private int index;

	/**
	 * Creates a new <code>EntityResult</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned by the parent
	 */
	EntityResult(SQLResultSetMapping parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalFieldResult addFieldResult(String name) {
		FieldResult fieldResult = buildFieldResult(-1);
		fieldResult.addSelf();
		fieldResult.setName(name);
		return fieldResult;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(ENTITY_CLASS);
		names.add(DISCRIMINATOR_COLUMN);
		return names;
	}

	private FieldResult buildFieldResult(int index) {
		return new FieldResult(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void calculateInsertionIndex(Element parent, Element child, String elementName) {
		if (elementName == getElementName()) {
			index = index(parent, child, elementName);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalFieldResult> fieldResults() {

		int count = fieldResultsSize();
		List<ExternalFieldResult> fieldResults = new ArrayList<ExternalFieldResult>(count);

		for (int index = 0; index < count; index++) {
			fieldResults.add(buildFieldResult(index));
		}

		return fieldResults;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int fieldResultsSize() {
		return getChildrenSize(FieldResult.FIELD_RESULT);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDiscriminatorColumnName() {
		return getAttribute(DISCRIMINATOR_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getDiscriminatorColumnNameTextRange() {
		return getAttributeTextRange(DISCRIMINATOR_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element getElement() {
		return getChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return ENTITY_RESULT;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getEntityClassName() {
		return getAttribute(ENTITY_CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getEntityClassNameTextRange() {
		return getAttributeTextRange(ENTITY_CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalFieldResult getFieldResult(int index) {

		if (hasChild(FieldResult.FIELD_RESULT, index)) {
			return buildFieldResult(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeFieldResult(int index) {
		FieldResult fieldResult = buildFieldResult(index);
		fieldResult.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSelf() {
		removeChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDiscriminatorColumnName(String name) {
		setAttribute(DISCRIMINATOR_COLUMN, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setEntityClassName(String className) {
		setAttribute(ENTITY_CLASS, className);
	}
}