/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.AbstractExternalForm;
import org.eclipse.persistence.tools.mapping.orm.ExternalStoredProcedureParameter;
import org.eclipse.persistence.tools.mapping.orm.ExternalNamedStoredProcedureQuery;
import org.eclipse.persistence.tools.utility.TextRange;

/**
 * The external form of a stored procedure query, which is a child of an entity or an ORM configuration.
 *
 * @see Entity
 * @see ORMConfiguration
 *
 * @version 2.6
 */
final class NamedStoredProcedureQuery extends AbstractQuery
                                      implements ExternalNamedStoredProcedureQuery {

	/**
	 * Creates a new <code>NamedStoredProcedureQuery</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned by the parent
	 */
	NamedStoredProcedureQuery(AbstractExternalForm parent, int index) {
		super(parent, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalStoredProcedureParameter addParameter() {
		StoredProcedureParameter parameter = buildParameter(-1);
		parameter.addSelf();
		return parameter;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addResultClassName(String name) {
		addChildTextNode(RESULT_CLASS, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addResultSetMapping(String mapping) {
		addChildTextNode(RESULT_SET_MAPPING, mapping);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(RESULT_CLASS);
		names.add(RESULT_SET_MAPPING);
		names.add(PROCEDURE_NAME);
		names.add(RETURNS_RESULT_SET);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(DESCRIPTION);
		return names;
	}

	private StoredProcedureParameter buildParameter(int index) {
		return new StoredProcedureParameter(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean doesReturnResultSet() {
		return getBooleanAttribute(RETURNS_RESULT_SET);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return NAMED_STORED_PROCEDURE_QUERY;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalStoredProcedureParameter getParameter(int index) {

		if (hasChild(StoredProcedureParameter.STORED_PROCEDURE_PARAMETER, index)) {
			return buildParameter(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getProcedureName() {
		return getAttribute(PROCEDURE_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getProcedureNameTextRange() {
		return getAttributeTextRange(PROCEDURE_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getResultClassName(int index) {
		return getChildTextNode(RESULT_CLASS, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getResultSetMapping(int index) {
		return getChildTextNode(RESULT_SET_MAPPING, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getResultSetMappingTextRange(int index) {
		return getChildTextNodeTextRange(RESULT_SET_MAPPING, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getReturnResultSetTextRange() {
		return getAttributeTextRange(RETURNS_RESULT_SET);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalStoredProcedureParameter> parameters() {

		int count = parametersSize();
		List<ExternalStoredProcedureParameter> parameters = new ArrayList<ExternalStoredProcedureParameter>(count);

		for (int index = 0; index < count; index++) {
			parameters.add(buildParameter(index));
		}

		return parameters;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int parametersSize() {
		return getChildrenSize(StoredProcedureParameter.STORED_PROCEDURE_PARAMETER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeParameter(int index) {
		StoredProcedureParameter parameter = buildParameter(-1);
		parameter.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeResultClassName(int index) {
		removeChild(RESULT_CLASS, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeResultSetMapping(int index) {
		removeChild(RESULT_SET_MAPPING, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<String> resultClassNames() {

		int count = resultClassNameSize();
		List<String> resultClassNames = new ArrayList<String>(count);

		for (int index = 0; index < count; index++) {
			resultClassNames.add(getResultClassName(index));
		}

		return resultClassNames;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int resultClassNameSize() {
		return getChildrenSize(RESULT_CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<String> resultSetMappings() {

		int count = resultSetMappingsSize();
		List<String> resultSetMappings = new ArrayList<String>(count);

		for (int index = 0; index < count; index++) {
			resultSetMappings.add(getResultSetMapping(index));
		}

		return resultSetMappings;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int resultSetMappingsSize() {
		return getChildrenSize(RESULT_SET_MAPPING);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDoesReturnResultSet(Boolean returnResultSet) {
		setAttribute(RETURNS_RESULT_SET, returnResultSet);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setProcedureName(String name) {
		setAttribute(PROCEDURE_NAME, name);
	}
}