/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.ExternalProperty;
import org.eclipse.persistence.tools.mapping.orm.AccessType;
import org.eclipse.persistence.tools.mapping.orm.ExternalAccessMethods;
import org.eclipse.persistence.tools.mapping.orm.ExternalNonTransientMapping;
import org.eclipse.persistence.tools.utility.ObjectTools;
import org.eclipse.persistence.tools.utility.TextRange;

/**
 * The external form for a non-transient mapping, which is a child of an entity.
 *
 * @see Embeddable
 *
 * @version 2.6
 */
abstract class NonTransientMapping extends Mapping
                                   implements ExternalNonTransientMapping {

	/**
	 * Creates a new <code>NonTransientMapping</code>.
	 *
	 * @param parent The parent of this external form
	 */
	NonTransientMapping(Embeddable parent) {
		super(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalAccessMethods addAccessMethods(String getMethodName, String setMethodName) {
		AccessMethods accessMethods = buildAccessMethods();
		accessMethods.addSelf();
		accessMethods.setGetMethod(getMethodName);
		accessMethods.setSetMethod(setMethodName);
		return accessMethods;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalProperty addProperty(String name, String value) {
		Property property = buildProperty(-1);
		property.addSelf();
		property.setName(name);
		property.setValue(value);
		return property;
	}

	private AccessMethods buildAccessMethods() {
		return new AccessMethods(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(Property.PROPERTY);
		names.add(AccessMethods.ACCESS_METHODS);
		return names;
	}

	private Property buildProperty(int index) {
		return new Property(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalAccessMethods getAccessMethods() {

		if (hasChild(AccessMethods.ACCESS_METHODS)) {
			return buildAccessMethods();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final AccessType getAccessType() {
		return getEnumAttribute(ACCESS, AccessType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getAccessTypeTextRange() {
		return getAttributeTextRange(ACCESS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final List<ExternalProperty> getProperties(String name) {

		List<ExternalProperty> properties = new ArrayList<ExternalProperty>();

		for (ExternalProperty property : properties()) {
			if (ObjectTools.equals(property.getName(), name)) {
				properties.add(property);
			}
		}

		return properties;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalProperty getProperty(int index) {

		if (hasChild(Property.PROPERTY, index)) {
			return buildProperty(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalProperty getProperty(String name) {

		for (ExternalProperty property : properties()) {
			if (ObjectTools.equals(property.getName(), name)) {
				return property;
			}
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalProperty getProperty(String name, int index) {

		ExternalProperty property = getProperty(index);

		if ((property != null) && ObjectTools.equals(name, property.getName())) {
			return property;
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalProperty getProperty(String name, String value) {

		for (ExternalProperty property : properties()) {

			if (ObjectTools.equals(property.getName(),  name) &&
			    ObjectTools.equals(property.getValue(), value)) {

				return property;
			}
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getPropertyNameTextRange(String name) {

		for (ExternalProperty property : properties()) {
			if (ObjectTools.equals(property.getName(), name)) {
				return property.getNameTextRange();
			}
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getPropertyTextRange(String name) {

		for (ExternalProperty property : properties()) {
			if (ObjectTools.equals(property.getName(), name)) {
				return property.getTextRange();
			}
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getPropertyTextRange(String name, String value) {

		for (ExternalProperty property : properties()) {

			if (ObjectTools.equals(property.getName(),  name) &&
			    ObjectTools.equals(property.getValue(), value)) {

				return property.getTextRange();
			}
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getPropertyValueTextRange(String name) {

		for (ExternalProperty property : properties()) {
			if (ObjectTools.equals(property.getName(), name)) {
				return property.getValueTextRange();
			}
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final List<ExternalProperty> properties() {

		int count = propertiesSize();
		List<ExternalProperty> properties = new ArrayList<ExternalProperty>(count);

		for (int index = 0; index < count; index++) {
			properties.add(buildProperty(index));
		}

		return properties;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final int propertiesSize() {
		return getChildrenSize(Property.PROPERTY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final int propertiesSize(String name) {

		int count = 0;

		for (ExternalProperty property : properties()) {
			if (ObjectTools.equals(property.getName(), name)) {
				count++;
			}
		}

		return count;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeAccessMethods() {
		removeChild(AccessMethods.ACCESS_METHODS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeProperty(int index) {
		Property property = buildProperty(index);
		property.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeProperty(String name) {
		Property property = (Property) getProperty(name);
		property.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeProperty(String name, String value) {
		Property property = (Property) getProperty(name, value);
		property.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setAccessType(AccessType type) {
		setAttribute(ACCESS, type);
	}
}