/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalWriteTransformer;
import org.w3c.dom.Element;

/**
 * The external form of a write transformer.
 *
 * @see TransformationMapping
 *
 * @version 2.6
 */
final class WriteTransfomer extends Transformer
                            implements ExternalWriteTransformer {

	/**
	 * The position of the element within the list of children with the same type owned by the parent.
	 */
	private int index;

	/**
	 * Creates a new <code>WriteTransfomer</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned by the parent
	 */
	WriteTransfomer(TransformationMapping parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addColumn() {
		EntityColumn column = buildColumn();
		column.addSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(TRANSFORMER_CLASS);
		names.add(METHOD);
		return names;
	}

	private EntityColumn buildColumn() {
		return new EntityColumn(this, EntityColumn.COLUMN, -1);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(EntityColumn.COLUMN);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void calculateInsertionIndex(Element parent, Element child, String elementName) {
		if (elementName == getElementName()) {
			index = index(parent, child, elementName);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn getColumn() {

		if (hasChild(EntityColumn.COLUMN)) {
			return buildColumn();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element getElement() {

		if (index == -1) {
			return super.getElement();
		}

		return getChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return WRITE_TRANSFORMER;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeColumn() {
		EntityColumn column = buildColumn();
		column.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSelf() {
		removeChild(getParent(), getElementName(), index);
	}
}