/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.persistence.dom;

import java.util.Collections;
import java.util.List;
import org.eclipse.persistence.tools.mapping.AbstractExternalForm;
import org.eclipse.persistence.tools.mapping.ExternalProperty;
import org.eclipse.persistence.tools.mapping.ExternalPropertyHolder;
import org.eclipse.persistence.tools.utility.TextRange;
import org.w3c.dom.Element;

/**
 * The external form of a persistence unit property, which is a child of the properties element.
 *
 * @see PersistenceUnit
 *
 * @version 2.6
 */
final class Property extends AbstractExternalForm
                     implements ExternalProperty {

	/**
	 * The index of this external form within the list of properties.
	 */
	private int index;

	/**
	 * Creates a new <code>Property</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	Property(PersistenceUnit parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element addSelf(String elementName, List<String> elementNamesOrder) {

		Element element = getParent().getChild(ExternalPropertyHolder.PROPERTIES);

		if (element == null) {
			element = getParent().addChild(ExternalPropertyHolder.PROPERTIES);
		}

		return addChild(element, elementName, Collections.<String>emptyList());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void calculateInsertionIndex(Element parent, Element child, String elementName) {
		if (elementName == getElementName()) {
			index = index(parent, child, elementName);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element getElement() {

		Element element = getChild(getParent(), ExternalPropertyHolder.PROPERTIES);

		if (element != null) {
			return getChild(element, PROPERTY, index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return PROPERTY;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getName() {
		return getAttribute(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getNameTextRange() {
		return getAttributeTextRange(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getValue() {
		return getAttribute(VALUE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getValueTextRange() {
		return getAttributeTextRange(VALUE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getValueType() {
		return getAttribute(VALUE_TYPE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSelf() {

		Element element = getChild(getParent(), ExternalPropertyHolder.PROPERTIES);

		if (element != null) {
			removeChild(element, PROPERTY, index);

			if (!hasAnyChildren(element)) {
				remove(getParentElement(), element);
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setName(String name) {
		setAttribute(NAME, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setValue(String value) {
		setAttribute(VALUE, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setValueType(String valueType) {
		setAttribute(VALUE_TYPE, valueType);
	}
}