/*******************************************************************************
 * Copyright (c) 2009, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 ******************************************************************************/
package org.eclipse.persistence.tools.db.driver;

import org.eclipse.persistence.tools.gen.db.Catalog;
import org.eclipse.persistence.tools.gen.db.Column;
import org.eclipse.persistence.tools.gen.db.Schema;
import org.eclipse.persistence.tools.gen.db.Sequence;
import org.eclipse.persistence.tools.gen.db.Table;

/**
 * Delegate DTP driver-specific behavior to implementations of this
 * interface:<ul>
 *   <li>catalogs/schemas
 *   <li>names/identifiers
 *   <li>database object selection
 * </ul>
 * <strong>NB:</strong><br>
 * We use <em>name</em> when dealing with the unmodified name of a database
 * object as supplied by the database itself (i.e. it is not delimited and it is
 * always case-sensitive).
 * <br>
 * We use <em>identifier</em> when dealing with a string representation of a
 * database object name (i.e. it may be delimited and, depending on the database,
 * it may be case-insensitive).
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.<p>
 *
 * @version 2.6
 */
public interface WorkbenchDriverAdapter {

	// ********** catalogs **********

	/**
	 * Returns whether the DTP driver supports "real" catalogs (e.g. Sybase).
	 * If it does, use {@link #getDTPCatalogs()} to retrieve them;
	 * otherwise, use {@link #getDTPSchemas()} to retrieve the schemas directly.
	 */
	boolean supportsCatalogs();

	/**
	 * Returns the DTP database's default catalog names.
	 * The first name in the list that identifies an existing catalog
	 * is <em>the</em> default.
	 * This will be empty if the database does not support catalogs.
	 */
	Iterable<String> getDefaultCatalogNames();

	// ********** schemas **********

	/**
	 * Returns the DTP database's default schema names.
	 * The first name in the list that identifies an existing schema
	 * is <em>the</em> default.
	 */
	Iterable<String> getDefaultSchemaNames();

	// ********** names/identifiers **********

	/**
	 * Convert the specified database object name to a database-specific
	 * identifier (i.e. delimit the name as appropriate).
	 */
	String convertNameToIdentifier(String name);

	/**
	 * Convert the specified database object name to a database-specific
	 * identifier (i.e. delimit the name as appropriate).
	 * Returns <code>null</code> if the resulting identifier matches the
	 * specified default name.
	 * <p>
	 * This is used by entity generation code to determine whether
	 * a generated annotation must explicitly identify a database object
	 * (e.g. a table) or the specified default adequately identifies the
	 * specified database object (taking into consideration case-sensitivity,
	 * special characters, etc.).
	 */
	String convertNameToIdentifier(String name, String defaultName);

	/**
	 * Convert the specified database object identifier to a database-specific
	 * name (i.e. stripping delimiters from and folding the identifier as
	 * appropriate).
	 */
	String convertIdentifierToName(String identifier);

	// ********** selecting database objects **********

	/**
	 * Select from the specified collection the catalog for the specified identifier.
	 */
	Catalog selectCatalogForIdentifier(Iterable<Catalog> catalogs, String identifier);

	/**
	 * Select from the specified collection the schema for the specified identifier.
	 */
	Schema selectSchemaForIdentifier(Iterable<Schema> schemata, String identifier);

	/**
	 * Select from the specified collection the table for the specified identifier.
	 */
	Table selectTableForIdentifier(Iterable<Table> tables, String identifier);

	/**
	 * Select from the specified collection the sequence for the specified identifier.
	 */
	Sequence selectSequenceForIdentifier(Iterable<Sequence> sequences, String identifier);

	/**
	 * Select from the specified collection the column for the specified identifier.
	 */
	Column selectColumnForIdentifier(Iterable<Column> columns, String identifier);
}