/**
 * Copyright (c) 2016-2018 TypeFox and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */
package org.eclipse.lsp4j;

import java.util.List;
import org.eclipse.lsp4j.jsonrpc.ProtocolDraft;
import org.eclipse.lsp4j.jsonrpc.ProtocolSince;
import org.eclipse.lsp4j.jsonrpc.util.ToStringBuilder;

/**
 * Code Action options.
 */
@SuppressWarnings("all")
public class CodeActionOptions extends AbstractWorkDoneProgressOptions {
  /**
   * CodeActionKinds that this server may return.
   * <p>
   * The list of kinds may be generic, such as {@link CodeActionKind#Refactor}, or the server
   * may list out every specific kind they provide.
   */
  @ProtocolSince("3.11.0")
  private List<String> codeActionKinds;

  /**
   * Static documentation for a class of code actions.
   * <p>
   * Documentation from the provider should be shown in the code actions
   * menu if either:
   * <ul>
   * <li>Code actions of {@code kind} are requested by the editor. In this case,
   * the editor will show the documentation that most closely matches the
   * requested code action kind. For example, if a provider has
   * documentation for both {@link CodeActionKind#Refactor} and
   * {@link CodeActionKind#RefactorExtract}, when the user requests
   * code actions for {@code RefactorExtract}, the editor will use
   * the documentation for {@code RefactorExtract} instead of the documentation
   * for {@code Refactor}.
   * <li>Any code actions of {@code kind} are returned by the provider.
   * </ul>
   * At most one documentation entry should be shown per provider.
   */
  @ProtocolDraft
  @ProtocolSince("3.18.0")
  private List<CodeActionKindDocumentation> documentation;

  /**
   * The server provides support to resolve additional
   * information for a code action.
   */
  @ProtocolSince("3.16.0")
  private Boolean resolveProvider;

  public CodeActionOptions() {
  }

  public CodeActionOptions(final List<String> codeActionKinds) {
    this.codeActionKinds = codeActionKinds;
  }

  /**
   * CodeActionKinds that this server may return.
   * <p>
   * The list of kinds may be generic, such as {@link CodeActionKind#Refactor}, or the server
   * may list out every specific kind they provide.
   */
  @ProtocolSince("3.11.0")
  public List<String> getCodeActionKinds() {
    return this.codeActionKinds;
  }

  /**
   * CodeActionKinds that this server may return.
   * <p>
   * The list of kinds may be generic, such as {@link CodeActionKind#Refactor}, or the server
   * may list out every specific kind they provide.
   */
  @ProtocolSince("3.11.0")
  public void setCodeActionKinds(final List<String> codeActionKinds) {
    this.codeActionKinds = codeActionKinds;
  }

  /**
   * Static documentation for a class of code actions.
   * <p>
   * Documentation from the provider should be shown in the code actions
   * menu if either:
   * <ul>
   * <li>Code actions of {@code kind} are requested by the editor. In this case,
   * the editor will show the documentation that most closely matches the
   * requested code action kind. For example, if a provider has
   * documentation for both {@link CodeActionKind#Refactor} and
   * {@link CodeActionKind#RefactorExtract}, when the user requests
   * code actions for {@code RefactorExtract}, the editor will use
   * the documentation for {@code RefactorExtract} instead of the documentation
   * for {@code Refactor}.
   * <li>Any code actions of {@code kind} are returned by the provider.
   * </ul>
   * At most one documentation entry should be shown per provider.
   */
  @ProtocolDraft
  @ProtocolSince("3.18.0")
  public List<CodeActionKindDocumentation> getDocumentation() {
    return this.documentation;
  }

  /**
   * Static documentation for a class of code actions.
   * <p>
   * Documentation from the provider should be shown in the code actions
   * menu if either:
   * <ul>
   * <li>Code actions of {@code kind} are requested by the editor. In this case,
   * the editor will show the documentation that most closely matches the
   * requested code action kind. For example, if a provider has
   * documentation for both {@link CodeActionKind#Refactor} and
   * {@link CodeActionKind#RefactorExtract}, when the user requests
   * code actions for {@code RefactorExtract}, the editor will use
   * the documentation for {@code RefactorExtract} instead of the documentation
   * for {@code Refactor}.
   * <li>Any code actions of {@code kind} are returned by the provider.
   * </ul>
   * At most one documentation entry should be shown per provider.
   */
  @ProtocolDraft
  @ProtocolSince("3.18.0")
  public void setDocumentation(final List<CodeActionKindDocumentation> documentation) {
    this.documentation = documentation;
  }

  /**
   * The server provides support to resolve additional
   * information for a code action.
   */
  @ProtocolSince("3.16.0")
  public Boolean getResolveProvider() {
    return this.resolveProvider;
  }

  /**
   * The server provides support to resolve additional
   * information for a code action.
   */
  @ProtocolSince("3.16.0")
  public void setResolveProvider(final Boolean resolveProvider) {
    this.resolveProvider = resolveProvider;
  }

  @Override
  public String toString() {
    ToStringBuilder b = new ToStringBuilder(this);
    b.add("codeActionKinds", this.codeActionKinds);
    b.add("documentation", this.documentation);
    b.add("resolveProvider", this.resolveProvider);
    b.add("workDoneProgress", getWorkDoneProgress());
    return b.toString();
  }

  @Override
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    if (!super.equals(obj))
      return false;
    CodeActionOptions other = (CodeActionOptions) obj;
    if (this.codeActionKinds == null) {
      if (other.codeActionKinds != null)
        return false;
    } else if (!this.codeActionKinds.equals(other.codeActionKinds))
      return false;
    if (this.documentation == null) {
      if (other.documentation != null)
        return false;
    } else if (!this.documentation.equals(other.documentation))
      return false;
    if (this.resolveProvider == null) {
      if (other.resolveProvider != null)
        return false;
    } else if (!this.resolveProvider.equals(other.resolveProvider))
      return false;
    return true;
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = super.hashCode();
    result = prime * result + ((this.codeActionKinds== null) ? 0 : this.codeActionKinds.hashCode());
    result = prime * result + ((this.documentation== null) ? 0 : this.documentation.hashCode());
    return prime * result + ((this.resolveProvider== null) ? 0 : this.resolveProvider.hashCode());
  }
}
