/**
 * Copyright (c) 2016-2018 TypeFox and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */
package org.eclipse.lsp4j;

import org.eclipse.lsp4j.jsonrpc.ProtocolDraft;
import org.eclipse.lsp4j.jsonrpc.ProtocolSince;
import org.eclipse.lsp4j.jsonrpc.util.Preconditions;
import org.eclipse.lsp4j.jsonrpc.util.ToStringBuilder;
import org.eclipse.lsp4j.jsonrpc.validation.NonNull;

/**
 * Describes the currently selected completion item.
 */
@ProtocolDraft
@ProtocolSince("3.18.0")
@SuppressWarnings("all")
public class SelectedCompletionInfo {
  /**
   * The range that will be replaced if this completion item is accepted.
   */
  @NonNull
  private Range range;

  /**
   * The text the range will be replaced with if this completion is
   * accepted.
   */
  @NonNull
  private String text;

  public SelectedCompletionInfo() {
  }

  public SelectedCompletionInfo(@NonNull final Range range, @NonNull final String text) {
    this.range = Preconditions.<Range>checkNotNull(range, "range");
    this.text = Preconditions.<String>checkNotNull(text, "text");
  }

  /**
   * The range that will be replaced if this completion item is accepted.
   */
  @NonNull
  public Range getRange() {
    return this.range;
  }

  /**
   * The range that will be replaced if this completion item is accepted.
   */
  public void setRange(@NonNull final Range range) {
    this.range = Preconditions.checkNotNull(range, "range");
  }

  /**
   * The text the range will be replaced with if this completion is
   * accepted.
   */
  @NonNull
  public String getText() {
    return this.text;
  }

  /**
   * The text the range will be replaced with if this completion is
   * accepted.
   */
  public void setText(@NonNull final String text) {
    this.text = Preconditions.checkNotNull(text, "text");
  }

  @Override
  public String toString() {
    ToStringBuilder b = new ToStringBuilder(this);
    b.add("range", this.range);
    b.add("text", this.text);
    return b.toString();
  }

  @Override
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    SelectedCompletionInfo other = (SelectedCompletionInfo) obj;
    if (this.range == null) {
      if (other.range != null)
        return false;
    } else if (!this.range.equals(other.range))
      return false;
    if (this.text == null) {
      if (other.text != null)
        return false;
    } else if (!this.text.equals(other.text))
      return false;
    return true;
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((this.range== null) ? 0 : this.range.hashCode());
    return prime * result + ((this.text== null) ? 0 : this.text.hashCode());
  }
}
