/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.core.internal.options;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.jsr220orm.core.options.IIntOption;
import org.eclipse.jsr220orm.core.options.IOptionsDescriptor;
import org.eclipse.jsr220orm.metadata.AttributeMetaData;
import org.eclipse.jsr220orm.metadata.EntityMetaData;

public class OptionsDescriptor implements IOptionsDescriptor{
    /**
     * The application is responsible for providing the pk.
     */
    public static final String ID_GENERATOR_TYPE_NONE = "None";

    /**
     * Use a lookup table to generate the pk.
     */
    public static final String ID_GENERATOR_TYPE_TABLE = "Table";

    /**
     * Use database sequence to generate the pk.
     */
    public static final String ID_GENERATOR_TYPE_SEQUENCE = "Sequence";

    /**
     * Use a database identity or autoincrement column for the pk.
     */
    public static final String ID_GENERATOR_TYPE_IDENTITY = "Identity";

    /**
     * Use a persistence provider chosen stragegy suitable to the target
     * database to generate the pk.
     */
    public static final String ID_GENERATOR_TYPE_AUTO = "Auto";

    private static final String ACCESS_TYPE_FIELD = "Field";

    private static final String ACCESS_TYPE_PROPERTY = "Property";

    Map keyMaps = new HashMap();
    
    public OptionsDescriptor() {
        Map idGeneratorTypes = new HashMap();
        Map accessTypes = new HashMap();
        Map fetchType = new HashMap();
        Map cascadeType = new HashMap();
        Map inheritance = new HashMap();
        addIntOption(idGeneratorTypes,
                EntityMetaData.ID_GENERATOR_TYPE_NONE, ID_GENERATOR_TYPE_NONE);
        addIntOption(idGeneratorTypes,
                EntityMetaData.ID_GENERATOR_TYPE_TABLE, ID_GENERATOR_TYPE_TABLE);
        addIntOption(idGeneratorTypes,
                EntityMetaData.ID_GENERATOR_TYPE_SEQUENCE, ID_GENERATOR_TYPE_SEQUENCE);
        addIntOption(idGeneratorTypes,
                EntityMetaData.ID_GENERATOR_TYPE_IDENTITY, ID_GENERATOR_TYPE_IDENTITY);
        addIntOption(idGeneratorTypes,
                EntityMetaData.ID_GENERATOR_TYPE_AUTO, ID_GENERATOR_TYPE_AUTO);
        addIntOption(accessTypes,
                EntityMetaData.ACCESS_TYPE_FIELD, ACCESS_TYPE_FIELD);
        addIntOption(accessTypes,
                EntityMetaData.ACCESS_TYPE_PROPERTY, ACCESS_TYPE_PROPERTY);
        addIntOption(fetchType, AttributeMetaData.FETCH_TYPE_EAGER,"Eager");
        addIntOption(fetchType, AttributeMetaData.FETCH_TYPE_LAZY,"Lazy");
        addIntOption(cascadeType, AttributeMetaData.CASCADE_TYPE_ALL, "All");
        addIntOption(cascadeType, AttributeMetaData.CASCADE_TYPE_MERGE, "Merge");
        addIntOption(cascadeType, AttributeMetaData.CASCADE_TYPE_NONE, "None");
        addIntOption(cascadeType, AttributeMetaData.CASCADE_TYPE_PERSIST, "Persist");
        addIntOption(cascadeType, AttributeMetaData.CASCADE_TYPE_REFRESH, "Refresh");
        addIntOption(cascadeType, AttributeMetaData.CASCADE_TYPE_REMOVE, "Remove");
        addIntOption(inheritance, EntityMetaData.INHERITANCE_NONE, "None");
        addIntOption(inheritance, EntityMetaData.INHERITANCE_JOINED, "Joined");
        addIntOption(inheritance, EntityMetaData.INHERITANCE_SINGLE_TABLE, "Single table");
        addIntOption(inheritance, EntityMetaData.INHERITANCE_TABLE_PER_CLASS, "Table per class");
        keyMaps.put(PROPERTY_KEY_ID_GENERATOR_TYPE, idGeneratorTypes);
        keyMaps.put(PROPERTY_KEY_ACCESS_TYPE, accessTypes);
        keyMaps.put("fetchType", fetchType);
        keyMaps.put("cascadeType", cascadeType);
        keyMaps.put("inheritance", inheritance);
        registerJavaSqlTypes();
    }

    private void addIntOption(Map map, int id, String descr) {
        map.put(new Integer(id), new IntOption(id, descr));
        map.put(descr, new IntOption(id, descr));
    }

    public Map getOptions(String propertyKey) {
        return (Map) keyMaps.get(propertyKey);
    }

    public IIntOption getIntOption(String propertyKey, String text) {
        return (IIntOption) ((Map)keyMaps.get(propertyKey)).get(text);
    }

    public Map getOptions(Class containerClass, String name) {
        return getOptions(name);
    }
    
    /**
     * Find and register all of the constants from java.sql.Types. 
     */
    protected void registerJavaSqlTypes() {
        Map types = new HashMap();
        Field[] a = java.sql.Types.class.getFields();
        for (int i = a.length - 1; i >= 0; i--) {
            Field f = a[i];
            int m = f.getModifiers();
            if (Modifier.isFinal(m) && Modifier.isStatic(m)) {
                Integer value;
                try {
                    value = (Integer) f.get(null);
                } catch (Exception e) {
                    // ignore, not possible really
                    continue;
                }
                String name = f.getName();
                addIntOption(types,
                        value.intValue(), name);
                    types.put(name, value);
            }
        }
        keyMaps.put("jdbcType", types);
    }

}
