/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.core.nature;

import java.io.IOException;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectNature;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IAdapterFactory;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.PlatformObject;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.window.Window;
import org.eclipse.jsr220orm.core.IEntityModelManager;
import org.eclipse.jsr220orm.core.OrmPlugin;
import org.eclipse.jsr220orm.core.internal.adaptors.RdbConnectionInfoFactory;
import org.eclipse.jsr220orm.core.internal.adaptors.RdbSchemaFactory;
import org.eclipse.jsr220orm.core.internal.options.OptionsDescriptor;
import org.eclipse.jsr220orm.core.internal.product.OrmProduct;
import org.eclipse.jsr220orm.core.internal.product.OrmProductRegistry;
import org.eclipse.jsr220orm.core.internal.product.OrmProject;
import org.eclipse.jsr220orm.core.options.IOptionsDescriptor;
import org.eclipse.jsr220orm.core.util.RdbUtils;
import org.eclipse.jsr220orm.metadata.EntityModel;
import org.eclipse.wst.rdb.connection.internal.ui.wizards.shared.UserIdentification;
import org.eclipse.wst.rdb.internal.core.connection.ConnectionInfo;
import org.eclipse.wst.rdb.internal.core.definition.DatabaseDefinition;
import org.eclipse.wst.rdb.internal.core.definition.DatabaseDefinitionRegistry;
import org.eclipse.wst.rdb.internal.core.rte.ICatalogProvider;
import org.eclipse.wst.rdb.internal.core.util.DatabaseProviderHelper;
import org.eclipse.wst.rdb.internal.models.sql.schema.Database;
import org.eclipse.wst.rdb.internal.models.sql.schema.Schema;
import org.osgi.framework.Bundle;

/**
 * Associates the active OrmProject with an IProject.
 */
public class OrmNature extends PlatformObject implements IProjectNature, IAdaptable, IResourceChangeListener {

    public static final String ID = "org.eclipse.jsr220orm.core.nature";

    private static final QualifiedName PERSISTENCE_PRODUCT_ID = new QualifiedName("OrmPersistence", "product_id");
    
    private PersistenceProperties persistenceProperties;
    private OrmProject activeProject;
    private IProject project;
    private List natureListeners = new ArrayList(5);
    private boolean internalSave;
    private Job reLoadActiveProjectJob = new Job("reLoadActiveProject") {
        protected IStatus run(IProgressMonitor monitor) {
            try{
                if(project.isOpen() && activeProject != null){
                    persistenceProperties.reload(project);
                    loadActiveProject();
                }
            }catch(Exception x){
                OrmPlugin.log(x);
                return new Status(IStatus.ERROR, OrmPlugin.ID, IStatus.OK, "Problem refreshing persistence.", x);
            }
            return new Status(IStatus.OK, OrmPlugin.ID, IStatus.OK, "",null); 
        }
    };

    public OrmNature() {
        reLoadActiveProjectJob.setPriority(Job.LONG);
    }

    public void configure() throws CoreException {
    }

    public void deconfigure() throws CoreException {
        closeProject();
    }

    public IProject getProject() {
        return project;
    }

    public void setProject(IProject project) {
        this.project = project;
        try {
            this.persistenceProperties = new PersistenceProperties(project);
            this.persistenceProperties.reload(project);
        } catch (Exception e) {
            OrmPlugin.log("Error reading .persistence file.",
                    "Error reading .persistence file form project: "
                            + project.getName(), e, IStatus.ERROR);
        }
        try {
            loadActiveProject();
        } catch (CoreException e) {
            OrmPlugin.log("Error activating persistence nature",
                    "An unexpected error occured activeting the persistence "
                            + "nature form project: " + project.getName(), e
                            .getStatus());
        } catch (Throwable e) {
            OrmPlugin.log( "An unexpected error occured activeting the " +
                    "persistence nature form project: " + project.getName(), e);
        }
    }

    void loadActiveProject() throws CoreException {
        closeProject();
        activeProject = createOrmProject(persistenceProperties.getPersistenceProduct());
        if (activeProject != null) {
            ResourcesPlugin.getWorkspace().addResourceChangeListener(this, IResourceChangeEvent.POST_CHANGE | IResourceChangeEvent.PRE_CLOSE);
            project.setSessionProperty(PERSISTENCE_PRODUCT_ID, activeProject.getProduct().getId());
            activeProject.getModel().registerAdapterFactory(new RdbConnectionInfoFactory(this));
            activeProject.getModel().registerAdapterFactory(new RdbSchemaFactory(this));
        }else{
            project.setSessionProperty(PERSISTENCE_PRODUCT_ID, null);
        }
        fireProjectActivated(activeProject);
   }

    public OrmProject getActiveOrmProject() {
        return activeProject;
    }

    /**
     * Get the persistence properties for the project. These are saved in a
     * file called .persistence in the project. ORM plugins can use these 
     * properties to associated whatever information they like with the
     * project. Keys should be prefixed with a vendor specific string to
     * avoid conflicts.
     * 
     * @see #savePersistenceProperties()
     */
    public PersistenceProperties getPersistenceProperties() {
        return persistenceProperties;
    }

    /**
     * Save changes to the {@link #getPersistenceProperties()} to the
     * .persistence file for the project. ORM plugins must call this to save
     * changes to the properties.
     */
    public void savePersistenceProperties() throws IOException, CoreException {
        persistenceProperties.save();
    }

    /**
     * Creates a new OrmProject, IEntityModelFactory and EntityModel and link
     * them to the OrmProject.
     */
    public OrmProject createOrmProject(String productKey) throws CoreException {
        if(productKey == null){
            return null;
        }
        OrmProduct ormProduct = OrmProductRegistry.INSTANCE.getOrmProductById(productKey);
        if(ormProduct == null){
            return null;
        }
        persistenceProperties.getProperties().putAll(ormProduct.getProperties());
        OrmProject ormProject = new OrmProject();
        activeProject = ormProject;
        ormProject.setIProject(project);
        ormProject.setProduct(ormProduct);
        Class clazz;
        try {
            Bundle bundle = Platform.getBundle(ormProduct.getNamespace());
            clazz = bundle.loadClass(ormProduct.getEntityModelManager());
        } catch (ClassNotFoundException e) {
            String message = ormProduct
                    + " has it's 'entityModelManager' property set to '"
                    + ormProduct.getEntityModelManager()
                    + "' but this class could not be found.";
            throw new CoreException(new Status(IStatus.ERROR, OrmPlugin.ID, 0,
                    message, e));
        }
        if (!IEntityModelManager.class.isAssignableFrom(clazz)) {
            String message = ormProduct
                    + " has it's 'entityModelManager' property set to '"
                    + ormProduct.getEntityModelManager()
                    + "' but this class this class does "
                    + "not implement org.eclipse.jsr220orm.core.IEntityModelManager.";
            throw new CoreException(new Status(IStatus.ERROR, OrmPlugin.ID, 0,
                    message, new Exception("Invalid IEntityModelFactory.")));
        }
        final IEntityModelManager entityModelManager;
        try {
            entityModelManager = (IEntityModelManager) clazz.newInstance();
        } catch (Throwable e) {
            String message = ormProduct
                    + " has it's 'entityModelManager' property set to '"
                    + ormProduct.getEntityModelManager()
                    + "' but this class could not be instantiated.";
            throw new CoreException(new Status(IStatus.ERROR, OrmPlugin.ID, 0,
                    message, e));
        }
        DatabaseDefinition databaseDefinition = null;
        try {
            databaseDefinition = getDatabaseDefinition();
            entityModelManager.setDatabaseDefinition(databaseDefinition);
        } catch (Throwable e) {
            OrmPlugin.log("Could not load database definition for "+getDatabaseName(), e);
        }
        try {
            ormProject.setModelManager(entityModelManager);
            entityModelManager.init(project, this, databaseDefinition);
        } catch (Throwable e) {
            entityModelManager.dispose();
            String message = ormProduct.getEntityModelManager()+ "for product: "+
            ormProduct + " could not be initialized:";
            throw new CoreException(new Status(IStatus.ERROR, OrmPlugin.ID, 0,
                    message, e));
        }
        try {
            EntityModel entityModel = entityModelManager.getEntityModel();
            entityModel.registerAdapterFactory(new IAdapterFactory() {
                //TODO make this load extra data from vendor def or extention point
                OptionsDescriptor descriptor = new OptionsDescriptor();

                public Class[] getAdapterList() {
                    return new Class[] { IOptionsDescriptor.class,
                            IProject.class, IEntityModelManager.class};
                }

                public Object getAdapter(Object adaptableObject,
                        Class adapterType) {
                    if (IProject.class.equals(adapterType)) {
                        return project;
                    } else if (IOptionsDescriptor.class.equals(adapterType)) {
                        return descriptor;
                    } else if (IEntityModelManager.class.equals(adapterType)) {
                        return entityModelManager;
                    }
                    return null;
                }

            });
            ormProject.setModel(entityModel);
        } catch (Throwable e) {
            entityModelManager.dispose();
            String message = ormProduct
                    + " could not create a entity mapping for project '"
                    + project.getName() + "'.";
            throw new CoreException(new Status(IStatus.ERROR, OrmPlugin.ID, 0,
                    message, e));
        }
        return ormProject;
    }

    public void addOrmNatureListener(OrmNatureListener listener) {
        natureListeners.add(listener);
    }

    public void removeOrmNatureListener(OrmNatureListener listener) {
        natureListeners.remove(listener);
    }

    void closeProject() throws CoreException {
        OrmProject oldProject = activeProject;
        activeProject = null;
        ResourcesPlugin.getWorkspace().removeResourceChangeListener(this);
        if (project.isOpen() && !ResourcesPlugin.getWorkspace().isTreeLocked()) {
            project.setSessionProperty(PERSISTENCE_PRODUCT_ID, null);
        }
        if(oldProject == null){
            return;
        }
        oldProject.dispose();
        fireProjectDeactivated(oldProject);
    }

    public ConnectionInfo getActiveConnectionInfo() {
        String rdbConnName = persistenceProperties.getConnectionName();
        if (rdbConnName != null) {
            ConnectionInfo[] rdbConnections = RdbUtils.getRdbConnections();
            for (int x = rdbConnections.length - 1; x >= 0; x--) {
                if (rdbConnName.equals(rdbConnections[x].getName())) {
                    return rdbConnections[x];
                }
            }
        }
        return null;
    }
    
    /**
     * Returns the database name if selected, else the default. 
     */
    public String getDatabaseName() {
        String database = persistenceProperties.getDatabaseName();
        DatabaseDefinitionRegistry ddr = RdbUtils
        .getDatabaseDefinitionRegistry();
        if(database == null || !ddr.getVersions(database).hasNext()){
            Iterator products = ddr.getProducts();
            String product = null;
            String[] defaults = new String[] {"MySql", "Oracle", "Derby", "SQL Server", "Sybase"};
            int index = defaults.length;
            while(products.hasNext()){
                String prodName = (String) products.next();
                if(prodName == null || prodName.trim().length() == 0){
                    continue;
                }
                for(int x=0; x < index; x++){
                    if(prodName.equals(defaults[x])){
                        index = x;
                        product = prodName;
                    }
                }
            }
            if(product == null){
                product = (String) ddr.getProducts().next();
            }
            return product;
        }
        return database;
    }
    
    /**
     * Returns the database version if selected and valid, else the default. 
     */
    public String getDatabaseVersion() {
        String database = getDatabaseName();
        String version = persistenceProperties.getDatabaseVersion();
        DatabaseDefinitionRegistry ddr = RdbUtils
                .getDatabaseDefinitionRegistry();
        Iterator versions = ddr.getVersions(database);
        String newVersion = version;
        while(versions.hasNext()){
            String tempVersion = (String) versions.next();
            if(tempVersion == null || tempVersion.trim().length() == 0){
                continue;
            }
            if(tempVersion.equals(version)){
                return version;
            }
            newVersion = tempVersion;
        }
        return newVersion;
    }    

    /**
     * Returns the database defenitions for this project. 
     */
    public DatabaseDefinition getDatabaseDefinition() {
        String product = getDatabaseName();
        String version = getDatabaseVersion();
        DatabaseDefinitionRegistry ddr = RdbUtils.getDatabaseDefinitionRegistry();
        return ddr.getDefinition(product, version);
    }
    
    public Database getActiveDatabase() throws Exception {
        String us = persistenceProperties.getConnectionUserName();
        String pa = persistenceProperties.getConnectionPassword();
        ConnectionInfo activeConnectionInfo = getActiveConnectionInfo();
        if(activeConnectionInfo == null){
            return null;
        }
        return getCatalogDatabase(this, activeConnectionInfo, us, pa);
    }
    
    public Schema getActiveSchema(){
        String schemaName = persistenceProperties.getSchemaName();
        if (schemaName != null) {
            String us = persistenceProperties.getConnectionUserName();
            String pa = persistenceProperties.getConnectionPassword();
            List schemas = getSchemas(this, getActiveConnectionInfo(), us, pa);
            if(schemas != null){
                for (Iterator it = schemas.iterator(); it.hasNext();) {
                    Schema schema = (Schema) it.next();
                    if (schemaName.equals(schema.getName())) {
                        return schema;
                    }
                }
            }
        }
        return null;
    }
    
    public static List getSchemas(OrmNature nature, ConnectionInfo info, String user, String pass){
        if(info == null){
            return Collections.EMPTY_LIST;
        }
        try {
            Database catalogDatabase = getCatalogDatabase(nature, info, user, pass);
            if (catalogDatabase != null) {
                return catalogDatabase.getSchemas();
            }
        } catch (Exception e) {
            OrmPlugin.log(null, "Could not connect to rdb connection: "+info.getName(), 
                    e, IStatus.WARNING);
        }
        return null;
    }
    
    private static Database getCatalogDatabase(OrmNature nature, ConnectionInfo info, String userName, 
            String password) throws Exception{
        Database sharedDatabase = info.getSharedDatabase();
        if(sharedDatabase != null){
            return sharedDatabase;
        }
        // try to connect
        Connection connection = info.getSharedConnection();
        if(connection == null){
            synchronized (info) {
                if (userName != null) {
                    info.setUserName(userName);
                }
                if (password != null) {
                    info.setPassword(password);
                }
                if(isPromptNeeded(info)){
                    promptIDPW(info, "Please enter your login info to reconnect to "+info.getDatabaseName()+".");
                    if(nature != null && !isPromptNeeded(info)){
                        PersistenceProperties persistenceProperties = nature.getPersistenceProperties();
                        persistenceProperties.setConnectionUserName(info.getUserName());
                        persistenceProperties.setConnectionPassword(info.getPassword());
                        try {
                            nature.internalSave = true;
                            persistenceProperties.save();
                        } finally {
                            nature.internalSave = false;
                        }
                    }
                }
                connection = info.connect();
                info.setSharedConnection(connection);
                info.saveConnectionInfo();
            }
//            SQLDBUtils.reestablishConnection(info);
        }
        new DatabaseProviderHelper().setDatabase(connection,
                info, info.getDatabaseName());
        sharedDatabase = info.getSharedDatabase();
        if (sharedDatabase == null) {
            ICatalogProvider catalogProvider = info.getDatabaseDefinition()
                    .getDatabaseCatalogProvider();
            sharedDatabase = catalogProvider.getCatalogDatabase(connection);
            sharedDatabase.setName(info.getName());
            if (sharedDatabase != null) {
                info.removeSharedDatabase();
                info.setSharedDatabase(sharedDatabase);
            }
        }
        return sharedDatabase;
    }

    /**
     * Determines if a user ID and password prompt is needed for the given
     * <code>ConnectionInfo</code> object.  This is done by checking if either 
     * the user ID or password is null or empty.
     * 
     * @param connInfo the <code>ConnectionInfo</code> to check
     * @param true when userid/password prompt is needed, otherwise false
     */
    private static boolean isPromptNeeded( ConnectionInfo connInfo ) {
        
        String username = null;
        String password = null;
        
        if (connInfo != null) {
            username = connInfo.getUserName();
            password = connInfo.getPassword();
        }
        
        return username == null || username.trim().length() == 0 || password == null || password.trim().length() == 0;
    }

    /**
     * Prompts for the user ID and password and updates the given <code>ConnectionInfo</code>
     * object
     * 
     * @param connInfo the <code>ConnectionInfo</code> object to update
     * @param promptMessage a message to display at the top of the prompt
     * @return true if the user clicks OK, otherwise false
     */
    private static synchronized boolean promptIDPW( ConnectionInfo connInfo, String promptMessage ) {
        boolean ok = false;
        if (connInfo != null) {
            String username = connInfo.getUserName();
            if (username == null || username.length() == 0)
                username = System.getProperty( "user.name" ); //$NON-NLS-1$
            UserIdentification idDialog = new UserIdentification( username, promptMessage );
            idDialog.setBlockOnOpen(true);
            if (idDialog.open() == Window.OK) {
                username = idDialog.getUserNameInformation();
                String password = idDialog.getPasswordInformation();
                connInfo.setUserName( username == null ? "" : username ); //$NON-NLS-1$
                connInfo.setPassword( password == null ? "" : password ); //$NON-NLS-1$
                ok = true;
                try {
                    connInfo.saveConnectionInfo();
                }
                catch (Exception e) {
                    ok = false;
                }
            }
        }
        return ok;
    }

    public void resourceChanged(IResourceChangeEvent event) {
        if(internalSave){
            return;
        }
        boolean closing = event.getType() == IResourceChangeEvent.PRE_CLOSE;
        if(closing){
            boolean isProject = project.equals(event.getResource());
            if(isProject){
                try {
                    closeProject();
                } catch (CoreException e) {
                    OrmPlugin.log(e);
                }
            }
            return;
        }
        IResourceDelta delta = event.getDelta();
        boolean propsChanged = delta.findMember(project.getFile(PersistenceProperties.PERSISTENCE_FILE_NAME).getFullPath()) != null;
        if(propsChanged && reLoadActiveProjectJob.getState() == Job.NONE){
            reLoadActiveProjectJob.schedule(50);
        }
    }

    public void fireNatureRemoved() {
        OrmNatureListener[] listeners = new OrmNatureListener[natureListeners.size()];
        natureListeners.toArray(listeners);
        OrmNatureEvent ormNatureEvent = new OrmNatureEvent(this, null);
        for(int x = listeners.length-1; x >= 0; x--){
            listeners[x].ormNatureRemoved(ormNatureEvent);
        }
        OrmNatureUtils.fireNatureRemoved(this);
    }
    
    public void fireProjectActivated(OrmProject activeProject){
        OrmNatureListener[] listeners = new OrmNatureListener[natureListeners.size()];
        natureListeners.toArray(listeners);
        OrmNatureEvent ormNatureEvent = new OrmNatureEvent(this, activeProject);
        for(int x = listeners.length-1; x >= 0; x--){
            listeners[x].projectActivated(ormNatureEvent);
        }
        OrmNatureUtils.fireProjectActivated(this, activeProject);
    }
    
    public void fireProjectDeactivated(OrmProject activeProject){
        OrmNatureListener[] listeners = new OrmNatureListener[natureListeners.size()];
        natureListeners.toArray(listeners);
        OrmNatureEvent ormNatureEvent = new OrmNatureEvent(this, activeProject);
        for(int x = listeners.length-1; x >= 0; x--){
            listeners[x].projectDeactivated(ormNatureEvent);
        }
        OrmNatureUtils.fireProjectDeactivated(this, activeProject);
    }
}
