/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.hibernate.internal;

import java.io.File;
import java.io.FileFilter;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.MalformedURLException;
import java.util.ArrayList;
import java.util.jar.JarEntry;
import java.util.jar.JarOutputStream;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.jdt.core.ClasspathContainerInitializer;
import org.eclipse.jdt.core.IClasspathContainer;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jsr220orm.core.nature.PersistenceProperties;
import org.eclipse.jsr220orm.core.util.RdbUtils;
import org.eclipse.jsr220orm.hibernate.HibernatePlugin;
import org.eclipse.wst.rdb.internal.core.connection.ConnectionInfo;

public class HibernateClasspathInitializer extends
        ClasspathContainerInitializer {


    private FileFilter jarFilter = new FileFilter() {
        public boolean accept(File pathname) {
            String name = pathname.getName().toLowerCase();
            return pathname.exists() && (name.endsWith(".jar") || name.endsWith(".zip"));
        }
    };
    private FileFilter dirFilter = new FileFilter() {
        public boolean accept(File pathname) {
            return pathname.exists() && pathname.isDirectory();
        }
    };

    public void initialize(IPath containerPath, IJavaProject project)
            throws CoreException {
        ArrayList entriesList = new ArrayList();
        try {
            PersistenceProperties props = new PersistenceProperties(project.getProject());
            String hibernatePath = props.getProperty("hibernate.path");
            String annotationsPath = props.getProperty("hibernate.annotations.path");
            String entitymanagerPath = props.getProperty("hibernate.entitymanager.path");
            addAllJarsFromPath(entriesList, hibernatePath);
            addAllJarsFromPath(entriesList, annotationsPath);
            addAllJarsFromPath(entriesList, entitymanagerPath);
            IProject project2 = project.getProject();
            String parFileName = props.getParFileName();
            IFile parIFile = project2.getFile(parFileName);
            IPath parPath = parIFile.getRawLocation();
            File parFile = new File(parPath.toOSString());
            if(!parFile.exists()){
                File parDir = parFile.getParentFile();
                if(!parDir.exists()){
                    parDir.mkdirs();
                }
                JarOutputStream jos = new JarOutputStream(
                        new FileOutputStream(parFile));
                JarEntry newEntry = new JarEntry(".");
                jos.putNextEntry(newEntry);
                jos.flush();
                jos.close();
                parIFile.refreshLocal(0, new NullProgressMonitor());
            }
            entriesList.add(0, JavaCore.newLibraryEntry(parIFile.getFullPath(), null, null));
            ConnectionInfo info = getActiveConnectionInfo(props);
            if (info != null) {
                String loadingPath = info.getLoadingPath();
                if (loadingPath != null) {
                    entriesList.add(JavaCore.newLibraryEntry(new Path(loadingPath), null, null));
                }
            }
        } catch (Exception e) {
            IStatus status = new Status(IStatus.ERROR, HibernatePlugin.getPluginId(), 0, 
                    "Problems adding Hibernate EJB3 jars to casspath", e); 
            throw new CoreException(status);
        }
        final IClasspathEntry[] entries = new IClasspathEntry[entriesList.size()];
        entriesList.toArray(entries);
        IClasspathContainer container = new IClasspathContainer() {
            public IClasspathEntry[] getClasspathEntries() {
                return entries;
            }

            public String getDescription() {
                return "Hibernate EJB3";
            }

            public IPath getPath() {
                return new Path("hibernatePath");
            }
        
            public int getKind() {
                return K_APPLICATION;
            }
        };
        JavaCore.setClasspathContainer(container.getPath(),
            new IJavaProject[] {project },
            new IClasspathContainer[] {container},
            null);
    }

    private void addAllJarsFromPath(ArrayList entries, String path) throws MalformedURLException, IOException {
        if(path == null || path.trim().length() == 0){
            return;
        }
        File dir = new File(path);
        if(!dir.exists()){
            return;
        }
        if(dir.isFile()){
            addFile(entries, dir);
            return;
        }
        addDir(entries, dir, false);
        addDir(entries, new File(dir,"lib"), true);
    }

    private void addDir(ArrayList entries, File dir, boolean search) throws MalformedURLException, IOException {
        File[] jars = dir.listFiles(jarFilter);
        for (int i = 0; i < jars.length; i++) {
            addFile(entries, jars[i]);
        }
        if(search){
            File[] dirs = dir.listFiles(dirFilter);
            for (int i = 0; i < dirs.length; i++) {
                addDir(entries, dirs[i], search);
            }
        }
    }

    private void addFile(ArrayList entries, File file) throws MalformedURLException, IOException {
        IClasspathEntry entry = JavaCore.newLibraryEntry(
                new Path(file.getAbsolutePath()), null, null);
        entries.add(entry);
    }

    public ConnectionInfo getActiveConnectionInfo(PersistenceProperties props) {
        String rdbConnName = props.getConnectionName();
        if (rdbConnName != null) {
            ConnectionInfo[] rdbConnections = RdbUtils.getRdbConnections();
            for (int x = rdbConnections.length - 1; x >= 0; x--) {
                if (rdbConnName.equals(rdbConnections[x].getName())) {
                    return rdbConnections[x];
                }
            }
        }
        return null;
    }
}