/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.generic;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.jar.JarEntry;
import java.util.jar.JarOutputStream;
import java.util.zip.CRC32;
import java.util.zip.ZipEntry;

import org.apache.xmlbeans.XmlException;
import org.apache.xmlbeans.XmlOptions;
import org.eclipse.core.resources.ICommand;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.jsr220Orm.xml.EntityManagerDocument;
import org.eclipse.jsr220Orm.xml.EntityManagerDocument.EntityManager;
import org.eclipse.jsr220orm.core.nature.OrmNature;
import org.eclipse.jsr220orm.core.nature.PersistenceProperties;

public class ParBuilder extends IncrementalProjectBuilder {

    public static final String ID = "org.eclipse.jsr220orm.generic.par.file.builder";

    protected static final String PERSISTENCE_XML_NAMESPACE = "http://jsr220orm.eclipse.org/xml";

    public ParBuilder() {
        super();
    }

    @Override
    protected IProject[] build(int kind, Map args, IProgressMonitor monitor)
            throws CoreException {
        IProject project = getProject();
        OrmNature nature = (OrmNature) project.getNature(OrmNature.ID);
        if (nature != null) {
            PersistenceProperties persistenceProperties = nature
                    .getPersistenceProperties();
            Properties props = persistenceProperties.getProperties();
            String name = props
                    .getProperty(PersistenceProperties.PROP_PERSISTENCE_XML);
            if (name == null) {
                name = "persistence.xml";
            }
            IFile file = project.getFile(name);
            String parFileName = persistenceProperties.getParFileName();
            IFile parIFile = project.getFile(parFileName);
            if (file.exists()) {
                try {
                    IPath parPath = parIFile.getRawLocation();
                    File parFile = new File(parPath.toOSString());
                    if(!parFile.exists()){
                        File parDir = parFile.getParentFile();
                        if(!parDir.exists()){
                            parDir.mkdirs();
                        }
                    }
                    JarOutputStream jos = new JarOutputStream(
                            new FileOutputStream(parFile));
                    InputStream contentStream = file.getContents(false);
                    addFile(jos, file, new Path("META-INF/persistence.xml"),
                            parFile);
                    jos.flush();
                    jos.close();
                    parIFile.refreshLocal(1, monitor);
                } catch (CoreException e) {
                    throw e;
                } catch (Exception e) {
                    IStatus status = new Status(IStatus.ERROR, GenericPlugin.getPluginId()
                            , 0, "Problem building .par file", e);
                    throw new CoreException(status);
                }
            }
        }
        return null;
    }

    /**
     * Creates a new JarEntry with the passed path and contents, and writes it
     * to the current archive.
     */
    protected void addFile(JarOutputStream jos, IFile resource, IPath path,
            File correspondingFile) throws IOException, CoreException {
        JarEntry newEntry = new JarEntry(path.toString().replace(
                File.separatorChar, '/'));
        byte[] readBuffer = new byte[4096];
        newEntry.setMethod(ZipEntry.STORED);
        calculateCrcAndSize(newEntry, resource, readBuffer);

        long lastModified = correspondingFile != null
                && correspondingFile.exists() ? correspondingFile
                .lastModified() : System.currentTimeMillis();
        // Set modification time
        newEntry.setTime(lastModified);
        InputStream contentStream = resource.getContents(false);

        try {
            jos.putNextEntry(newEntry);
            int count;
            while ((count = contentStream
                    .read(readBuffer, 0, readBuffer.length)) != -1)
                jos.write(readBuffer, 0, count);
        } finally {
            if (contentStream != null)
                contentStream.close();
        }
    }

    /**
     * Calculates the crc and size of the resource and updates the jarEntry.
     * 
     * @param jarEntry
     *            the JarEntry to update
     * @param resource
     *            the file to calculate
     * @param readBuffer
     *            a shared read buffer to store temporary data
     * 
     * @throws IOException
     *             if an I/O error has occurred
     * @throws CoreException
     *             if the resource can-t be accessed
     */
    private void calculateCrcAndSize(JarEntry jarEntry, IFile resource,
            byte[] readBuffer) throws IOException, CoreException {
        InputStream contentStream = resource.getContents(false);
        int size = 0;
        CRC32 checksumCalculator = new CRC32();
        int count;
        try {
            while ((count = contentStream
                    .read(readBuffer, 0, readBuffer.length)) != -1) {
                checksumCalculator.update(readBuffer, 0, count);
                size += count;
            }
        } finally {
            if (contentStream != null)
                contentStream.close();
        }
        jarEntry.setSize(size);
        jarEntry.setCrc(checksumCalculator.getValue());
    }

    /**
     * Create our XmlBeans graph from the contents of persistence.xml. If this
     * resource does not exist in the project then create a new persistence.xml.
     */
    protected EntityManager loadPersistenceXml(IProject project, String name)
            throws CoreException, XmlException, IOException {
        if (name == null) {
            name = "persistence.xml";
        }
        IFile file = project.getProject().getFile(name);
        EntityManagerDocument doc;
        if (file.exists()) {
            InputStream ins = file.getContents();
            try {
                doc = EntityManagerDocument.Factory.parse(ins,
                        getXmlOptions(PERSISTENCE_XML_NAMESPACE));
            } finally {
                Utils.close(ins);
            }
            return doc.getEntityManager();
        }
        return null;
    }

    /**
     * XmlOptions to use when loading and saving deployment descriptions that
     * already exist. This returns a copy of the options configured so that
     * defaultNamespace does not have to be declared in the input XML.
     */
    public XmlOptions getXmlOptions(String defaultNamespace) {
        XmlOptions ops = new XmlOptions();
        HashMap map = new HashMap();
        map.put("", defaultNamespace);
        ops.setLoadSubstituteNamespaces(map);
        return ops;
    }
}
