/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.generic.io;

import java.util.List;

import javax.persistence.Basic;
import javax.persistence.Column;
import javax.persistence.GeneratorType;
import javax.persistence.Id;
import javax.persistence.Transient;
import javax.persistence.Version;

import org.eclipse.jsr220orm.core.internal.options.IntOption;
import org.eclipse.jsr220orm.generic.GenericEntityModelManager;
import org.eclipse.jsr220orm.generic.Utils;
import org.eclipse.jsr220orm.generic.reflect.RAnnotatedElement;
import org.eclipse.jsr220orm.generic.reflect.RClass;
import org.eclipse.jsr220orm.metadata.AttributeMetaData;
import org.eclipse.jsr220orm.metadata.BasicAttribute;
import org.eclipse.jsr220orm.metadata.EntityMetaData;
import org.eclipse.jsr220orm.metadata.MetadataPackage;
import org.eclipse.jsr220orm.metadata.OrmColumn;
import org.eclipse.jsr220orm.metadata.SimpleTypeMetaData;
import org.eclipse.jsr220orm.metadata.TypeMetaData;

/**
 * Handles mapping for basic attributes (Basic and Column). 
 */
public class BasicAttributeIO extends AttributeIO {

	protected BasicAttribute amd;
	
	public static final IntOption MAPPING_BASIC = new IntOption(1,
			"Basic", "Single column in table for entity",
			Utils.getImage("Simple16"));
	
	public BasicAttributeIO(EntityIO entityIO) {
		super(entityIO);
	}
	
	public AttributeMetaData getAttributeMetaData() {
		return amd;
	}
	
    public void getPossibleMappings(List ans) {
        ans.add(MAPPING_BASIC);
    	ans.add(MAPPING_NOT_PERSISTENT);
    }	
	
	public IntOption getMapping() {
		if (amd == null || amd.isNonPersistent()) {
			return MAPPING_NOT_PERSISTENT;
		}
		return MAPPING_BASIC;
	}

	public void setMapping(IntOption mapping) {
		setPersistent(MAPPING_BASIC.equals(mapping));
	}

	public boolean updateModelFromMetaData(RClass cls,
			RAnnotatedElement attribute, boolean metaDataChanged) {
		
		amd = (BasicAttribute)initAttributeMetaData(amd, attribute, 
				MetadataPackage.eINSTANCE.getBasicAttribute());
		if (amd.isNonPersistent()) {
			return true;
		}
				
		GenericEntityModelManager mm = entityIO.getModelManager();
		EntityMetaData emd = entityIO.getEntityMetaData();
		
		if (amd.getJavaType() == null) {
			updateModelCleanup();
			addNullJavaTypeProblem(attribute);
			return true;
		}
		
		OrmColumn dc = getDefaultColumn();
		
		Basic basic = (Basic)getBasicAnnotation(attribute, dc);
		setFetchType(amd, basic.fetch());
		amd.setOptional(basic.optional());

		OrmColumn ormCol = entityIO.ensureOrmColumn(amd.getColumn());
		Column column = (Column)getColumnAnnotation(attribute, dc);
		ormCol.setJdbcType(column.jdbcType());
		ormCol.setInsertable(column.insertable());
		if (mm.getVendorDef().isNumericType(ormCol.getJdbcType())) {
			ormCol.setLength(column.precision());		
		} else {
			ormCol.setLength(column.length());			
		}
		ormCol.setScale(column.scale());
		ormCol.setName(column.name());
		ormCol.setNullable(column.nullable());
		ormCol.setUpdatable(column.updatable());
		ormCol.setTable(emd.getTable());
		String cd = column.columnDefinition();
		if (cd.length() == 0) {
			ormCol.setColumnDefinitionSpecified(false);
			cd = mm.getColumnDefinition(ormCol);
		} else {
			ormCol.setColumnDefinitionSpecified(true);
		}
		ormCol.setColumnDefinition(cd);
		ormCol.setOriginalColumnDefinition(cd);
		mm.updateDatabaseType(ormCol);
		ormCol.setComment(getComment(amd));
		
		if (amd.getColumn() != ormCol) {
			amd.setColumn(ormCol);
		}
		
		int relativePositionInTable = getRelativePosition(attribute);
		
		Version version = attribute.getAnnotation(Version.class);
		Id id = attribute.getAnnotation(Id.class);
		
		if (emd.isBaseEntity() || emd.isEmbeddableSuperclass()) {
			if (id != null) {
				boolean hasValues = ((AnnotationEx)id).getValueCount() > 0;
				if (hasValues) {
					if (entityIO.getIdGeneratorAttribute() != null) {
						String msg = "Only one Id attribute is allowed to " + 
							"have generate and generator values";
						entityIO.addProblem(msg, ((AnnotationEx)id).getLocation(null));
					} else {
						entityIO.setIdGeneratorAttribute(amd);
						emd.setIdGeneratorType(getGeneratorTypeCode(id.generate()));
						emd.setIdGeneratorName(id.generator());
					}
				}
				updateModelSetPrimaryKey(true);
				if (version != null) {
					entityIO.addProblem("A primary key (Id) attribute may not " +
							"also be a Version attribute", 
							((AnnotationEx)version).getLocation(null));
					version = null;
				}
				relativePositionInTable += mm.getColumnPositionPrimaryKey();
			} else {
				updateModelSetPrimaryKey(false);
			}
			
			if (version != null) {
				if (!amd.isVersion()) {
					emd.getVersionList().add(amd);
				}
				relativePositionInTable += mm.getColumnPositionVersion();
			} else {
				if (amd.isVersion()) {
					emd.getVersionList().remove(amd);
				}
			}

		} else { // not a base class
			if (id != null) {
				String msg;
				if (emd.isEntity()) {
					msg = "Subclasses may not have Id attributes";
				} else {
					msg = getTypeName(emd) + " may not have Id attributes";
				}
				entityIO.addProblem(msg, id);
			}
			if (version != null) {
				String msg;
				if (emd.isEntity()) {
					msg = "Subclasses may not have Version attributes";
				} else {
					msg = getTypeName(emd) + " may not have Version attributes";
				}
				entityIO.addProblem(msg, version);				
			}
			updateModelSetPrimaryKey(false);
			if (amd.isVersion()) {
				emd.getVersionList().remove(amd);
			}
		}
				
		ormCol.setRelativePositionInTable(relativePositionInTable);

		return true;
	}

	/**
	 * Get rid of anything for our attribute currently in the model. 
	 */
	protected void updateModelCleanup() {
		OrmColumn c = amd.getColumn();
		if (c != null) {
			c.delete();
			amd.setColumn(null);
		}
		EntityMetaData emd = entityIO.getEntityMetaData();
		if (amd.isPrimaryKey()) {
			emd.getPrimaryKeyList().remove(amd);
		}
		if (amd.isVersion()) {
			emd.getVersionList().remove(amd);
		}
	}

	/**
	 * Change the primary key status of our attribute. This will also change
	 * our column. 
	 */
	protected void updateModelSetPrimaryKey(boolean on) {
		EntityMetaData emd = entityIO.getEntityMetaData();
		OrmColumn ormCol = amd.getColumn();
		if (on) { 
			if (!amd.isPrimaryKey()) {
				emd.getPrimaryKeyList().add(amd);
			}
			if (!ormCol.isPrimaryKey()) {
				ormCol.getTable().getPrimaryKeyList().add(ormCol);
			}
			ormCol.setIdentity(getModelManager().isUsingIdentityColumns(emd));			
		} else {
			if (amd.isPrimaryKey()) {
				emd.getPrimaryKeyList().remove(amd);
			}
			if (ormCol != null) {
				if (ormCol.isPrimaryKey()) {
					emd.getTable().getPrimaryKeyList().remove(ormCol);
				}
				if (ormCol.isIdentity()) {
					ormCol.setIdentity(false);
				}
			}
		}
	}
	
	protected int getGeneratorTypeCode(GeneratorType t) {
		switch (t) {
		case NONE:		return EntityMetaData.ID_GENERATOR_TYPE_NONE;
		case AUTO:		return EntityMetaData.ID_GENERATOR_TYPE_AUTO;
		case IDENTITY:	return EntityMetaData.ID_GENERATOR_TYPE_IDENTITY;
		case SEQUENCE:	return EntityMetaData.ID_GENERATOR_TYPE_SEQUENCE;
		case TABLE:		return EntityMetaData.ID_GENERATOR_TYPE_TABLE;
		}
		return EntityMetaData.ID_GENERATOR_TYPE_NONE;
	}

	protected GeneratorType getGeneratorType(int code) {
		switch (code) {
		case EntityMetaData.ID_GENERATOR_TYPE_NONE:		
			return GeneratorType.NONE;
		case EntityMetaData.ID_GENERATOR_TYPE_AUTO:
			return GeneratorType.AUTO;
		case EntityMetaData.ID_GENERATOR_TYPE_IDENTITY:
			return GeneratorType.IDENTITY;
		case EntityMetaData.ID_GENERATOR_TYPE_SEQUENCE:
			return GeneratorType.SEQUENCE;
		case EntityMetaData.ID_GENERATOR_TYPE_TABLE:
			return GeneratorType.TABLE;
		}
		return GeneratorType.NONE;
	}
	
	protected OrmColumn getDefaultColumn() {
		TypeMetaData tmd = amd.getJavaType();
		if (tmd instanceof SimpleTypeMetaData) {
			return ((SimpleTypeMetaData)tmd).getColumn();
		} else {
			return null;
		}		
	}
	
	protected AnnotationEx getBasicAnnotation(RAnnotatedElement attribute,
			OrmColumn dc) {
		AnnotationEx basic = (AnnotationEx)attribute.getAnnotation(
				Basic.class, true);
		if (dc != null) {
			basic.setDefault("optional", dc.isNullable());
		}
		return basic;
	}
	
	protected AnnotationEx getColumnAnnotation(RAnnotatedElement attribute,
			OrmColumn dc) {
		AnnotationEx column = (AnnotationEx)attribute.getAnnotation(
				Column.class, true);
		column.setDefault("name", 
				entityIO.getModelManager().getDefaultColumnName(amd));
		column.setDefault("nullable", amd.isOptional());
		if (dc != null) {
			column.setDefault("jdbcType", dc.getJdbcType());
			column.setDefault("length", dc.getLength());
			column.setDefault("precision", dc.getLength());
			column.setDefault("scale", dc.getScale());
		}
		return column;
	}

	public void updateMetaDataFromModel(RClass cls, RAnnotatedElement attribute) {

		GenericEntityModelManager mm = getModelManager();
		
		if (amd.isNonPersistent()) {
			Utils.removeAnnotation(attribute, Basic.class);
			Utils.removeAnnotation(attribute, Column.class);
			Utils.removeAnnotation(attribute, Version.class);
			Utils.removeAnnotation(attribute, Id.class);
			ensureTransient(attribute);
			return;
		} else {
			Utils.removeAnnotation(attribute, Transient.class);
		}
		
		if (amd.getJavaType() == null) {
			return;
		}		
		
		OrmColumn dc = getDefaultColumn();		
		AnnotationEx basic = getBasicAnnotation(attribute, dc);
		Utils.setIfNotNull(basic, "fetch", getFetchType(amd));
		basic.set("optional", amd.isOptional());
		basic.setMarker(mm.isUseMarkerAnnotations());
		
		OrmColumn oc = amd.getColumn();
		if (oc == null) {
			// we were non-persistent previously - the column will be created
			// during the re-read with defaults so we have nothing more todo
			return;			
		}
		
		AnnotationEx column = getColumnAnnotation(attribute, dc);
		column.set("jdbcType", oc.getJdbcType());
		column.set("insertable", oc.isInsertable());
		if (mm.getVendorDef().isNumericType(oc.getJdbcType())) {
			column.set("precision", oc.getLength());
			column.set("length", null);
		} else {
			column.set("length", oc.getLength());	
			column.set("precision", null);		
		}
		column.set("name", oc.getName(), true);
		oc.setName(((Column)column).name());
		column.set("nullable", oc.isNullable());
		column.set("scale", oc.getScale());
		column.set("updatable", oc.isUpdatable());

		// if the columnDefinition has changed then apply the change to the
		// annotation - this may remove it if it now matches the default
		boolean colDefChanged = !oc.getColumnDefinition().equals(
				oc.getOriginalColumnDefinition());
		String stdColDef = mm.getColumnDefinition(oc);
		if (colDefChanged) {
			column.setDefault("columnDefinition", stdColDef);			
			column.set("columnDefinition", oc.getColumnDefinition(), true);
		}
		
		// if the annotation now has no columnDefinition value then put
		// the default value into the model
		if (!column.hasValue("columnDefinition")) {
			oc.setColumnDefinition(stdColDef);
		}
		
		// reset the originalColumnDefinition attribute to detect more changes
		oc.setOriginalColumnDefinition(oc.getColumnDefinition());
		
		boolean primaryKey = amd.isPrimaryKey();
        AnnotationEx id = (AnnotationEx)attribute.getAnnotation(Id.class, 
        		primaryKey);
		if (primaryKey) {
			EntityMetaData emd = entityIO.getEntityMetaData();
			BasicAttribute ida = entityIO.getIdGeneratorAttribute();
			if (ida == null || ida == amd) {
				id.set("generate", getGeneratorType(emd.getIdGeneratorType()));
				id.set("generator", emd.getIdGeneratorName());	
				if (ida == null) {
					entityIO.setIdGeneratorAttribute(amd);
				}
			} else {
				id.set("generate", null);
				id.set("generator", null);								
			}
			id.setMarker(true);
		} else {
			if (id != null) {
				id.delete();
			}
		}

		boolean isVersion = amd.isVersion();
        AnnotationEx version = (AnnotationEx)attribute.getAnnotation(
        		Version.class, isVersion);
		if (isVersion) {
			version.setMarker(true);
		} else {
			if (version != null) {
				version.delete();
			}
		}
		
		mm.updateDatabaseType(oc);
	}

}
