/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.generic.io;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.persistence.AttributeOverride;
import javax.persistence.AttributeOverrides;
import javax.persistence.Embedded;
import javax.persistence.Transient;

import org.eclipse.jsr220orm.core.internal.options.IntOption;
import org.eclipse.jsr220orm.generic.GenericEntityModelManager;
import org.eclipse.jsr220orm.generic.Utils;
import org.eclipse.jsr220orm.generic.reflect.RAnnotatedElement;
import org.eclipse.jsr220orm.generic.reflect.RClass;
import org.eclipse.jsr220orm.metadata.AttributeMetaData;
import org.eclipse.jsr220orm.metadata.BasicAttribute;
import org.eclipse.jsr220orm.metadata.EmbeddedAttribute;
import org.eclipse.jsr220orm.metadata.EntityMetaData;
import org.eclipse.jsr220orm.metadata.MetadataPackage;

/**
 * Handles mapping for embedded attributes. 
 */
public class EmbeddedAttributeIO extends AttributeIO {

	protected EmbeddedAttribute amd;
	
	public static final IntOption MAPPING_EMBEDDED = new IntOption(1,
			"Embedded", "Attributes of referenced class in table for entity",
			Utils.getImage("Embedded16"));
	
	public EmbeddedAttributeIO(EntityIO entityIO) {
		super(entityIO);
	}
	
	public AttributeMetaData getAttributeMetaData() {
		return amd;
	}
	
    public void getPossibleMappings(List ans) {
        ans.add(MAPPING_EMBEDDED);
    	ans.add(MAPPING_NOT_PERSISTENT);
    }	
	
	public IntOption getMapping() {
		if (amd == null || amd.isNonPersistent()) {
			return MAPPING_NOT_PERSISTENT;
		}
		return MAPPING_EMBEDDED;
	}

	public void setMapping(IntOption mapping) {
		setPersistent(MAPPING_EMBEDDED.equals(mapping));
	}
	
	public boolean updateModelFromMetaData(RClass cls,
			RAnnotatedElement attribute, boolean metaDataChanged) {
		
		amd = (EmbeddedAttribute)initAttributeMetaData(amd, attribute, 
				MetadataPackage.eINSTANCE.getEmbeddedAttribute());
		if (amd.isNonPersistent()) {
			return true;
		}
				
		GenericEntityModelManager mm = entityIO.getModelManager();
		EntityMetaData emd = entityIO.getEntityMetaData();
		
		if (amd.getJavaType() == null) {
			updateModelCleanup();
			addNullJavaTypeProblem(attribute);
			return true;
		}

		EntityMetaData target = (EntityMetaData)amd.getJavaType();
		if (!hasBasicAttributes(target)) {
			return false;
		}
		
		entityIO.addDependencyOn(target);
		
		// remove any of our attributes that have a null overrideOf i.e. that
		// were referencing an attribute that no longer exists + build a map
		Map<AttributeMetaData, AttributeMetaData> found = new HashMap();
		List attrList = amd.getAttributeList();
		for (int i = attrList.size() - 1; i >= 0; ) {
			AttributeMetaData a = (AttributeMetaData)attrList.get(i);
			AttributeMetaData overrideOf = a.getOverrideOf();
			if (overrideOf == null) {
				AttributeIO aio = AttributeIO.get(a);
				if (aio != null) {
					aio.dispose();
				}
				a.delete();
			} else {
				found.put(overrideOf, a);
				--i;
			}
		}
		
		// now create new attributes to override new ones in our target
		// and update the ones we already have
		AttributeOverrideMap map = new AttributeOverrideMap(attribute, entityIO);
		int pos = getRelativePosition(attribute);
		for (Iterator i = target.getAttributeList().iterator(); i.hasNext(); ) {
			AttributeMetaData overrideOf = (AttributeMetaData)i.next();
			OverrideAttributeIO aio;
			AttributeMetaData a = found.get(overrideOf);
			if (a == null) {
				if (!(overrideOf instanceof BasicAttribute)) {
					// problem will have already been created by target so
					// just ignore non-basic attributes
					continue;
				}
				aio = new OverrideAttributeIO(entityIO, amd);
			} else {
				aio = (OverrideAttributeIO)AttributeIO.get(a);
			}
			String name = overrideOf.getName();
			AttributeOverride attrOverride = map.get(name);
			if (attrOverride != null) {
				map.remove(name);
			}
			aio.updateModelFromMetaData(cls, overrideOf, attrOverride, 
					metaDataChanged, pos++);
		}
		
		// add problems for all unused AttributeOverride's
		for (AttributeOverride override : map.values()) {
			entityIO.addProblem("AttributeOverride does not match any embedded " +
					"attributes: " + override.name(), override);
		}		
		
		return true;
	}

	/**
	 * Get rid of anything for our attribute currently in the model. 
	 */
	protected void updateModelCleanup() {
		List attrList = amd.getAttributeList();
		for (int i = attrList.size() - 1; i >= 0; ) {
			AttributeMetaData a = (AttributeMetaData)attrList.get(i);
			a.delete();
		}
	}

	/**
	 * Does target have a table and its basic attributes? This
	 * will return false if target is null.
	 */
	protected boolean hasBasicAttributes(EntityMetaData target) {
		if (target == null || target.getTable() == null 
				|| EntityIO.get(target).getModelUpdateStatus() 
				< EntityIO.STATUS_STARTED_ATTRIBUTES) {
			return false;
		}
		return true;
	}
	
	public void updateMetaDataFromModel(RClass cls, RAnnotatedElement attribute) {

		boolean nonPersistent = amd.isNonPersistent();
		if (nonPersistent) {
			Utils.removeAnnotation(attribute, Embedded.class);
			Utils.removeAnnotation(attribute, AttributeOverride.class);
			Utils.removeAnnotation(attribute, AttributeOverrides.class);
			ensureTransient(attribute);
			return;
		} else {
			Utils.removeAnnotation(attribute, Transient.class);
		}
		
		if (amd.getJavaType() == null) {
			return;
		}		
		
		AttributeOverrideMap map = new AttributeOverrideMap(attribute, null);
		AttributeOverride extra = null;
		for (Iterator i = amd.getAttributeList().iterator(); i.hasNext(); ) {
			AttributeMetaData a = (AttributeMetaData)i.next();
			OverrideAttributeIO aio = (OverrideAttributeIO)AttributeIO.get(a);
			String name = a.getName();
			AttributeOverride override = map.get(name);
			if (override == null) {
				if (extra == null) {
					extra = map.append();
				}
				override = extra;
			}
			if (aio.updateMetaDataFromModel(override)) {
				if (override == extra && ((AnnotationEx)extra).hasValue("column")) {
					((AnnotationEx)extra).set("name", name);
					extra = null;
				}
			}
		}	
		map.cleanupAfterMetaDataUpdates();
	}

}
