/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.generic.io;

import java.lang.annotation.Annotation;
import java.lang.reflect.Constructor;
import java.sql.Types;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.persistence.AccessType;
import javax.persistence.AttributeOverride;
import javax.persistence.DiscriminatorColumn;
import javax.persistence.DiscriminatorType;
import javax.persistence.Embeddable;
import javax.persistence.EmbeddableSuperclass;
import javax.persistence.Embedded;
import javax.persistence.Entity;
import javax.persistence.IdClass;
import javax.persistence.Inheritance;
import javax.persistence.InheritanceType;
import javax.persistence.JoinColumn;
import javax.persistence.JoinColumns;
import javax.persistence.PrimaryKeyJoinColumn;
import javax.persistence.PrimaryKeyJoinColumns;
import javax.persistence.Table;
import javax.persistence.Transient;

import org.eclipse.core.resources.IMarker;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdapterFactory;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.jdt.core.ElementChangedEvent;
import org.eclipse.jdt.core.IElementChangedListener;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaElementDelta;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jsr220orm.core.OrmPlugin;
import org.eclipse.jsr220orm.core.internal.options.IntOption;
import org.eclipse.jsr220orm.core.options.IIntOption;
import org.eclipse.jsr220orm.generic.GenericEntityModelManager;
import org.eclipse.jsr220orm.generic.Utils;
import org.eclipse.jsr220orm.generic.io.ast.AstState;
import org.eclipse.jsr220orm.generic.reflect.RAnnotatedElement;
import org.eclipse.jsr220orm.generic.reflect.RClass;
import org.eclipse.jsr220orm.metadata.AttributeMetaData;
import org.eclipse.jsr220orm.metadata.BasicAttribute;
import org.eclipse.jsr220orm.metadata.CollectionTypeMetaData;
import org.eclipse.jsr220orm.metadata.EntityMetaData;
import org.eclipse.jsr220orm.metadata.EntityModel;
import org.eclipse.jsr220orm.metadata.Join;
import org.eclipse.jsr220orm.metadata.JoinPair;
import org.eclipse.jsr220orm.metadata.OrmColumn;
import org.eclipse.jsr220orm.metadata.OrmTable;
import org.eclipse.jsr220orm.metadata.TypeMetaData;
import org.eclipse.ui.PlatformUI;

/**
 * Maps between an Entity and its meta data (source code annotations and
 * XML). When the model changes the meta data is updated and visa versa.
 * One of these is created to manage each entity in the model.
 */
public class EntityIO implements IElementChangedListener, Adapter, Comparable,
		IAdapterFactory {

	protected final GenericEntityModelManager mm;
	protected final EntityMetaData emd;
	protected final IType type;
	protected final JoinIO joinIO;
	
	/** The non-persistent attributes by attribute name. */
	protected Map<String, AttributeMetaData> npAttributeMap = new HashMap();
	
	protected AstState astState;
	protected List<AttributeInfo> todo;
	protected Map<String, String> renamedAttributeMap;
	protected int oldAccessType;
		
	protected BasicAttribute idGeneratorAttribute;
	protected List<IMarker> markers = new ArrayList();
	
	protected Set<EntityIO> weDependOn = new HashSet();
	protected Set<EntityIO> dependOnUs = new HashSet();

	/** We have not started updating our model. **/
	public static final int STATUS_NOT_STARTED = 0;
	/** We have filled in our superEntity. */
	public static final int STATUS_SUPER_ENTITY_DONE = 1;
	/** We have done at least one update pass through our attributes. */
	public static final int STATUS_STARTED_ATTRIBUTES = 2;
	/** We have completed updating our model. */
	public static final int STATUS_COMPLETE = 10;
	
	protected int modelUpdateStatus;
	
	protected boolean updateAddDiscriminatorColumn;
	protected String defaultDiscriminatorValue;
	
    public static final IntOption MAPPING_NOT_PERSISTENT = 
    	new IntOption(0, "Not persistent", 
    			"Class is not stored in the database",
    			Utils.getImage("NotPersistent16"));	
    
    public static final IntOption MAPPING_ENTITY = 
    	new IntOption(1, "Entity", 
    			"Class is stored in the database with its own identity",
    			Utils.getImage("TreeClass16"));	
    
    public static final IntOption MAPPING_EMBEDDABLE = 
    	new IntOption(2, "Embeddable", 
    			"Class attributes are stored in the table for another class",
    			Utils.getImage("Embedded16"));	
    
    public static final IntOption MAPPING_EMBEDDABLE_SUPERCLASS = 
    	new IntOption(3, "Embeddable superclass", 
    			"Class attributes are stored in the tables for its subclasses",
    			Utils.getImage("EmbeddableSuperclass16"));	
    
	protected static final String[] DEFAULT_PACKAGES = new String[]{
		"java.lang.",
		"java.math.",
		"java.util.",
		"java.sql.",
	};
	
	/**
	 * Create a new EntityIO and attach it to the entity.
	 */
	public EntityIO(GenericEntityModelManager mm, EntityMetaData emd, 
			IType type) {
		this.emd = emd;
		this.mm = mm;
		this.type = type;
		joinIO = createJoinIO();
		emd.registerAdapterFactory(this);
		emd.eAdapters().add(this);
        JavaCore.addElementChangedListener(this,
                ElementChangedEvent.POST_RECONCILE);
	}
	
	protected JoinIO createJoinIO() {
		return new JoinIO(this);
	}

	/**
	 * Get the EntityIO for the EntityMetaData or null if none. 
	 */
	public static EntityIO get(EntityMetaData emd) {
		if (emd == null) {
			return null;
		}
		return (EntityIO)emd.adapt(EntityIO.class);		
	}
	
	/**
	 * This is called when our entity is removed so we can cleanup markers 
	 * and other stuff associated with the entity. 
	 */
	public void dispose() {
		deleteMarkers();
		clearWeDependOn();
		weDependOn = null;
		for (Iterator<EntityIO> i = dependOnUs.iterator(); i.hasNext(); ) {
			Set<EntityIO> set = i.next().weDependOn;
			if (set != null) {
				set.remove(this);
			}
		}
		dependOnUs = null;
		for (Iterator<AttributeMetaData> i = npAttributeMap.values().iterator(); 
				i.hasNext(); ) {
			i.next().delete();
		}
		npAttributeMap = null;
		emd.eAdapters().remove(this);
        JavaCore.removeElementChangedListener(this);
	}

	/**
	 * Clear the set of entities that we depend on. 
	 */
	protected void clearWeDependOn() {
		for (Iterator<EntityIO> i = weDependOn.iterator(); i.hasNext(); ) {
			Set<EntityIO> set = i.next().dependOnUs;
			if (set != null) {
				set.remove(this);
			}
		}
	}
	
	/**
	 * Delete all of the markers associated with our entity.
	 */
	protected void deleteMarkers() {
		for (IMarker m : markers) {
			try {
				m.delete();
			} catch (CoreException e) {
				OrmPlugin.log(e);
			}
		}
	}
	
	/**
	 * Get ready to update the model from our meta data. This is called once on
	 * each EntityIO's to be updated before any calls to 
	 * {@link #updateModelFromMetaData()} start.
	 */
	public void updateModelFromMetaDataPre() throws Exception {
		todo = null;
        idGeneratorAttribute = null;
        modelUpdateStatus = STATUS_NOT_STARTED;
        renamedAttributeMap = null;
        deleteMarkers();        
		astState = mm.getAstRClassFactory().getAstState();	
		clearWeDependOn();
	}

	/**
	 * Figure out which attributes have been renamed and update their names
	 * in the model. They are also added to the renamedAttributeMap.
	 * Currently this handles renaming of a single field or
	 * method.
	 * 
	 * @see #getNewAttributeName(String)
	 */
	public void renameAttributes(ElementChangedEvent event) {
		renamedAttributeMap = null;
		IJavaElementDelta delta = findOurTypeDelta(event);
		if (delta == null) {
			return;
		}
		IJavaElementDelta[] added = delta.getAddedChildren();
		IJavaElementDelta[] removed = delta.getRemovedChildren();
		if (added.length != 1 || removed.length != 1) {
			return;
		}
		IJavaElement a = added[0].getElement();
		IJavaElement r = removed[0].getElement();
		int elementType = a.getElementType();
		if (elementType != r.getElementType()) {
			return;
		}
		boolean field;
		switch (elementType) {
		case IJavaElement.FIELD:	
			field = true;	
			break;
		case IJavaElement.METHOD:	
			field = false;	
			break;
		default:					
			return;
		}
		String oldName = r.getElementName();
		String newName = a.getElementName();
		if (!field) {
			if (!mm.isValidPropertyName(oldName) 
					|| !mm.isValidPropertyName(newName)) {
				return;
			}
			oldName = Utils.getAttributeNameForMethod(oldName);
			newName = Utils.getAttributeNameForMethod(newName);
		}
		AttributeMetaData amd = emd.findAttributeMetaData(oldName);
		if (amd != null && amd.isField() == field) {
			System.out.println("$ " + oldName + " -> " + newName + ": " + amd);
			renamedAttributeMap = new HashMap();
			renamedAttributeMap.put(oldName, newName);
			amd.setName(newName);
		}
	}

	/**
	 * Find the delta for our type or null if none.
	 */
	protected IJavaElementDelta findOurTypeDelta(ElementChangedEvent event) {
		if (event == null) {
			return null;
		}
        IJavaElementDelta delta = (IJavaElementDelta)event.getSource();
        if (!delta.getElement().getAncestor(IJavaElement.COMPILATION_UNIT).equals(
        		type.getParent())) {
        	return null;
        }
        String ourName = type.getElementName();
    	IJavaElementDelta[] changed = delta.getChangedChildren();
    	if (changed.length == 0) {
    		return null;
    	}
    	for (int i = 0; i < changed.length; i++) {
    		IJavaElement e = changed[i].getElement();
    		if (e.getElementType() == IJavaElement.TYPE 
    				&& e.getElementName().equals(ourName)) {
    			return changed[i];
    		}
    	}
    	return null;
	}
	
	protected void print(IJavaElementDelta delta, Map<String, String> ans,
			String indent) {
		String kind = "U";
		switch (delta.getKind()) {
		case IJavaElementDelta.ADDED:	kind = "A";		break;
		case IJavaElementDelta.CHANGED:	kind = "C";		break;
		case IJavaElementDelta.REMOVED:	kind = "R";		break;
		}
		String tp;
		switch (delta.getElement().getElementType()) {
		case IJavaElement.FIELD:	tp = "FIELD";		break;
		case IJavaElement.METHOD:	tp = "METHOD";	break;
		case IJavaElement.TYPE:		tp = "TYPE";		break;
		default:	
			tp = "type " + delta.getElement().getElementType();
		}
		System.out.println("> " + indent + kind + " " + tp + " " + 
				delta.getElement().getElementName() + " " + 
				(delta.getElement() == type));
    	indent = indent + "  ";
    	IJavaElementDelta[] affected = delta.getAffectedChildren();
    	for (int i = 0; i < affected.length; i++) {
    		print(affected[i], ans, indent);
    	}
	}

	/**
	 * Update the entity from its meta data (annotations and XML). This
	 * method must return true if progress was made on this pass. If all 
	 * entities being processed return false then the processing loop can 
	 * stop as it is not possible to complete the model. The
	 * {@link #getModelUpdateStatus()} indicates if processing for the entity 
	 * has been completed. It is not necessary to return true if the status
	 * was changed.</p>
	 * 
	 * The metaDataChanged parameter is true If our meta data may have
	 * changed. This is used to decide what to do when column names	
	 * (e.g. referencedColumnName) and so on in the meta data no longer exist 
	 * in the model. If our meta data is unchanged and we still have a column 
	 * then the name of the column has probably changed and our meta data 
	 * should be written out again. If our meta data has changed then a 
	 * problem is created.</p>
	 */
	public boolean updateModelFromMetaData(RClass cls, boolean metaDataChanged) 
			throws Exception {
		
		if (modelUpdateStatus == STATUS_NOT_STARTED) {
			// fill in our superEntity and entityType on a separate pass so 
			// the model class heirachy information is complete for all 
			// classes before any inheritance processing is done
			updateModelSuperEntity(cls);
			modelUpdateStatus = STATUS_SUPER_ENTITY_DONE;
			return true;
		}

		if (todo == null) {

			// if we have a superclass then don't start our processing until 
			// it has a primary key, table and so on
			EntityMetaData superEntity = emd.getSuperEntity();
			if (superEntity != null) {
				EntityIO entityIO = EntityIO.get(superEntity);
				if (entityIO == null || entityIO.getModelUpdateStatus() < 
						STATUS_STARTED_ATTRIBUTES) {
					return false;
				}
			}
			
			todo = new ArrayList();
			
			defaultDiscriminatorValue = mm.getDefaultDiscriminatorValue(emd);
			switch (emd.getEntityType()) {
			case EntityMetaData.TYPE_ENTITY:
				if (emd.isBaseEntity()) {
					updateModelInheritanceBase(cls, metaDataChanged);
				} else {
					updateModelInheritanceNonBase(cls, metaDataChanged);				
				}
				break;
			case EntityMetaData.TYPE_EMBEDDABLE_SUPERCLASS:
			case EntityMetaData.TYPE_EMBEDDABLE:
				updateModelInheritanceEmbeddable(cls, metaDataChanged);
				break;
			};
			
			IdClass idc = cls.getAnnotation(IdClass.class, true);
			emd.setIdClass(((AnnotationEx)idc).getClassValue("value"));
			
			fillTodoWithAttributeInfos(cls);
		}

		// process attributes
		List<AttributeInfo> newTodo = null;
		for (Iterator<AttributeInfo> i = todo.iterator(); i.hasNext(); ) {
			AttributeInfo info = i.next();
			try {
				if (!info.updateModelFromMetaData(cls, metaDataChanged)) {
					if (newTodo == null) {
						newTodo = new ArrayList();
					}
					newTodo.add(info);
				}
			} catch (Exception e) {
				OrmPlugin.log(emd.getClassName() + "." + info, e);
			}
		}
		modelUpdateStatus = STATUS_STARTED_ATTRIBUTES;

		if (newTodo == null) {
			if (idGeneratorAttribute == null) {
				emd.setIdGeneratorName(null);
				emd.setIdGeneratorType(EntityMetaData.ID_GENERATOR_TYPE_NONE);
			}
			OrmTable table = emd.getTable();
			if (table != null) {
				table.sortColumns();
			}
			modelUpdateStatus = STATUS_COMPLETE;
			todo = null;
			return true;
		}
	
		// we did something if at least one attribute completed its processing
		// on this pass
		boolean ans = newTodo.size() < todo.size();
		todo = newTodo;
		return ans;
	}

	/**
	 * Fill in our superEntity, entityType, accessType and name.
	 */
	protected void updateModelSuperEntity(RClass cls) {
		String scName = cls.getSuperclassName();
		EntityMetaData superEntity = 
			scName == null ? null : mm.getEntityMetaData(scName);
		if (emd.getSuperEntity() != superEntity) {
			emd.setSuperEntity(superEntity);
		}
		if (superEntity != null) {
			addDependencyOn(superEntity);				
		}

		Entity entity = cls.getAnnotation(Entity.class);
		Embeddable emb = cls.getAnnotation(Embeddable.class);
		EmbeddableSuperclass embSup = cls.getAnnotation(
				EmbeddableSuperclass.class);
		
		if (entity == null && emb == null && embSup == null) {
			// use the current entityType to decide what annotation to use
			AnnotationRegistry reg = mm.getAnnotationRegistry();
			switch (emd.getEntityType()) {
			case EntityMetaData.TYPE_EMBEDDABLE:
				emb = reg.getDefaultProxyEx(Embeddable.class);
				break;
			case EntityMetaData.TYPE_EMBEDDABLE_SUPERCLASS:
				embSup = reg.getDefaultProxyEx(EmbeddableSuperclass.class);
				break;
			default:
				entity = reg.getDefaultProxyEx(Entity.class);
			}
			// this will put the annotation into the code after model update
			registerForMetaDataUpdate();
		}
		
		AccessType access = null;
		AnnotationEx main;
		if (entity != null) {
			emd.setEntityType(EntityMetaData.TYPE_ENTITY);
			if (emb != null) {
				addProblem("Cannot have Embeddable and Entity", emb);
			}
			if (embSup != null) {
				addProblem("Cannot have EmbeddableSuperclass and Entity", 
						embSup);
			}
			String name = entity.name();
			if (name.length() == 0) {
				name = cls.getSimpleName();
			}
			emd.setSchemaName(name);
			access = entity.access();
		} else if (emb != null) {
			emd.setEntityType(EntityMetaData.TYPE_EMBEDDABLE);
			if (embSup != null) {
				addProblem("Cannot have EmbeddableSuperclass and Embeddable", 
						embSup);
			}
			access = emb.access();
		} else {
			emd.setEntityType(EntityMetaData.TYPE_EMBEDDABLE_SUPERCLASS);
			access = embSup.access();
		}

		if (access != null && access == AccessType.FIELD) {
			emd.setAccessType(EntityMetaData.ACCESS_TYPE_FIELD);
		} else {
			emd.setAccessType(EntityMetaData.ACCESS_TYPE_PROPERTY);
		}
		oldAccessType = emd.getAccessType();
	}

	/**
	 * Complete inheritance related stuff in the model for a base entity.
	 * This will also make sure our table is set.
	 */
	protected void updateModelInheritanceBase(RClass cls, boolean metaDataChanged) {
		ensureModelNoSuperJoin(cls);

		AnnotationRegistry reg = mm.getAnnotationRegistry();
		EntityMetaData superEntity = emd.getSuperEntity();

		Inheritance inheritance = cls.getAnnotation(Inheritance.class);
		DiscriminatorColumn discColumn = cls.getAnnotation(
				DiscriminatorColumn.class, updateAddDiscriminatorColumn);
		if (updateAddDiscriminatorColumn) {
			registerForMetaDataUpdate();
		}
		
		if (inheritance == null) {
			// default to SINGLE_TABLE if there is a hierachy or the user has
			// specified a DiscriminatorColumn for a base class
			if (discColumn != null || emd.hasSubEntities()) {
				inheritance = reg.getDefaultProxy(Inheritance.class);
			}
		}

		emd.setInheritanceSpecified(inheritance != null);
		if (inheritance == null) {
			emd.setInheritance(EntityMetaData.INHERITANCE_NONE);
			emd.setDiscriminatorType(EntityMetaData.DISC_TYPE_STRING);
			emd.setDiscriminatorValue(null);
		} else {
			emd.setInheritance(getInheritanceTypeCode(inheritance.strategy()));
			emd.setDiscriminatorType(getDiscriminatorTypeCode(
					inheritance.discriminatorType()));
			String dv = inheritance.discriminatorValue();
			if (dv.length() == 0) {
				dv = defaultDiscriminatorValue;
			}
			emd.setDiscriminatorValue(dv);
		}
		
		updateModelEnsureOwnTable(cls);			

		switch (emd.getInheritance()) {
		case EntityMetaData.INHERITANCE_NONE:
			updateModelDiscriminatorColumn(null);
			if (discColumn != null) {
				addProblem("DiscriminatorColumn only allowed when inheritance " +
						"is used", discColumn);
			}
			break;
		case EntityMetaData.INHERITANCE_SINGLE_TABLE:
			if (discColumn == null) {
				discColumn = mm.getAnnotationRegistry().getDefaultProxy(
						DiscriminatorColumn.class);
			}
			updateModelDiscriminatorColumn(discColumn);
			break;
		case EntityMetaData.INHERITANCE_JOINED:
			updateModelDiscriminatorColumn(discColumn);
			break;
		case EntityMetaData.INHERITANCE_TABLE_PER_CLASS:
			updateModelDiscriminatorColumn(null);
			if (discColumn != null) {
				addProblem("DiscriminatorColumn is not allowed for " +
						"TABLE_PER_CLASS inheritance", discColumn);
			}
			break;
		}
	}

	protected int getDiscriminatorTypeCode(DiscriminatorType t) {
		switch (t) {
		default:
		case STRING:
			return EntityMetaData.DISC_TYPE_STRING;
		case CHAR:
			return EntityMetaData.DISC_TYPE_CHAR;
		case INTEGER:
			return EntityMetaData.DISC_TYPE_INTEGER;
		}
	}

	protected DiscriminatorType getDiscriminatorType(int code) {
		switch (code) {
		default:
		case EntityMetaData.DISC_TYPE_STRING:
			return DiscriminatorType.STRING;
		case EntityMetaData.DISC_TYPE_CHAR:
			return DiscriminatorType.CHAR;
		case EntityMetaData.DISC_TYPE_INTEGER:
			return DiscriminatorType.INTEGER;
		}
	}
	
	/**
	 * If we have a superJoin get rid of it. Add problems for 
	 * PrimaryKeyJoinColumn and so on.
	 */
	protected void ensureModelNoSuperJoin(RClass cls) {
		if (Utils.deleteJoinAndSrcCols(emd.getSuperJoin())) {
			emd.setSuperJoin(null);
		}
		PrimaryKeyJoinColumn pkjc = cls.getAnnotation(
				PrimaryKeyJoinColumn.class);
		if (pkjc != null) {
			addProblem("PrimaryKeyJoinColumn only valid for a JOINED subclass",
					pkjc);
		}
		PrimaryKeyJoinColumns pkjcs = cls.getAnnotation(
				PrimaryKeyJoinColumns.class);
		if (pkjc != null) {
			addProblem("PrimaryKeyJoinColumns only valid for a JOINED subclass",
					pkjcs);
		}
	}
	
	/**
	 * Complete inheritance related stuff in the model for an embeddable 
	 * class. This will also make sure our table is set.
	 */
	protected void updateModelInheritanceEmbeddable(RClass cls, 
			boolean metaDataChanged) {
		boolean embeddableSuperclass = emd.isEmbeddableSuperclass();
		String msg = embeddableSuperclass ? "EmbeddableSuperclass" : "Embeddable";
		
		EntityMetaData superEntity = emd.getSuperEntity();
		if (superEntity != null) {
			if (embeddableSuperclass) {
				if (!superEntity.isEmbeddableSuperclass()) {
					addProblem("Persistent superclass of an " +
							"EmbeddableSuperclass must also be an " +
							"EmbeddableSuperclass", 
							cls.getLocation());
				}
			} else {
				addProblem(msg + " class may not have a persistent superclass", 
						cls.getLocation());
			}
		}
		ensureModelNoSuperJoin(cls);
		Inheritance inheritance = cls.getAnnotation(Inheritance.class);
		if (inheritance != null) {
			addProblem("Inheritance not allowed for an " + msg + " class", 
					inheritance);
		}
		DiscriminatorColumn discColumn = cls.getAnnotation(
				DiscriminatorColumn.class);
		if (discColumn != null) {
			addProblem("DiscriminatorColumn not allowed for an " + msg + 
					" class", discColumn);
		}
		updateModelEnsureOwnTable(cls).setVirtual(true);			
		updateModelDiscriminatorColumn(null);
		emd.setInheritance(EntityMetaData.INHERITANCE_NONE);
		emd.setDiscriminatorValue(null);
	}
	
	protected void registerForMetaDataUpdate() {
		mm.registerForMetaDataUpdate(this);
	}
	
	/**
	 * Make sure our entity has its own table creating one if needed.
	 */
	protected OrmTable updateModelEnsureOwnTable(RClass cls) {
		Table table = getTableAnnotation(cls);
		OrmTable ormTable = emd.getTable();
		if (ormTable == null || ormTable.getParentElement() != emd) {
			emd.setTable(ormTable = mm.getFactory().createOrmTable());
            ormTable.setParentElement(emd);
			ormTable.eAdapters().add(this);	
		}
		ormTable.setName(table.name());
		ormTable.setCatalog(table.catalog());
		ormTable.setSchema(table.schema());
		ormTable.setComment(emd.getSchemaName());		
		return ormTable;
	}

	/**
	 * If discColumn is null then get rid of any discriminator column. 
	 * Otherwise create and set and or update one.
	 */
	protected void updateModelDiscriminatorColumn(
			DiscriminatorColumn discColumn) {
		OrmColumn dc = emd.getDiscriminatorColumn();
		if (discColumn == null) {
			if (dc != null) {
				dc.delete();
				emd.setDiscriminatorColumn(null);
			}
		} else {
			if (dc == null) {
				dc = mm.getFactory().createOrmColumn();
				dc.eAdapters().add(this);							
				emd.setDiscriminatorColumn(dc);
			}
			dc.setRelativePositionInTable(mm.getColumnPositionDiscriminator());
			String name = discColumn.name();
			if (name.length() == 0) {
				name = "TYPE";
			}
			dc.setName(name);
			switch (emd.getDiscriminatorType()) {
			default:
			case EntityMetaData.DISC_TYPE_INTEGER:		
				dc.setJdbcType(Types.INTEGER);	
				break;
			case EntityMetaData.DISC_TYPE_CHAR:
				dc.setJdbcType(Types.CHAR);	
				break;
			case EntityMetaData.DISC_TYPE_STRING:
				dc.setJdbcType(Types.VARCHAR);
				dc.setLength(discColumn.length());
				break;
			}
			dc.setInsertable(true);
			dc.setUpdatable(true);
			dc.setNullable(false);
			dc.setTable(emd.getTable());
			String cd = discColumn.columnDefinition();
			if (cd.length() == 0) {
				dc.setColumnDefinitionSpecified(false);
				cd = mm.getColumnDefinition(dc);
			} else {
				dc.setColumnDefinitionSpecified(true);
			}
			dc.setColumnDefinition(cd);
			dc.setOriginalColumnDefinition(cd);
			mm.updateDatabaseType(dc);
			dc.setComment("Discriminator for " + emd.getSchemaName());
		}
	}
	
	/**
	 * Complete inheritance related stuff in the model for a middle or leaf
	 * class. This will also make sure our table is set.
	 */
	protected void updateModelInheritanceNonBase(RClass cls, 
			boolean metaDataChanged) {
		DiscriminatorColumn discColumn = cls.getAnnotation(
				DiscriminatorColumn.class);		
		if (discColumn != null) {
			addProblem("DiscriminatorColumn not valid for a non-base class",
					discColumn);
		}
		updateModelDiscriminatorColumn(null);

		Inheritance inheritance = cls.getAnnotation(Inheritance.class);
		if (inheritance == null) {
			emd.setDiscriminatorValue(defaultDiscriminatorValue);
		} else {
			String dv = inheritance.discriminatorValue();
			if (dv.length() == 0) {
				dv = defaultDiscriminatorValue;
			}
			emd.setDiscriminatorValue(dv);
			AnnotationEx inheritanceEx = ((AnnotationEx)inheritance);
			if (inheritanceEx.hasValue("strategy")) {
				addProblem("Inheritance.strategy may only be set for the base " +
						"class of a heirachy", 
						inheritanceEx.getLocation("strategy"));				
			}
			if (inheritanceEx.hasValue("discriminatorType")) {
				addProblem("Inheritance.discriminatorType may only be set for " +
						"the base class of a heirachy", 
						inheritanceEx.getLocation("discriminatorType"));				
			}
		}
		EntityMetaData superEntity = emd.getSuperEntity();
		emd.setInheritance(superEntity.getInheritance());

		if (superEntity.getInheritance() != EntityMetaData.INHERITANCE_JOINED) {
			Join sj = emd.getSuperJoin();
			if (sj != null) {
				sj.delete();
				emd.setSuperJoin(null);
			};
		}
		
		OrmTable table;
		switch (superEntity.getInheritance()) {
		default:
		case EntityMetaData.INHERITANCE_SINGLE_TABLE:
			table = superEntity.getTable();
			if (emd.getTable() != table) {
				emd.setTable(table);
			}
			Table tableAnn = cls.getAnnotation(Table.class);
			if (tableAnn != null) {
				addProblem("Table not allowed for SINGLE_TABLE inheritance " +
						"subclass", tableAnn);
			}
			break;
		case EntityMetaData.INHERITANCE_JOINED:
			updateModelInheritanceJoinedSubclass(cls, metaDataChanged);
			break;
		case EntityMetaData.INHERITANCE_TABLE_PER_CLASS:
			updateModelEnsureOwnTable(cls);
			break;
		}
	}
	
	/**
	 * Complete inheritance stuff for a JOINED subclass.
	 */
	protected void updateModelInheritanceJoinedSubclass(RClass cls, 
			boolean metaDataChanged) {
		updateModelEnsureOwnTable(cls);
		AnnotationEx jcs = (AnnotationEx)cls.getAnnotation(
				PrimaryKeyJoinColumns.class);
		Join join = joinIO.updateModelJoin(emd.getSuperJoin(), 
				(AnnotationEx)cls.getAnnotation(PrimaryKeyJoinColumn.class),
				jcs == null ? null : (Object[])jcs.get("value"),
				jcs == null ? null : jcs.getLocation(null),
				metaDataChanged, emd.getTable(), emd.getSuperEntity().getTable(),
				mm.getJoinedInheritanceCNS(emd), 
				"Superclass " + emd.getSuperEntity().getSchemaName(),
				false, mm.getColumnPositionPrimaryKey());
		if (emd.getSuperJoin() != join) {
			emd.setSuperJoin(join);
		}
		if (join != null) {
			for (Iterator i = join.getPairList().iterator(); i.hasNext(); ) {
				JoinPair p = (JoinPair)i.next();
				OrmColumn src = p.getSrc();
				if (!src.isPrimaryKey()) {
					src.getTable().getPrimaryKeyList().add(src);
				}
			}
		}
		AnnotationEx ann = (AnnotationEx)cls.getAnnotation(JoinColumns.class);
		if (ann != null) {
			addProblem("JoinColumns not allowed here" + 
					(!cls.isAnnotationPresent(PrimaryKeyJoinColumns.class) 
							? " (have you tried PrimaryKeyJoinColumns?)" 
							: ""), 
					ann);
		}
		ann = (AnnotationEx)cls.getAnnotation(JoinColumn.class);
		if (ann != null) {
			addProblem("JoinColumn not allowed here" + 
					(!cls.isAnnotationPresent(PrimaryKeyJoinColumn.class) 
							? " (have you tried PrimaryKeyJoinColumn?)" 
							: ""), 
					ann);
		}
	}
	
	/**
	 * Find all possible attributes and make sure we have an
	 * AttributeIO instance for each in the todo list. For existing attributes
	 * the current AttributeIO instance is checked to make sure it is still
	 * the correct class. If it is not then it is replaced with a new one.
	 * This also delete's AttributeMetaData for attributes that no longer 
	 * exist.
	 */
	protected void fillTodoWithAttributeInfos(RClass cls) {
		Set<AttributeMetaData> found = new HashSet();
		fillTodoWithInheritedAttributes(cls, found);
		fillTodoWithAttributeInfos(cls.getDeclaredFields(), true, found);
		fillTodoWithAttributeInfos(cls.getDeclaredMethods(), false, found);
		
		// get rid of attributes and associated IOs that no longer exist
		List<AttributeMetaData> toDelete = null;
		for (Iterator i = emd.getAttributeList().iterator(); i.hasNext(); ) {
			Object o = i.next();
			if (!found.contains(o)) {
				if (toDelete == null) {
					toDelete = new ArrayList();
				}
				toDelete.add((AttributeMetaData)o);
			}
		}
		if (toDelete != null) {
			for (AttributeMetaData amd : toDelete) {
				AttributeIO aio = AttributeIO.get(amd);
				if (aio != null) {
					aio.dispose();
				}
				if (amd.isNonPersistent()) {
					removeNonPersistentAttribute(amd);
				}
				amd.delete();				
			}
		}
	}
	
	/**
	 * Create and add AttributeInfo's for attributes inherited from our
	 * superclass but stored in our table (e.g. from an EmbeddableSuperclass). 
	 */
	protected void fillTodoWithInheritedAttributes(RClass cls,
			Set<AttributeMetaData> found) {
		EntityMetaData superEntity = emd.getSuperEntity();
		if (superEntity == null || !superEntity.isEmbeddableSuperclass()) {
			return;
		}
		AttributeOverrideMap map = new AttributeOverrideMap(cls, this);
		for (Iterator i = superEntity.getAttributeList().iterator(); 
				i.hasNext(); ) {
			AttributeMetaData overrideOf = (AttributeMetaData)i.next();
			if (!(overrideOf instanceof BasicAttribute)) {
				// problem will have already been created by superEntity so
				// just ignore non-basic attributes
				continue;
			}
			AttributeMetaData amd = emd.findAttributeMetaData(
					overrideOf.getName());
			OverrideAttributeIO aio = null;
			if (amd != null) {
				found.add(amd);
				aio = (OverrideAttributeIO)AttributeIO.get(amd);
			}
			if (aio == null) {
				aio = new OverrideAttributeIO(this, null);
			}
			String name = overrideOf.getName();
			AttributeOverride attrOverride = map.get(name);
			if (attrOverride != null) {
				map.remove(name);
			}
			todo.add(new AttributeInfoOverride(overrideOf, attrOverride, aio));
		}
		// add problems for all unused AttributeOverride's
		for (AttributeOverride override : map.values()) {
			addProblem("AttributeOverride does not match any inherited " +
					"attributes: " + override.name(), override);
		}
	}
	
	protected void fillTodoWithAttributeInfos(RAnnotatedElement[] all, 
			boolean allContainsFields, Set<AttributeMetaData> found) {
		boolean accessField = emd.getAccessType() 
			== EntityMetaData.ACCESS_TYPE_FIELD;
		if (accessField == allContainsFields) {
			// access type matches
			for (RAnnotatedElement attribute : all) {
				if (mm.checkValidAttribute(attribute) == null) {
					todo.add(createAttributeInfo(attribute, found));
				} else {
					checkNoPersistentAnnotations(attribute);					
				}
			}
		} else {
			// access type does not match so none of these can be persistent
			for (RAnnotatedElement attribute : all) {
				checkNoPersistentAnnotations(attribute);
			}			
		}
	}
		
	/**
	 * Create an AttributeInfo for the element. This will never return null.
	 * The AttributeIO instance chosen is responsible for adding problems
	 * for invalid types and so on.
	 */
	protected AttributeInfo createAttributeInfo(RAnnotatedElement attribute,
			Set<AttributeMetaData> found) {
		Class aioCls = selectAttributeIOClass(attribute);
		String attributeName = Utils.getAttributeName(attribute);
		AttributeIO aio = null;
		AttributeMetaData amd = emd.findAttributeMetaData(attributeName);
		if (amd == null) {
			amd = getNonPeristentAttribute(attributeName);
		}
		if (amd != null) {
			aio = AttributeIO.get(amd);
			if (aio != null) {
				if (aio.getClass() != aioCls) {
					aio.dispose();
					amd.delete();
					aio = createAttributeIO(aioCls);
				} else {
					found.add(amd);					
				}
			} else { 
				// it should not be possible to get an attribute with no IO
				amd.delete();
				aio = createAttributeIO(aioCls);
			}
		} else {	// attribute is not currently in model
			aio = createAttributeIO(aioCls);			
		}
		return new AttributeInfoNormal(attribute, aio);
	}

	/**
	 * Select the correct AttributeIO class to manage the persistence of
	 * attribute. This must always return non-null.
	 */
	protected Class selectAttributeIOClass(RAnnotatedElement attribute) {
		String typeName = attribute.getTypeName();
		TypeMetaData tmd = findTypeByClassName(typeName);
		if (tmd instanceof EntityMetaData) {
			if (((EntityMetaData)tmd).isEntity()) {
				return ReferenceAttributeIO.class;				
			} else {			
				return EmbeddedAttributeIO.class;
			}
		} else if (tmd instanceof CollectionTypeMetaData) {
			return CollectionAttributeIO.class;
		} else {
			return BasicAttributeIO.class;
		}
	}
	
	/**
	 * Add a problem if attribute has any registered annotations other than
	 * Transient. 
	 */
	protected void checkNoPersistentAnnotations(RAnnotatedElement attribute) {
		String type = attribute.isField() ? "Field " : "Method ";
		Annotation[] all = attribute.getAnnotations();
		for (Annotation ann : all) {
			if (ann.annotationType() != Transient.class) {
				StringBuffer s = new StringBuffer();
				boolean field = attribute.isField();
				boolean badAccess;
				if (field) {
					s.append("Field '");
					badAccess = emd.getAccessType() 
						!= EntityMetaData.ACCESS_TYPE_FIELD;
				} else {
					s.append("Property '");
					badAccess = emd.getAccessType() 
						!= EntityMetaData.ACCESS_TYPE_PROPERTY;
				}
				s.append(attribute.getName());
				s.append("' is not persistent: ");
				if (badAccess) {
					s.append(emd.getShortName() + " is using ");
					if (emd.getAccessType() == EntityMetaData.ACCESS_TYPE_FIELD) {
						s.append("FIELD");
					} else {
						s.append("PROPERTY");
					}
					s.append(" access");
				} else {
					s.append(mm.checkValidAttribute(attribute));
				}
				addProblem(s.toString(), ann);
				break;
			}
		}
	}

	/**
	 * This method is invoked after all EntityIO updateModelFromMetaData
	 * processing has been completed. If failed is true then this EntityIO
	 * did not complete processing. This is a good time to log problems and 
	 * so on.
	 */
	public void updateModelFromMetaDataPost(RClass cls, boolean failed) 
			throws Exception {	
		todo = null;
		checkEntity(cls.getLocation());		
	}	
	
	/**
	 * Look for problems with the entity that are best discovered after all
	 * attributes have been created (e.g. no primary key). These are reported
	 * as problems.
	 */
	protected void checkEntity(Map location) {
		checkPrimaryKey(location);
		checkVersion(location);
	}
	
	/**
	 * Make sure the entity has a valid primary key.
	 */
	protected void checkPrimaryKey(Map location) {
		if (emd.isBaseEntity()) {
			if (emd.getPrimaryKeyList().isEmpty()) {
				addProblem("Entity has no primary key (Id) attributes", 
						location);
			}
		}
	}
	
	/**
	 * Warn if the entity has multiple version attributes.
	 */
	protected void checkVersion(Map location) {
		if (emd.getVersionList().size() > 1) {
			addProblem("Entity has more than one Version attribute, " +
					"this is not portable", 
					location, IMarker.SEVERITY_WARNING);
		}
	}
	
	protected Table getTableAnnotation(RClass cls) {
		Table t = cls.getAnnotation(Table.class, true);
		((AnnotationEx)t).setDefault("name", 
				getDefaultTableName(cls.getSimpleName()));
		return t;
	}
	
	/**
	 * Return true if the attribute is considered persistent even if it has
	 * no annotations and so on.
	 */
	public boolean isPersistentByDefault(RAnnotatedElement attribute) {
		if (mm.checkValidAttribute(attribute) != null) {
			return false;
		}
		if (attribute.isField()) {
			return emd.getAccessType() == EntityMetaData.ACCESS_TYPE_FIELD;
		} else {
			return emd.getAccessType() == EntityMetaData.ACCESS_TYPE_PROPERTY;
		}
	}	

	/**
	 * Convert a type name (e.g. String or java.lang.String or an Entity class)
	 * into TypeMetaData or null if none.
	 */
	public TypeMetaData findTypeByClassName(String typeName) {
		EntityModel model = mm.getEntityModel();
		TypeMetaData tmd = model.findTypeByClassName(typeName);
		if (tmd == null) {
			for (int i = 0; i < DEFAULT_PACKAGES.length; i++) {
				tmd = model.findTypeByClassName(DEFAULT_PACKAGES[i] + typeName);
				if (tmd != null) {
					break;
				}
			}
		}
		return tmd;
	}	
	
	/**
	 * Create an instanceof cls which must be an AttributeIO subclass. Logs
	 * an error and returns false if anything goes wrong.
	 */
	protected AttributeIO createAttributeIO(Class cls) {
		try {
			Constructor con = cls
					.getConstructor(new Class[]{EntityIO.class});
			return (AttributeIO) con.newInstance(this);
		} catch (Exception e) {
			OrmPlugin.log(e);
			return null;
		}
	}
	
	protected BasicAttributeIO createBasicAttributeIO() {
		return new BasicAttributeIO(this);
	}
	
	protected ReferenceAttributeIO createReferenceAttributeIO() {
		return new ReferenceAttributeIO(this);
	}
	
	/**
	 * Update the annotations and/or XML from the entity. 
	 */
	public void updateMetaDataFromModel(RClass cls) {
		
		Class mainType;
		switch (emd.getEntityType()) {
		case 0:	// NOT PERSISTENT
			Utils.removeAnnotation(cls, EmbeddableSuperclass.class);
			Utils.removeAnnotation(cls, Embeddable.class);
			Utils.removeAnnotation(cls, Entity.class);
			return;
		case EntityMetaData.TYPE_EMBEDDABLE_SUPERCLASS:
			mainType = EmbeddableSuperclass.class;
			Utils.removeAnnotation(cls, Embeddable.class);
			Utils.removeAnnotation(cls, Entity.class);
			break;
		case EntityMetaData.TYPE_EMBEDDABLE:
			mainType = Embeddable.class;
			Utils.removeAnnotation(cls, EmbeddableSuperclass.class);
			Utils.removeAnnotation(cls, Entity.class);
			break;
		default:
			mainType = Entity.class;
			Utils.removeAnnotation(cls, Embeddable.class);
			Utils.removeAnnotation(cls, EmbeddableSuperclass.class);
			break;
		};
		
		AnnotationEx main = (AnnotationEx)cls.getAnnotation(mainType, true);
		if (main instanceof Entity) {
			main.setDefault("name", cls.getSimpleName());
			main.set("name", emd.getSchemaName());			
		}
		
		AccessType access;
		switch (emd.getAccessType()) {
			case EntityMetaData.ACCESS_TYPE_FIELD:
				access = AccessType.FIELD;
				break;
			case EntityMetaData.ACCESS_TYPE_PROPERTY:
				access = AccessType.PROPERTY;
				break;
			default:
				access = null;
		}		
		boolean accessChanged = oldAccessType != emd.getAccessType();
		if (access != null) {
			main.set("access", access);
		}
		main.setMarker(true);
				
		AnnotationEx idc = (AnnotationEx)cls.getAnnotation(IdClass.class, true);
		idc.set("value", emd.getIdClass());
		
		AnnotationEx table = (AnnotationEx)getTableAnnotation(cls);
		table.set("name", emd.getTable().getName(), true);
		table.set("catalog", emd.getTable().getCatalog(), true);
		table.set("schema", emd.getTable().getSchema(), true);

		switch (emd.getEntityType()) {
		default:
		case EntityMetaData.TYPE_ENTITY:
			if (emd.isBaseEntity()) {
				updateMetaDataInheritanceTop(cls);
			} else {
				updateMetaDataInheritanceNonBase(cls);				
			}
			break;
		case EntityMetaData.TYPE_EMBEDDABLE_SUPERCLASS:
			updateMetaDataInheritanceTop(cls);
			break;
		case EntityMetaData.TYPE_EMBEDDABLE:
			Utils.removeAnnotation(cls, DiscriminatorColumn.class);
			Utils.removeAnnotation(cls, Inheritance.class);
			Utils.removeAnnotation(cls, PrimaryKeyJoinColumn.class);
			Utils.removeAnnotation(cls, PrimaryKeyJoinColumns.class);
			break;
		}
		
		updateMetaDataInheritedAttributes(cls);
		updateMetaDataAttributes(cls, accessChanged);
		updateMetaDataNonPersistentAttributes(cls, accessChanged);
	}

	/**
	 * Update meta data for inherited persistent attributes e.g. from an
	 * EmbeddableSuperclass.
	 */
	protected void updateMetaDataInheritedAttributes(RClass cls) {
		List<AttributeIO> toDelete = null;
		AttributeOverrideMap map = new AttributeOverrideMap(cls, null);
		AttributeOverride extra = null;
		for (Iterator i = emd.getAttributeList().iterator(); i.hasNext(); ) {
			AttributeMetaData amd = (AttributeMetaData)i.next();
			AttributeIO o = AttributeIO.get(amd);
			if (!(o instanceof OverrideAttributeIO)) {
				continue;
			}
			OverrideAttributeIO aio = (OverrideAttributeIO)o;
			String name = amd.getName();
			AttributeOverride override = map.get(name);
			if (override == null) {
				if (extra == null) {
					extra = map.append();
				}
				override = extra;
			}
			if (!aio.updateMetaDataFromModel(override)) {
				if (toDelete == null) {
					toDelete = new ArrayList();
				}
				toDelete.add(aio);
			} else {
				if (override == extra && ((AnnotationEx)extra).hasValue("column")) {
					((AnnotationEx)extra).set("name", name);
					extra = null;
				}
			}
		}	
		map.cleanupAfterMetaDataUpdates();
	}

	/**
	 * Update meta data for normal persistent attributes. 
	 */
	protected void updateMetaDataAttributes(RClass cls, boolean accessChanged) {
		List<AttributeIO> toDelete = null;
		for (Iterator i = emd.getAttributeList().iterator(); i.hasNext(); ) {
			AttributeMetaData amd = (AttributeMetaData)i.next();
			AttributeIO aio = AttributeIO.get(amd);
			if (aio != null && !(aio instanceof OverrideAttributeIO)) {
				if (!aio.updateMetaDataFromModel(cls, accessChanged)) {
					if (toDelete == null) {
						toDelete = new ArrayList();
					}
					toDelete.add(aio);
				}
			}
		}
		deleteModelAttributes(toDelete);
	}

	/**
	 * Delete the attributes from the model. NOP if toDelete is null.
	 */
	protected void deleteModelAttributes(List<AttributeIO> toDelete) {
		if (toDelete != null) {
			for (AttributeIO aio : toDelete) {
				AttributeMetaData amd = aio.getAttributeMetaData();
				emd.getAttributeList().remove(amd);
				amd.delete();
				aio.dispose();
			}
		}
	}

	/**
	 * Update meta data for non-persistent attributes. 
	 */
	protected void updateMetaDataNonPersistentAttributes(RClass cls, 
			boolean accessChanged) {
		List<AttributeIO> toDelete = null;
		for (Map.Entry<String, AttributeMetaData> e : npAttributeMap.entrySet()) {
			AttributeIO aio = AttributeIO.get(e.getValue());
			if (aio != null) {
				if (!aio.updateMetaDataFromModel(cls, accessChanged)) {
					if (toDelete == null) {
						toDelete = new ArrayList();
					}
					toDelete.add(aio);					
				}
			}				
		}
		if (toDelete != null) {
			for (AttributeIO aio : toDelete) {
				AttributeMetaData amd = aio.getAttributeMetaData();
				npAttributeMap.remove(amd.getName());
				amd.delete();
				aio.dispose();
			}
		}
	}	
	
	/**
	 * Update our inheritance meta data for a top level class (normal base 
     * entity or embeddable superclass).
	 */
	protected void updateMetaDataInheritanceTop(RClass cls) {
		Utils.removeAnnotation(cls, PrimaryKeyJoinColumn.class);
		Utils.removeAnnotation(cls, PrimaryKeyJoinColumns.class);
		AnnotationEx inheritance = null;
		if (emd.getInheritance() != EntityMetaData.INHERITANCE_NONE) {
			inheritance = (AnnotationEx)cls.getAnnotation(Inheritance.class, 
					true);
			inheritance.setDefault("discriminatorValue", 
					defaultDiscriminatorValue);
			inheritance.set("discriminatorValue", emd.getDiscriminatorValue());
		}
		updateMetaDataDiscriminatorColumn(cls, inheritance);
		switch (emd.getInheritance()) {
		case EntityMetaData.INHERITANCE_NONE:
			Utils.removeAnnotation(cls, Inheritance.class);
			break;
		case EntityMetaData.INHERITANCE_SINGLE_TABLE:
			inheritance.set("strategy", InheritanceType.SINGLE_TABLE);
			break;
		case EntityMetaData.INHERITANCE_JOINED:
			inheritance.set("strategy", InheritanceType.JOINED);
			break;
		case EntityMetaData.INHERITANCE_TABLE_PER_CLASS:
			inheritance.set("strategy", InheritanceType.TABLE_PER_CLASS);
			break;
		}
	}
	
	/**
	 * Update meta data for our discriminator column (if any). 
	 */
	protected void updateMetaDataDiscriminatorColumn(RClass cls, 
			AnnotationEx inheritance) {
		OrmColumn dc = emd.getDiscriminatorColumn();
		if (dc == null) {
			Utils.removeAnnotation(cls, DiscriminatorColumn.class);			
		} else {
			switch (dc.getJdbcType()) {
			case Types.CHAR:
				inheritance.set("discriminatorType", DiscriminatorType.CHAR);
				break;
			case Types.INTEGER:
				inheritance.set("discriminatorType", DiscriminatorType.INTEGER);
				break;
			case Types.VARCHAR:
			default:
				inheritance.set("discriminatorType", DiscriminatorType.STRING);
				break;
			}
			AnnotationEx discColumn = (AnnotationEx)cls.getAnnotation(
					DiscriminatorColumn.class, true);
			discColumn.setMarker(true);
			discColumn.setDefault("name", "TYPE");
			discColumn.set("name", dc.getName());
			discColumn.set("length", dc.getLength());
			
			// if the columnDefinition has changed then apply the change to the
			// annotation - this may remove it if it now matches the default
			boolean colDefChanged = !dc.getColumnDefinition().equals(
					dc.getOriginalColumnDefinition());
			String stdColDef =  mm.getColumnDefinition(dc);
			if (colDefChanged) {
				discColumn.setDefault("columnDefinition", stdColDef);			
				discColumn.set("columnDefinition", dc.getColumnDefinition(), true);
			}
			
			// if the annotation now has no columnDefinition value then put
			// the default value into the model
			if (!discColumn.hasValue("columnDefinition")) {
				dc.setColumnDefinition(stdColDef);
			}
			
			// reset the originalColumnDefinition attribute to detect more changes
			dc.setOriginalColumnDefinition(dc.getColumnDefinition());		

			mm.updateDatabaseType(dc);
		}
	}

	/**
	 * Update our inheritance meta data for a middle or leaf class.
	 */
	protected void updateMetaDataInheritanceNonBase(RClass cls) {
		Utils.removeAnnotation(cls, DiscriminatorColumn.class);
		String dv = emd.getDiscriminatorValue();
		if (dv == null) {
			Utils.removeAnnotation(cls, Inheritance.class);
		} else {
			AnnotationEx inheritance = (AnnotationEx)cls.getAnnotation(
					Inheritance.class, true);
			inheritance.setDefault("discriminatorValue", 
					defaultDiscriminatorValue);
			inheritance.set("discriminatorValue", dv);
		}
		EntityMetaData superEntity = emd.getSuperEntity();
		if (superEntity != null && superEntity.getInheritance() 
				== EntityMetaData.INHERITANCE_JOINED) {
			updateMetaDataInheritanceJoined(cls);
		} else {
			Utils.removeAnnotation(cls, PrimaryKeyJoinColumn.class);
			Utils.removeAnnotation(cls, PrimaryKeyJoinColumns.class);			
		}
	}

	/**
	 * Update our inheritance meta data for a JOIND subclass. This just has
	 * to put in the PrimaryKeyJoinColumn/s annotation(s).
	 */
	protected void updateMetaDataInheritanceJoined(RClass cls) {
		Join join = emd.getSuperJoin();
		if (join == null) {
			// model not complete due to errors so do nothing
			return;
		}
		if (join.getPairList().size() == 1) {
			Utils.removeAnnotation(cls, PrimaryKeyJoinColumns.class);	
			AnnotationEx jc = (AnnotationEx)cls.getAnnotation(
						PrimaryKeyJoinColumn.class, true);
			joinIO.updateMetaDataJoinColumn(jc, join, (JoinPair)join.getPairList().get(0),
					mm.getJoinedInheritanceCNS(emd), false);
		} else {
			Utils.removeAnnotation(cls, PrimaryKeyJoinColumn.class);	
			List pairList = join.getPairList();
			AnnotationEx joinColumns = (AnnotationEx)cls.getAnnotation(
					PrimaryKeyJoinColumns.class, true);
			int n = pairList.size();
			if (joinColumns.setArraySize("value", n)) {
				PrimaryKeyJoinColumn[] a = 
					((PrimaryKeyJoinColumns)joinColumns).value();
				for (int i = 0; i < n; i++) {
					joinIO.updateMetaDataJoinColumn((AnnotationEx)a[i], join, 
							(JoinPair)pairList.get(i), 
							mm.getJoinedInheritanceCNS(emd), false);
				}
				// get rid of the annotations if they are all default
				if (n > 1) {
					int i = 0;
					for (; i < n; i++) {
						AnnotationEx jc = (AnnotationEx)a[i];
						int c = jc.getValueCount();
						if (jc.hasValue("referencedColumnName")) {
							--c;
						}
						if (c > 0) {
							break;
						}
					}	
					if (i == n) {
						joinColumns.delete();					
					}
				}
			}
		}		
	}
	
	protected InheritanceType getInheritanceType(int code) {
		switch (code) {
		default:
		case EntityMetaData.INHERITANCE_SINGLE_TABLE:
			return InheritanceType.SINGLE_TABLE;
		case EntityMetaData.INHERITANCE_JOINED:
			return InheritanceType.JOINED;
		case EntityMetaData.INHERITANCE_TABLE_PER_CLASS:
			return InheritanceType.TABLE_PER_CLASS;
		}
	}
	
	protected int getInheritanceTypeCode(InheritanceType t) {
		switch (t) {
		case SINGLE_TABLE:		return EntityMetaData.INHERITANCE_SINGLE_TABLE;
		case JOINED:			return EntityMetaData.INHERITANCE_JOINED;
		case TABLE_PER_CLASS:	return EntityMetaData.INHERITANCE_TABLE_PER_CLASS;
		}
		return EntityMetaData.INHERITANCE_NONE;
	}

	/**
	 * This is fired when something we are interested in in the model changes.
	 */
	public void notifyChanged(Notification no) {
		mm.notifyChanged(no, this);
	}

	public Notifier getTarget() {
		return null;
	}

	public void setTarget(Notifier newTarget) {		
	}

	public boolean isAdapterForType(Object type) {
		return type instanceof Class 
			&& EntityMetaData.class.isAssignableFrom((Class)type)
			&& OrmTable.class.isAssignableFrom((Class)type)
			&& OrmColumn.class.isAssignableFrom((Class)type);
	}

	public EntityMetaData getEntityMetaData() {
		return emd;
	}
	
	public GenericEntityModelManager getModelManager() {
		return mm;
	}

	/**
	 * This is fired when the source code changes.
	 */
    public void elementChanged(final ElementChangedEvent event) {
    	if (mm.isIgnoreEvents()) {
    		return;
    	}
        IJavaElementDelta delta = (IJavaElementDelta)event.getSource();
        if (delta.getElement().getAncestor(IJavaElement.COMPILATION_UNIT).equals(
        		type.getParent())){
            PlatformUI.getWorkbench().getDisplay().asyncExec(new Runnable() {
                public void run() {
            		mm.updateModelFromMetaData(EntityIO.this, event);
                }});          
        }
    }

    /**
     * If this entity has an attribute with an @Id annotation with any values
     * set (e.g. generate or generator) then this is it. This property is used
     * to make sure that any changes to the generator on the EntityMetaData
     * are written out by the correct attribute. There may be more than one
     * attribute with @Id but only one is allowed to have values.
     */
	public BasicAttribute getIdGeneratorAttribute() {
		return idGeneratorAttribute;
	}

	public void setIdGeneratorAttribute(BasicAttribute idGeneratorAttribute) {
		this.idGeneratorAttribute = idGeneratorAttribute;
	}

	/**
	 * Add a problem to the resource associated with our entity.
	 */
	public void addProblem(String msg, Map location) {
		addProblem(msg, location, IMarker.SEVERITY_ERROR);
	}
	
	/**
	 * Add a problem to the resource associated with our entity.
	 */
	public void addProblem(String msg, Map location, int severity) {
        IMarker marker;
        try {
            marker = type.getResource().createMarker(IMarker.PROBLEM);
            marker.setAttribute(IMarker.MESSAGE, msg);
            marker.setAttribute(IMarker.SEVERITY, severity);
            if (location != null) {
	            for (Iterator i = location.entrySet().iterator(); i.hasNext(); ) {
	            	Map.Entry e = (Map.Entry)i.next();
	            	marker.setAttribute((String)e.getKey(), e.getValue());
	            }
            }
            marker.setAttribute(IMarker.TRANSIENT, true);
            markers.add(marker);
        } catch (CoreException e) {
        	OrmPlugin.log(e);
        }		
	}

	/**
	 * Add a problem to the resource associated with our entity.
	 */
	public void addProblem(String msg, Annotation annotation, String valueName) {
		addProblem(msg, (AnnotationEx)annotation, valueName);
	}
	
	/**
	 * Add a problem to the resource associated with our entity.
	 */
	public void addProblem(String msg, Annotation annotation) {
		addProblem(msg, (AnnotationEx)annotation, null);
	}
	
	/**
	 * Add a problem to the resource associated with our entity.
	 */
	public void addProblem(String msg, AnnotationEx annotation, String valueName) {
		addProblem(msg, annotation.getLocation(valueName));
	}
	
	/**
	 * Add a problem to the resource associated with our entity.
	 */
	public void addProblem(String msg, AnnotationEx annotation) {
		addProblem(msg, annotation, null);
	}
	
	/**
	 * Order by fully qualified entity name.
	 */
	public int compareTo(Object o) {
		EntityIO e = (EntityIO)o;
		return emd.getClassName().compareTo(e.emd.getClassName());
	}
	
	public IType getType() {
		return type;
	}
	
	public String toString() {
		if (emd == null) {
			return "EntityIO EntityMetaData == null";
		}
		return "EntityIO " + emd.getClassName();
	}
	
	/**
	 * Return the default table name for an entity name.
	 */
	protected String getDefaultTableName(String entityName) {
        return mm.getDefaultTableName(entityName);
	}
	
	/**
	 * Return the set of EntityIO's that should have their models updated when
	 * our model is updated.
	 */
	public Set<EntityIO> getDependOnUs() {
		return dependOnUs;
	}
	
	/**
	 * Make us depend on other. Our model will be updated whenever its model
	 * is updated. NOP if other is null.
	 */
	public void addDependencyOn(EntityMetaData other) {
		if (other == null) {
			return;
		}
		EntityIO otherIO = EntityIO.get(other);
		otherIO.dependOnUs.add(this);
		weDependOn.add(otherIO);
	}

	public Object getAdapter(Object adaptableObject, Class adapterType) {
		return this;
	}

	public Class[] getAdapterList() {
		return new Class[]{EntityIO.class};
	}
	
	/**
	 * How far along are we with our model update?
	 */
	public int getModelUpdateStatus() {
		return modelUpdateStatus;
	}
	
	public AttributeMetaData getNonPeristentAttribute(String name) {
		return npAttributeMap.get(name);
	}
	
	public void addNonPersistentAttribute(AttributeMetaData amd) {
		npAttributeMap.put(amd.getName(), amd);
	}

	public void removeNonPersistentAttribute(AttributeMetaData amd) {
		npAttributeMap.remove(amd.getName());
	}
	
	/**
	 * If the attribute has been renamed then return its new name. Otherwise
	 * return null.
	 * 
	 * @see #renameAttributes(ElementChangedEvent)
	 */
	public String getNewAttributeName(String name) {
		return renamedAttributeMap == null ? null : renamedAttributeMap.get(name);
	}
	
	/**
	 * Create a new OrmColumn and add our entityIO to its adapter list so that 
	 * it receives model events.
	 */
	public OrmColumn createOrmColumn() {
		OrmColumn c = mm.getFactory().createOrmColumn();
		c.eAdapters().add(this);			
		return c;
	}
	
	/**
	 * If c is null then create a new OrmColumn and add our entityIO to its
	 * adapter list so that it receives model events. Return c or the newly
	 * created OrmColumn. 
	 */
	protected OrmColumn ensureOrmColumn(OrmColumn c) {
		if (c == null) {
			c = createOrmColumn();
		}
		return c;
	}
	
	/**
	 * If t is null then create a new OrmTable and add our entityIO to its
	 * adapter list so that it receives model events. Return t or the newly
	 * created OrmTable. 
	 */
	protected OrmTable ensureTable(OrmTable t) {
		if (t == null) {
			t = mm.getFactory().createOrmTable();
			t.eAdapters().add(this);			
		}
		return t;
	}
	
	public AstState getAstState() {
		return astState;
	}

	public void setAstState(AstState astState) {
		this.astState = astState;
	}

	public boolean isUseDiscriminatorColumnEnabled() {
		if (!emd.isBaseEntity()) {
			return false;
		}
		switch (emd.getInheritance()) {
		case EntityMetaData.INHERITANCE_NONE:
		case EntityMetaData.INHERITANCE_JOINED:
			return true;
		case EntityMetaData.INHERITANCE_SINGLE_TABLE:
			return !emd.isInheritanceSpecified()
				|| emd.getSubEntityList().isEmpty();
		};
		return false;
	}

	public void setUseDiscriminatorColumn(boolean on) {
		if (on) {
			if (emd.getDiscriminatorColumn() == null) {
				try {
					updateAddDiscriminatorColumn = true;
					mm.updateModelFromMetaData(this);
				} finally {
					updateAddDiscriminatorColumn = false;
				}
			}
		} else {
			final OrmColumn dc = emd.getDiscriminatorColumn();
			if (dc != null) {
				mm.executeModelChanges(new Runnable() {
					public void run() {
						dc.delete();
						emd.setDiscriminatorColumn(null);
					}
				});
			}
		}
	}
	
	/**
	 * Get the base {@link OrmColumn#getRelativePositionInTable()} value for
	 * columns belonging to this entity. This is based on the position of 
	 * the entity in its heirachy and relative to its siblings.
	 */
	public int getRelativePosition() {
		if (emd == null) {
			return 0;
		}
		EntityMetaData superEmd = emd.getSuperEntity();
		EntityIO sio = EntityIO.get(superEmd);
		if (sio == null) {
			return 0;
		}
		return 1000000 + sio.getRelativePosition() + 
			100000 * superEmd.getSubEntityList().indexOf(emd);
	}	
	
	public void getPossibleEntityMappings(List ans) {
		ans.add(MAPPING_ENTITY);
		ans.add(MAPPING_EMBEDDABLE);
		ans.add(MAPPING_EMBEDDABLE_SUPERCLASS);
		ans.add(MAPPING_NOT_PERSISTENT);
	}

	public IIntOption getEntityMapping() {
		switch (emd.getEntityType()) {
		case EntityMetaData.TYPE_ENTITY:
			return MAPPING_ENTITY;
		case EntityMetaData.TYPE_EMBEDDABLE:
			return MAPPING_EMBEDDABLE;
		case EntityMetaData.TYPE_EMBEDDABLE_SUPERCLASS:
			return MAPPING_EMBEDDABLE_SUPERCLASS;
		}
		return null;
	}

	public void setEntityMapping(IIntOption option) {
		int et;
		if (MAPPING_NOT_PERSISTENT.equals(option)) {
			et = 0;
		} else if (MAPPING_EMBEDDABLE.equals(option)) {
			et = EntityMetaData.TYPE_EMBEDDABLE;
		} else if (MAPPING_EMBEDDABLE_SUPERCLASS.equals(option)) {
			et = EntityMetaData.TYPE_EMBEDDABLE_SUPERCLASS;
		} else {
			et = EntityMetaData.TYPE_ENTITY;
		}
		if (et != emd.getEntityType()) {
			emd.setEntityType(et);
		}
	}

	public JoinIO getJoinIO() {
		return joinIO;
	}
	
	/**
	 * Base class for classes we use to track attributes to be processed
	 * when updating the model.
	 */
	protected abstract class AttributeInfo {

		public abstract boolean updateModelFromMetaData(RClass cls, 
				boolean metaDataChanged);
	}
	
	protected class AttributeInfoNormal extends AttributeInfo {
		
		public RAnnotatedElement attribute;
		public AttributeIO attributeIO;
		
		public AttributeInfoNormal(RAnnotatedElement attribute, 
				AttributeIO attributeIO) {
			this.attribute = attribute;
			this.attributeIO = attributeIO;
		}
		
		public boolean updateModelFromMetaData(RClass cls, 
				boolean metaDataChanged) {
			return attributeIO.updateModelFromMetaData(cls, 
					attribute, metaDataChanged);
		}
		
		public String toString() {
			return attribute + " " + attributeIO;
		}
	}
	
	protected class AttributeInfoOverride extends AttributeInfo  {
		
		public AttributeOverride override;
		public AttributeMetaData overrideOf;
		public OverrideAttributeIO attributeIO;
		
		public AttributeInfoOverride(AttributeMetaData overrideOf, 
				AttributeOverride override, OverrideAttributeIO attributeIO) {
			this.overrideOf = overrideOf;
			this.override = override;
			this.attributeIO = attributeIO;
		}
		
		public boolean updateModelFromMetaData(RClass cls, 
				boolean metaDataChanged) {
			return attributeIO.updateModelFromMetaData(
					cls, overrideOf, override, metaDataChanged, 0);
		}
		
		public String toString() {
			return "Override " + overrideOf.getName() + " " + attributeIO;
		}
	}
	
}
